from viavi.evm import *
from viavi.diag import *
from time import time, sleep
from viavi.mts.uts_bdt import UtsBdt
import csv
import traceback
import collections
from pathlib import Path
from .eotdrv2_regs import *

num_acquisition = 0



class Otdr_Acq_Exception(Exception):
    CONST_EXCEPTION_TIMEOUT_PLL = -1
    CONST_EXCEPTION_TIMEOUT_START_ACQ = -2
    CONST_EXCEPTION_TIMEOUT_ACQ = -3
    pass

    def __init__(self, except_no):
        self.except_no = except_no

    def __str__(self):
        if self.except_no == self.CONST_EXCEPTION_TIMEOUT_PLL:
            return "OTDR Exception: PLL Lock Timeout"
        elif self.except_no == self.CONST_EXCEPTION_TIMEOUT_START_ACQ:
            return "OTDR Exception: Timeout Start Acquisition"
        elif self.except_no == self.CONST_EXCEPTION_TIMEOUT_ACQ:
            return "OTDR Exception: Timeout Acquisition"
        else:
            return "OTDR Exception: Unknown"

class Otdr_Acq(object):
    ARRAY_VERIF_RAMAN = None

    def get_array_raman(self):
        if self.ARRAY_VERIF_RAMAN is None:
            self.ARRAY_VERIF_RAMAN =  [ \
                                    ['GAIN_1K',   16384, int(CONST("raman.gain_1k.dac16384.value",20730300)),  int(CONST("raman.gain_1k.dac16384.tolerance",5)),  float(CONST("raman.gain_1k.dac16384.noise",0.5))], \
                                    ['GAIN_1K',   32768, int(CONST("raman.gain_1k.dac32768.value",20724200)),  int(CONST("raman.gain_1k.dac32768.tolerance",5)),  float(CONST("raman.gain_1k.dac32768.noise",0.5))], \
                                    ['GAIN_1K',   49152, int(CONST("raman.gain_1k.dac49152.value",20719000)),  int(CONST("raman.gain_1k.dac49152.tolerance",5)),  float(CONST("raman.gain_1k.dac49152.noise",0.5))], \
                                    ['GAIN_4K64', 16384, int(CONST("raman.gain_4k.dac16384.value",20711900)),  int(CONST("raman.gain_4k.dac16384.tolerance",5)),  float(CONST("raman.gain_4k.dac16384.noise",1))], \
                                    ['GAIN_4K64', 32768, int(CONST("raman.gain_4k.dac32768.value",20684500)),  int(CONST("raman.gain_4k.dac32768.tolerance",5)),  float(CONST("raman.gain_4k.dac32768.noise",1))], \
                                    ['GAIN_4K64', 49152, int(CONST("raman.gain_4k.dac49152.value",20657000)),  int(CONST("raman.gain_4k.dac49152.tolerance",5)),  float(CONST("raman.gain_4k.dac49152.noise",1))], \
                                    ['GAIN_17K8', 16384, int(CONST("raman.gain_17k.dac16384.value",20635500)), int(CONST("raman.gain_17k.dac16384.tolerance",10)), float(CONST("raman.gain_17k.dac16384.noise",5))], \
                                    ['GAIN_17K8', 32768, int(CONST("raman.gain_17k.dac32768.value",20531000)), int(CONST("raman.gain_17k.dac32768.tolerance",10)), float(CONST("raman.gain_17k.dac32768.noise",5))], \
                                    ['GAIN_17K8', 49152, int(CONST("raman.gain_17k.dac49152.value",20427300)), int(CONST("raman.gain_17k.dac49152.tolerance",10)), float(CONST("raman.gain_17k.dac49152.noise",5))], \
                                    ['GAIN_68K',  16384, int(CONST("raman.gain_68k.dac16384.value",20355600)), int(CONST("raman.gain_68k.dac16384.tolerance",10)), float(CONST("raman.gain_68k.dac16384.noise",10))], \
                                    ['GAIN_68K',  32768, int(CONST("raman.gain_68k.dac32768.value",19974600)), int(CONST("raman.gain_68k.dac32768.tolerance",10)), float(CONST("raman.gain_68k.dac32768.noise",10))], \
                                    ['GAIN_68K',  49152, int(CONST("raman.gain_68k.dac49152.value",19593900)), int(CONST("raman.gain_68k.dac49152.tolerance",10)), float(CONST("raman.gain_68k.dac49152.noise",10))], \
                                    ['GAIN_1M',       0, int(CONST("raman.gain_1m.dac0.value",     20720600)), int(CONST("raman.gain_1m.dac0.tolerance",10)),  float(CONST("raman.gain_1m.dac0.noise",50))], \
                                    ['GAIN_1M',   16384, int(CONST("raman.gain_1m.dac16384.value",14783000)),  int(CONST("raman.gain_1m.dac16384.tolerance",10)),  float(CONST("raman.gain_1m.dac16384.noise",50))]]
        return self.ARRAY_VERIF_RAMAN

    def __init__(self):
        self.utsbdt = UtsBdt("/dev/bdt_pws1")
        self.CONST_GAINS = collections.OrderedDict()
        self.CONST_GAINS.update({'GAIN_1K': 0x0})
        self.CONST_GAINS.update({'GAIN_4K64': 0x1})
        self.CONST_GAINS.update({'GAIN_17K8': 0x2})
        self.CONST_GAINS.update({'GAIN_68K': 0x3})
        self.CONST_GAINS.update({'GAIN_1M': 0x4})
        # set ADC ouput mode in Offset binary instead of 2 complement
        sysfs_write("/sys/platform/adcbdt/address", 0x14)
        sysfs_write("/sys/platform/adcbdt/value",   0x00)   
        sysfs_write("/sys/platform/adcbdt/address", 0xFF)
        sysfs_write("/sys/platform/adcbdt/value", 0x1)

    def StartAcq(self,laser_enable):
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC,  0xFFFFFFFF, 0)
        timeout = time() + 1
        while ( ( self.utsbdt.rw_register(UtsBdt.READ, REG_MISC,  0, 0)  & REG_MISC_PLL_LOCKED ) != REG_MISC_PLL_LOCKED ):
            if (time() > timeout):
                raise Otdr_Acq_Exception(Otdr_Acq_Exception.CONST_EXCEPTION_TIMEOUT_PLL)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_BDT_ON | REG_MISC_CLEAR_IRQ,     REG_MISC_BDT_ON | REG_MISC_CLEAR_IRQ)
        if laser_enable:
            self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_LASER_ON| REG_MISC_CLEAR_IRQ,  REG_MISC_LASER_ON)
        else:
            self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_LASER_ON| REG_MISC_CLEAR_IRQ,  0)
            
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_START_MEAS | REG_MISC_ENABLE_IRQ, REG_MISC_START_MEAS | REG_MISC_ENABLE_IRQ)

        timeout = time() + 2
        while ( (not self.IsMeasInProgress()) and not self.IsIrqEndMeasureRised()):
            sleep(0.2)
            if (time() > timeout):
                raise Otdr_Acq_Exception(Otdr_Acq_Exception.CONST_EXCEPTION_TIMEOUT_START_ACQ)


    def IsMeasInProgress(self):
        return (self.utsbdt.rw_register(UtsBdt.READ, REG_MISC,  0, 0) & REG_MISC_MEAS_IN_PROGRESS ) == REG_MISC_MEAS_IN_PROGRESS

    def IsIrqEndMeasureRised(self):
        return ( self.utsbdt.rw_register(UtsBdt.READ, REG_MISC,  0, 0) & REG_MISC_IRQ_BDT ) == REG_MISC_IRQ_BDT

    def IsLaserInProgress(self):
        return ( self.utsbdt.rw_register(UtsBdt.READ, REG_MISC,  0, 0)  & REG_MISC_LASER_IN_PROGRESS ) == REG_MISC_LASER_IN_PROGRESS

    def StartAndPoolAcq(self,laser_enable):
        self.StartAcq(laser_enable)    
    # WAIT END MEASURE
        timeout = time() + 32   #permet 100kpts @3�s, 20kpts@10�s, 10kpts@20�s
        while ( self.IsMeasInProgress()  ):
            if (time() > timeout):
                raise Otdr_Acq_Exception(Otdr_Acq_Exception.CONST_EXCEPTION_TIMEOUT_ACQ)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_CLEAR_IRQ, REG_MISC_CLEAR_IRQ)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_CLEAR_IRQ, 0)
        return 0
        

    def StartAndPoolAcqAndMeasI(self,laser_enable):
        ilas=[]
        self.StartAcq(laser_enable)    
        # WAIT END MEASURE
        timeout = time() + 62
        while ( self.IsMeasInProgress() ):
            if (time() > timeout):
                raise Otdr_Acq_Exception(Otdr_Acq_Exception.CONST_EXCEPTION_TIMEOUT_ACQ)
            if ( self.IsLaserInProgress() ):
                ilas.append(XADC('adc_meas_ilas'))
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_CLEAR_IRQ, REG_MISC_CLEAR_IRQ)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, REG_MISC_CLEAR_IRQ, 0)
        # Compute Ilas
        moy=0.0
        for el in ilas:
            moy+=el
        moy/=len(ilas)
        #~ dumpListinFile(ilas)
        return(moy)

    def ConfigAcqBIST(self,teeth,avg,pulse,laser):
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_TEETH_NUMBER,     0xFFFFFFFF, teeth)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_XOFFSET_DURATION, 0xFFFFFFFF, 300)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MASQ_DURATION, 0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_WAIT_DURATION,    0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_NOISE_DURATION,   0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_AVERAGE_NUMBER,   0xFFFFFFFF, avg)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_RAMP_INCR,        0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_RAMP_RESET_VAL,   0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_COMB_NUMBER,      0xFFFFFFFF, 1)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_COMB_INCR,        0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_CRUNCH_NUMBER,    0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_LASER_WIDTH_ENT,  0xFFFFFFFF, pulse)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_LASER_WIDTH_DEC,  0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_LASER_ENABLE,     0xFFFFFFFF, laser)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_OFFSET_CONTINU,   0xFFFFFFFF, 0)
        recurrence=max(teeth+310,pulse*100) # pour respecter le 1% sur le laser
        #~ print("recurrence=%d"%recurrence)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_PRIME_RECUR,     0xFFFFFFFF, recurrence)
        return 0

    def StartModulationSource(self, freq):
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC,  0xFFFFFFFF, 0)
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_LASER_FREQ,     0xFFFFFFFF, freq)
        return 0


    def DumpAcq(self, sample):
        handle = open("/dev/bdt_pws1", "rb")
        for i in range(0,sample):
            data = handle.read(4)
            datab = bytearray(data)
            val = datab[0] + datab[1]*0x100 + datab[2]*0x10000 + datab[3]*0x1000000
            print("%d:%x"%(i,val))
        return 0

    def AvgAcq(self, sample):
        self.avg = 0
        self.min_acq = 0xFFFFFFFF
        self.max_acq = 0
        handle = open("/dev/bdt_pws1", "rb")
        sleep(1)
        for i in range(0,sample+1):
            data = handle.read(4)
            if i > 0: #skip first point
                datab = bytearray(data)
                val = datab[0] + datab[1]*0x100 + datab[2]*0x10000 + datab[3]*0x1000000
                if  val < self.min_acq:
                    self.min_acq = val
                if  val > self.max_acq:
                    self.max_acq = val
                self.avg = self.avg + val
        self.avg = self.avg / sample
        return self.avg


    def VerifBufferAcq(self, sample, average, expvalue):
        err = 0
        handle = open("/dev/bdt_pws1", "rb")
        exp = average * expvalue
        for i in range(0,sample):
            data = handle.read(4)
            if i > 0:   #skip first point
                datab = bytearray(data)
                val = datab[0] + datab[1]*0x100 + datab[2]*0x10000 + datab[3]*0x1000000
                if val != exp:
                    print_error("# unexpected value in acquisition buffer index=%d , value = %d, exp = %d" %(i, val, expvalue))
                    err = err + 1
        return err

    def DumpAcq(self, sample):
        err = 0
        handle = open("/dev/bdt_pws1", "rb")
        for i in range(0,sample):
            data = handle.read(4)
            if i > 0:   #skip first point
                datab = bytearray(data)
                val = datab[0] + datab[1]*0x100 + datab[2]*0x10000 + datab[3]*0x1000000
                print_info("%d: 0x%08x"%(i,val))
        return

    def getBufferNoise(self, sample):
        handle = open("/dev/bdt_pws1", "rb")
        noise = 0
        for i in range(0,sample):
            data = handle.read(4)
            if i > 0:   #skip first point
                datab = bytearray(data)
                val = datab[0] + datab[1]*0x100 + datab[2]*0x10000 + datab[3]*0x1000000
                if val > noise:
                    noise = val
        return noise

    def setConsigneRaman(self,consigne):
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_DAC_CONSIGNE_RAMAN, 0x000FFFF, consigne)
        sleep(0.1)
        return 0

    def setGain(self, gain):
        err = -1
        self.utsbdt.rw_register(UtsBdt.WRITE, REG_HARDWARE_CFG, 0x0000007, gain)
        sleep(0.1)
        return 0

    def enableAPD(self, enable):
        if enable:
            self.utsbdt.rw_register(UtsBdt.WRITE, REG_HARDWARE_CFG, 0x0000018, 0x10)
        else:
            self.utsbdt.rw_register(UtsBdt.WRITE, REG_HARDWARE_CFG, 0x0000018, 0)
        sleep(0.5)
        return 0

    def loopOnGainAndRaman(self):
        first_acq_skipped = False
        csvfile = open('dump_gain_raman.csv','w')
        csvwriter = csv.writer(csvfile, dialect='excel')
        csvwriter.writerow(['Gain APD','Consigne Raman','Acquisition Average','Acquisition Min','Acquisition Max'])
        self.ConfigAcqBIST(2000,10000,0,0)
        for gain in self.CONST_GAINS:
            self.setGain(self.CONST_GAINS[gain])
            consigne = 0
            print("# Gain = %s"%(gain))
            while consigne < 65536:
                self.setConsigneRaman(consigne)
                if not first_acq_skipped:
                    self.StartAndPoolAcq(False)
                    first_acq_skipped = True
                self.StartAndPoolAcq(False)
                self.DumpAcq(10)
                self.AvgAcq(1000)
                csvwriter.writerow([gain,consigne,self.avg,self.min_acq,self.max_acq])
                consigne = consigne + 4096
        csvfile.close()
        print_ok("# Log Gain/raman finished; file =  dump_gain_raman.csv")
        return 0


    def verifOnGainAndRaman(self):
        err = 0
        first_acq_skipped = False
        self.ConfigAcqBIST(2000,10000,0,0)
        for consigne in self.get_array_raman():
            self.setGain(self.CONST_GAINS[consigne[0]])
            self.setConsigneRaman(consigne[1])
            print("# Gain:%s, Dac Raman=%d" % (consigne[0],consigne[1] ) ,end=' : ',flush=True)
            if not first_acq_skipped:
                self.StartAndPoolAcq(False)
                first_acq_skipped = True
            self.StartAndPoolAcq(False)
            self.AvgAcq(1000)
            ecart=( abs(self.avg - consigne[2]) / self.avg ) * 100
            noise = (( self.max_acq - self.min_acq) / self.avg ) * 100
            if ecart >  consigne[3]:
                print('%+.3f%% Noise=%.2f'%(ecart,noise))
                print_error("# Gain:%s, Dac Raman=%d, avg value = %f  not in [ %f +/- %d%%]" % (consigne[0],consigne[1],self.avg, consigne[2], consigne[3] ) )
                err = err + 1
            else:
                print_ok('%+.3f%% Noise=%.2f'%(ecart,noise))
            if noise >  consigne[4]:
                print_error("# Gain:%s, Noise=%f %%  > %f %%" % (consigne[0], noise , consigne[4] ) )
                err = err + 1
        return err
        
    def dump_regs(self):
        print("REG_TEETH_NUMBER       %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_TEETH_NUMBER, 0, 0) )
        print("REG_XOFFSET_DURATION   %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_XOFFSET_DURATION, 0, 0) )
        print("REG_MASQ_DURATION      %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_MASQ_DURATION, 0, 0) )
        print("REG_WAIT_DURATION      %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_WAIT_DURATION, 0, 0) )
        print("REG_NOISE_DURATION     %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_NOISE_DURATION, 0, 0) )
        print("REG_AVERAGE_NUMBER     %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_AVERAGE_NUMBER, 0, 0) )
        print("REG_RAMP_INCR          %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_RAMP_INCR, 0, 0) )
        print("REG_RAMP_RESET_VAL     %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_RAMP_RESET_VAL, 0, 0) )
        print("REG_COMB_NUMBER        %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_COMB_NUMBER, 0, 0) )
        print("REG_COMB_INCR          %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_COMB_INCR, 0, 0) )
        print("REG_CRUNCH_NUMBER      %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_CRUNCH_NUMBER, 0, 0) )
        print("REG_LPDDR_ADDR_OFFSET  %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_LPDDR_ADDR_OFFSET, 0, 0) )
        print("REG_LASER_WIDTH_ENT    %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_LASER_WIDTH_ENT, 0, 0) )
        print("REG_LASER_WIDTH_DEC    %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_LASER_WIDTH_DEC, 0, 0) )
        print("REG_LASER_ENABLE       %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_LASER_ENABLE, 0, 0) )
        print("REG_OFFSET_CONTINU     %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_OFFSET_CONTINU, 0, 0) )
        print("REG_MISC               %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_MISC, 0, 0) )
        print("REG_HARDWARE_CFG       %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_HARDWARE_CFG, 0, 0) )
        print("REG_TOTAL_TEETH_NB_CAL %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_TOTAL_TEETH_NB_CAL, 0, 0) )
        print("REG_FORCE_COMB         %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_FORCE_COMB, 0, 0) )
        print("REG_FSM                %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_FSM, 0, 0) )
        print("REG_COUNT_SYNC         %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_COUNT_SYNC, 0, 0) )
        print("REG_PRIME_RECUR        %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PRIME_RECUR, 0, 0) )
        print("REG_DATA_LOG           %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_DATA_LOG, 0, 0) )
        print("REG_DAC_VCC_APD        %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_DAC_VCC_APD, 0, 0) )
        print("REG_DAC_CONSIGNE_RAMAN %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_DAC_CONSIGNE_RAMAN, 0, 0) )
        print("REG_DAC_CONSIGNE_LASER %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_DAC_CONSIGNE_LASER, 0, 0) )
        print("REG_DAC_CONSIGNE_TEC   %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_DAC_CONSIGNE_TEC, 0, 0) )
        print("REG_DAC_CONSIGNE_TEC_A %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_DAC_CONSIGNE_TEC_AUTO, 0, 0) )
        print("REG_PID_Kp             %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_Kp, 0, 0) )
        print("REG_PID_Ki             %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_Ki, 0, 0) )
        print("REG_PID_Kd             %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_Kd, 0, 0) )
        print("REG_PID_P_DEBUG        %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_P_DEBUG, 0, 0) )
        print("REG_PID_I_DEBUG        %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_I_DEBUG, 0, 0) )
        print("REG_PID_D_DEBUG        %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_D_DEBUG, 0, 0) )
        print("REG_PID_LOG            %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_LOG, 0, 0) )
        print("REG_PID_STATUS         %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_PID_STATUS, 0, 0) )
        print("REG_MEASI_AVG_RAW      %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_MEASI_AVG_RAW, 0, 0) )
        print("REG_MEASI_AVG_DUTY     %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_MEASI_AVG_DUTY, 0, 0) )
        print("REG_LAS_SRC_EN         %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_LASER_SOURCE_ENABLE, 0, 0) )
        print("REG_LAS_SRC_FREQ_MOD   %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_LASER_SOURCE_FREQ_MODULATION, 0, 0) )
        print("REG_LAS_FREQ_FAST_MOD  %08x" % self.utsbdt.rw_register(UtsBdt.READ, REG_LASER_FREQ_FAST_MODULATION, 0, 0) )
        
    def write_regs(self,adr,mask,val):
        self.utsbdt.rw_register(UtsBdt.WRITE, adr, mask, val)

def XADC(channel=None):
    gain={  #gain hardware entre valeurs mesurees et entrees ADC
                "adc_vin":1/(1+4.64+10),   #R2+R4+R5=1K+4.64K+10K
                "adc_vcc_apd":0.99/100,    #0.99%
                "adc_temp_ana":9e-3,    #9mV/deg + 454mV-318mV
                "adc_det_shield":1,
                "adc_vtec":0.1,
                "adc_meas_v_ana":1,
                "adc_meas_vee_laser":9.54978/100,  #9.54978%
                "adc_meas_ilas":0.9561717/2,  #0.9561717V/A
                "adc_itec":0.21}

    offset={#offset en V
                "adc_vin":0,
                "adc_vcc_apd":0, 
                "adc_temp_ana":0.5*100/110-5.0*10/157,    
                "adc_det_shield":0,
                "adc_vtec":0,
                "adc_meas_v_ana":0,
                "adc_meas_vee_laser":0,
                "adc_meas_ilas":0,
                "adc_itec":0}

    if channel in gain.keys():
        #~ link_val="/sys/platform/"+channel+"/value"
        #~ value=float(open(link_val,'r').read())
        chaine=open("/sys/platform/"+channel+"/raw",'r').read()
        raw=str2int(chaine)
        value=(raw/1000.0-offset[channel])/gain[channel]
        return(value)
        
    else:   # On affiche toutes les voies
        for name in gain.keys():
            chaine=open("/sys/platform/"+name+"/raw",'r').read()
            raw=str2int(chaine)
            value=float(open("/sys/platform/"+name+"/value",'r').read())
            print("%20s = %11s, %5.2f, %5.2f"%(name,chaine.strip(),(raw/1000.0-offset[name])/gain[name],value)) #raw/1000 car le driver retourne en mV

def str2int(chaine):
    if int(chaine,base=0)>0x0FFFFFFF : #bit de poids fort=1 donc nb negatif
        raw=int(chaine,base=0)-0x100000000
    else: # nb positif
        raw=int(chaine,base=0) #la valeur raw est en hexa
    return(raw)
    


@DIAG("Test OTDR acquisition",depends=["analog_powersupply"])
def t_acq_otdr(cmd):

    try:
        (command,log_gain_raman) = cmd.split()
    except:
        log_gain_raman = "NO"

    # VERIF COMM AD9634
    sysfs_write("/sys/platform/adcbdt/address", 1)
    id_ad = sysfs_read("/sys/platform/adcbdt/value")
    if (id_ad!=0x87) and (id_ad!=0x86):
        print_error("# AD96xx Chip ID = 0x%x (expect 0x86 or 0x87)"%(id_ad))
        return False
    else:
        print_info("# AD96xx Chip ID OK (0x%x)"%(id_ad))
    
    # VERIF ACQ WITH KNOW PATTERN
    sysfs_write("/sys/platform/adcbdt/address", 0x0d)
    sysfs_write("/sys/platform/adcbdt/value",   0x08)

    sysfs_write("/sys/platform/adcbdt/address", 0x19)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0x1A)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0x1B)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0x1C)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0x1D)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0x1E)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0x1F)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0x20)
    sysfs_write("/sys/platform/adcbdt/value",   0xAA)
    sysfs_write("/sys/platform/adcbdt/address", 0xFF)
    sysfs_write("/sys/platform/adcbdt/value",   0x01)

    otdr = Otdr_Acq()
    otdr.ConfigAcqBIST(2000,10,0,0)
    otdr.StartAndPoolAcq(True)
    nberr= otdr.VerifBufferAcq(1000, 10, 0xAAA)
    if nberr == 0:
        print_ok("# Acquisition: AD9634 Pattern check: OK")
    else:
        return False

    # reset AD9634 to real mode

    sysfs_write("/sys/platform/adcbdt/address", 0x0d)
    sysfs_write("/sys/platform/adcbdt/value",   0x00)
    sysfs_write("/sys/platform/adcbdt/address", 0xFF)
    sysfs_write("/sys/platform/adcbdt/value",   0x01)

    if log_gain_raman == 'YES':
        print_info("LOG GAIN RAMAN!!!!")
        otdr.loopOnGainAndRaman()

    nberr = otdr.verifOnGainAndRaman()
    if nberr == 0:
        print_ok("# Noise + Avg value for Raman offset: OK")
    else:
        return False

    return True


@DIAG("Test Pulse Laser",depends=["analog_powersupply"])
def t_laser(cmd):

    try:
        (command,laser_num) = cmd.split()
    except:
        print_error("# Command must be: t_laser <laser> where <laser> is L1|L2|L3")
        return False

    # set laser consigne
    vee_laser_simu_laser_value = float(CONST("vee_laser_simu_laser.value",1.0))
    consigne=int(vee_laser_simu_laser_value/10*65535)
    setStandByAmpli(0)
    print_info("# Laser courant measurment") 
    otdr = Otdr_Acq()
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_DAC_CONSIGNE_LASER, 0xFFFFFFFF, consigne)
    # let time to power supply to settle
    sleep(0.6)
    otdr.ConfigAcqBIST(80000,2000,480,  1 <<  (int(laser_num[1]) - 1) )
    otdr.StartAcq(True)
    i=0
    mes_crete = 0

    laser_courant_min = float(CONST("simu_laser.courant.min",0.180))
    laser_courant_max = float(CONST("simu_laser.courant.max",0.220))
    # theoricaly: 1V /4.2Ohm = 0.238A  --> but tool contact resistance is not nul ~1 Ohm
    while  ( otdr.IsMeasInProgress() ):
        mes_crete = mes_crete + sysfs_read("/sys/platform/adc_meas_ilas/value")
        i = i +1

    mes_crete = (mes_crete / i )
    print_info("# Laser %s : Average courant= %5.3f A"%(laser_num,mes_crete))
    setStandByAmpli(1)
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_DAC_CONSIGNE_LASER, 0xFFFFFFFF, 0)
    if(  ( mes_crete < laser_courant_min) or (mes_crete > laser_courant_max ) ):
        print_error("# Laser %s : Average courant= %5.3f A not in spec [%5.3f A, %5.3f A]" %(laser_num,mes_crete,laser_courant_min,laser_courant_max))
        return False 
        
    return True 








def setStandByAmpli(state):
    with open("/sys/platform/powersupply/standby_ampli",'w') as f:    #StandbyAmpli
        f.write(str(state))

@DIAG("Dump TimeBase Registers")
def dump_regs(cmd):
    otdr = Otdr_Acq()
    otdr.dump_regs()

@DIAG("Write TimeBase Registers")
def write_regs(cmd):
    (command,adr,mask,val) = cmd.split()
    otdr = Otdr_Acq()
    otdr.write_regs(int(adr,16),int(mask,16),int(val,16))


def dumpAcqFile(sample, filename="acq.csv"):
    handle = open("/dev/bdt_pws1", "rb")
    directory=r"/var/lib/ftp/"
    if filename[-4:]=='.csv':
        fileName=filename
    else:
        fileName=filename+'.csv'
    if Path(directory+fileName).is_file():
        fileName0=fileName[:-4]
        for i in range(100):
            fileName=fileName0+'_'+str(i)+'.csv'
            if not Path(directory+fileName).is_file():
                break
    with open(directory+fileName,'w') as f:
        for i in range(0,sample):
            data = handle.read(4)
            if i > 0:   #skip first point
                datab = bytearray(data)
                val = datab[0] + datab[1]*0x100 + datab[2]*0x10000 + datab[3]*0x1000000
                f.write(str(val))
                f.write('\n')
    print("'%s' written in %s"%(fileName,directory))

@DIAG("Debug Acq 40M",depends=["analog_powersupply"])
def debug40(cmd):
    sysfs_write("/sys/platform/adcbdt/address", 1)
    check = sysfs_check("/sys/platform/adcbdt/value","AD9634 Chip ID", hexval=0x87)
    if check==False:
        return False

        # VERIF ACQ WITH KNOW PATTERN
    sysfs_write("/sys/platform/adcbdt/address", 0x0d)
    sysfs_write("/sys/platform/adcbdt/value",   0x08)

    sysfs_write("/sys/platform/adcbdt/address", 0x19)
    sysfs_write("/sys/platform/adcbdt/value",   0x11)
    sysfs_write("/sys/platform/adcbdt/address", 0x1A)
    sysfs_write("/sys/platform/adcbdt/value",   0x00)
    sysfs_write("/sys/platform/adcbdt/address", 0x1B)
    sysfs_write("/sys/platform/adcbdt/value",   0x11)
    sysfs_write("/sys/platform/adcbdt/address", 0x1C)
    sysfs_write("/sys/platform/adcbdt/value",   0x00)
    sysfs_write("/sys/platform/adcbdt/address", 0x1D)
    sysfs_write("/sys/platform/adcbdt/value",   0x11)
    sysfs_write("/sys/platform/adcbdt/address", 0x1E)
    sysfs_write("/sys/platform/adcbdt/value",   0x00)
    sysfs_write("/sys/platform/adcbdt/address", 0x1F)
    sysfs_write("/sys/platform/adcbdt/value",   0x11)
    sysfs_write("/sys/platform/adcbdt/address", 0x20)
    sysfs_write("/sys/platform/adcbdt/value",   0x00)

    sysfs_write("/sys/platform/adcbdt/address", 0x0B)
    sysfs_write("/sys/platform/adcbdt/value",   0x03)

    sysfs_write("/sys/platform/adcbdt/address", 0xFF)
    sysfs_write("/sys/platform/adcbdt/value",   0x01)


    print_info("# Essai a 40MHz")

    otdr = Otdr_Acq()
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, 0xE0000000, 0xA0000000 )
    otdr.ConfigAcqBIST(2000,10000,0,0)
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_CRUNCH_NUMBER,    0xFFFFFFFF, 1)
    otdr.StartAndPoolAcq(True)
    otdr.DumpAcq(10);

    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, 0xE0000000, 0x00000000 )

    print_info("# Essai a 80MHz")
    sysfs_write("/sys/platform/adcbdt/address", 0x0B)
    sysfs_write("/sys/platform/adcbdt/value",   0x01)

    sysfs_write("/sys/platform/adcbdt/address", 0xFF)
    sysfs_write("/sys/platform/adcbdt/value",   0x01)

    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, 0xE0000000, 0x60000000 )
    otdr.ConfigAcqBIST(2000,10000,0,0)
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_CRUNCH_NUMBER,    0xFFFFFFFF, 1)
    otdr.StartAndPoolAcq(True)
    otdr.DumpAcq(10);
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, 0xE0000000, 0x00000000 )

    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, 0xE0000000, 0x00000000 )

    print_info("# Essai a 160MHz")
    sysfs_write("/sys/platform/adcbdt/address", 0x0B)
    sysfs_write("/sys/platform/adcbdt/value",   0x00)

    sysfs_write("/sys/platform/adcbdt/address", 0xFF)
    sysfs_write("/sys/platform/adcbdt/value",   0x01)

    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, 0xE0000000, 0x20000000 )
    otdr.ConfigAcqBIST(2000,10000,0,0)
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_CRUNCH_NUMBER,    0xFFFFFFFF, 1)
    otdr.StartAndPoolAcq(True)
    otdr.DumpAcq(10);
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_MISC, 0xE0000000, 0x00000000 )

    sysfs_write("/sys/platform/adcbdt/address", 0x0d)
    sysfs_write("/sys/platform/adcbdt/value",   0x00)
    sysfs_write("/sys/platform/adcbdt/address", 0xFF)
    sysfs_write("/sys/platform/adcbdt/value",   0x01)

@DIAG("Test Peltier",depends=["analog_powersupply"])
def t_peltier(cmd):
    try:
        (command,sens,mode) = cmd.split()
    except:
        print_error("# Command must be: t_peltier <AUTO|HEAT|COOL|OFF> <LOW|HIGH>")
        return False
    otdr = Otdr_Acq()
    # COEFF DIGITAL PID (idem 1610 step 2 --> FAST Response)
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_PID_Kp, 0x00FFFFFF, 0x0A0200 )   # Kp = 512 / 2^10
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_PID_Ki, 0x00FFFFFF, 0x0A0005 )   # Ki = 5 / 2^10
    otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_PID_Kd, 0x00FFFFFF, 0x0A0800 )   # Kd = 2048 / 2^10
    if (sens=="AUTO"):
        print_info("HEATING LOW POWER")
        otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_DAC_CONSIGNE_TEC, 0x0001FFFF, 0x0FFFF )
        sysfs_write("/sys/platform/powersupply/low_i_tec", 1)
        sysfs_write("/sys/platform/powersupply/en_tec", 1)
        sleep(2)
        res = sysfs_check("/sys/platform/adc_itec/value", "ITec Heat Low", vmin=float(CONST("itec.heat.low.min",-0.50)), vmax=float(CONST("itec.heat.low.max",-0.40)))
        res = sysfs_check("/sys/platform/adc_vtec/value", "VTec Heat Low", vmin=float(CONST("vtec.heat.low.min",-1.1)), vmax=float(CONST("vtec.heat.low.max",-0.9))) and res
        print_info("HEATING HIGH POWER")
        sysfs_write("/sys/platform/powersupply/low_i_tec", 0)
        sleep(1)
        res = sysfs_check("/sys/platform/adc_itec/value", "ITec Heat High", vmin=float(CONST("itec.heat.high.min",-1.4)), vmax=float(CONST("itec.heat.high.max",-1.2))) and res
        res = sysfs_check("/sys/platform/adc_vtec/value", "VTec Heat High", vmin=float(CONST("vtec.heat.high.min",-2.9)), vmax=float(CONST("vtec.heat.high.max",-2.7))) and res
        print_info("COOLING LOW POWER")
        sysfs_write("/sys/platform/powersupply/low_i_tec", 1)
        otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_DAC_CONSIGNE_TEC, 0x0001FFFF, 0x00000 )
        sleep(3)
        res = sysfs_check("/sys/platform/adc_itec/value", "ITec Cool Low", vmin=float(CONST("itec.cool.low.min",0.9)), vmax=float(CONST("itec.cool.low.max",1.1))) and res
        res = sysfs_check("/sys/platform/adc_vtec/value", "VTec Cool Low", vmin=float(CONST("vtec.cool.low.min",1.9)), vmax=float(CONST("vtec.cool.low.max",2.1))) and res
        print_info("COOLING HIGH POWER")
        sysfs_write("/sys/platform/powersupply/low_i_tec", 0)
        sleep(1)
        res = sysfs_check("/sys/platform/adc_itec/value", "ITec Cool High", vmin=float(CONST("itec.cool.high.min",1.1)), vmax=float(CONST("itec.cool.high.max",1.3))) and res
        res = sysfs_check("/sys/platform/adc_vtec/value", "VTec Cool High", vmin=float(CONST("vtec.cool.high.min",2.4)), vmax=float(CONST("vtec.cool.high.max",2.6))) and res
        print_info("STOP PELTIER")
        sysfs_write("/sys/platform/powersupply/en_tec", 0)
        return res
    else:
        if (mode=="LOW"):
            sysfs_write("/sys/platform/powersupply/low_i_tec", 1)
        else:
            sysfs_write("/sys/platform/powersupply/low_i_tec", 0)
        if (sens=="HEAT"):
            otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_DAC_CONSIGNE_TEC, 0x0001FFFF, 0x0FFFF )
        else:
            otdr.utsbdt.rw_register(UtsBdt.WRITE, REG_DAC_CONSIGNE_TEC, 0x0001FFFF, 0x00000 )
        if (sens=="OFF"):
            sysfs_write("/sys/platform/powersupply/en_tec", 0)
        else:
            sysfs_write("/sys/platform/powersupply/en_tec", 1)
    return None
