import QtQuick 2.1

import common 1.0
import graph_plugin 1.0

FocusScope { id: root

    property alias view: worldView.view
    default property alias plots: plotContainer.data
    property bool worldViewFocus: false

    property alias zoomingChecked: zoomToggle.checked
    property alias zoomToggleFocus: buttonFocusRect.visible

    signal worldViewPressed

    Rectangle {
        id: worldBackground
        anchors {
            leftMargin: 5; left: parent.left
            rightMargin: 10; right: zoomToggle.left
            topMargin: 5; top: parent.top
            bottomMargin: 5; bottom: parent.bottom
        }

        focus: false
        border.width: 1
        border.color: "#484848"
        gradient: Gradient {
            GradientStop { position: 0.0; color: "#D0D1D2" }
            GradientStop { position: 0.2; color: "#EAEBEC" }
            GradientStop { position: 0.6; color: "#F1F2F3" }
            GradientStop { position: 1.0; color: "#F1F2F3" }
        }
    }

    WorldView {  id: worldView
        anchors.margins: 1
        anchors.fill: worldBackground
        focus: false

        Item { id: plotContainer
            anchors.fill: parent
        }

        Item { id: curtain
            anchors.fill: parent

            Rectangle { id: leftShade
                anchors {
                    left: parent.left
                    right: activeView.left
                    top: parent.top
                    bottom: parent.bottom
                }
                color: "#60000000"
            }

            Rectangle { id: rightShade
                anchors {
                    left: activeView.right
                    right: parent.right
                    top: parent.top
                    bottom: parent.bottom
                }
                color: "#60000000"
            }

            Rectangle { id: bottomShade
                anchors {
                    left: leftShade.right
                    right: rightShade.left
                    top: activeView.bottom
                    bottom: parent.bottom
                }
                color: "#60000000"
            }

            Rectangle { id: topShade
                anchors {
                    left: leftShade.right
                    right: rightShade.left
                    top: parent.top
                    bottom: activeView.top
                }
                color: "#60000000"
            }

            Rectangle { id: activeView
                width: curtain.width * worldView.zoomRect.width
                height: curtain.height * worldView.zoomRect.height
                color: "#0047B2"
                opacity: dragPan.pressed ? 0.5 : 0.0
                Behavior on opacity { NumberAnimation { duration: 250 } }

                MouseArea { id: dragPan
                    anchors.margins: -20
                    anchors.fill: parent
                    drag.target: parent
                    drag.axis: Drag.XAndYAxis //| Drag.XAxis
                    drag.minimumX: 0
                    drag.maximumX: curtain.width - activeView.width
                    drag.minimumY: 0
                    drag.maximumY: curtain.height - activeView.height
                }
                onXChanged: {
                    if (dragPan.pressed)
                        worldView.zoomRect.moveLeft(x / curtain.width);
                }
                onYChanged: {
                    if (dragPan.pressed)
                        worldView.zoomRect.moveBottom(1.0 - (y / curtain.height));
                }
            }

            Rectangle {
                anchors { margins: -1; fill: activeView }
                color: "#00000000"
                border.color: "#484848"
                border.width: 1
                visible: !focusRect.visible
            }

            Rectangle { id: focusRect
                anchors { margins: -4; fill: activeView }
                color: "#00000000"
                border.color: systemColor.FocusOnLightBg
                border.width: 4
                radius: 4
                visible: root.worldViewFocus && (worldView.zoomRect.width < 1.0 || worldView.zoomRect.height < 1.0)
            }

            Component.onCompleted: {
                worldView.zoomRect.leftChanged.connect(updateLeft);
                worldView.zoomRect.bottomChanged.connect(updateTop);
            }

            function updateLeft() {
                activeView.x = width * worldView.zoomRect.left;
            }

            function updateTop() {
                activeView.y = height * (1.0 - worldView.zoomRect.bottom);
            }
        }
    }

    MouseArea {
        anchors.fill: worldBackground
        focus: false
        onPressed: {
            worldViewPressed();
            mouse.accepted = false;
        }
    }

    Button { id: zoomToggle
        width: 110
        anchors {
            margins: 5
            right: parent.right
            top: parent.top
            bottom: parent.bottom
        }
        focus: true
        checkable: true
        iconSource: theme.dirPrefix + "magnifier-128.png"
        style: ButtonStyle {
            textColor: enabled ? systemFont.DynamicDarkFontColor : "gray"

            background: Rectangle {
                implicitWidth: control.width
                implicitHeight: control.height
                border.width:  1
                border.color: "#484848"
                gradient: control.checked ? pressedGradient : buttonGradient
                Gradient { id: buttonGradient
                    GradientStop { position: 0.0; color: "#ffffff" }
                    GradientStop { position: 1.0; color: "#acacac" }
                }
                Gradient { id: pressedGradient
                    GradientStop { position: 0.0; color: "#333333" }
                    GradientStop { position: 0.3; color: "#4d4878" }
                    GradientStop { position: 1.0; color: "#676298" }
                }
            }
            label: ColorKeyImage {
                anchors { fill: parent; margins: 8 }
                fillMode: Image.PreserveAspectFit
                source: control.iconSource
                color: control.checked ? "#ffffff" : "#333333"
            }
        }
    }

    Rectangle { id: buttonFocusRect
        anchors { margins: -3; fill: zoomToggle }
        focus: false
        color: "transparent"
        border.color: systemColor.FocusOnLightBg
        border.width: 4
        radius: 4
    }

    property int numIntervals: 10
    property real panXInterval: worldView.zoomRect.width / numIntervals
    property real panYInterval: worldView.zoomRect.height / numIntervals

    function upPressed () {
        worldView.zoomRect.moveBottom (worldView.zoomRect.bottom + panYInterval);
    }
    function downPressed () {
        worldView.zoomRect.moveBottom (worldView.zoomRect.bottom - panYInterval);
    }
}
