import QtQuick 2.5

import com.viavi.style 1.0

Rectangle {
    id: buttonRoot

    readonly property string leftState: 'left'
    readonly property string middleState: 'middle'
    readonly property string rightState: 'right'
    property bool hasMiddleState: true
    property bool fixedState: false
    state: leftState

    property alias mouseArea: mouseArea
    property int toggleSize: 150


    property bool isLeftStateActive: state === leftState
    property bool isRightStateActive: state === rightState

    states: [
        State {
            name: leftState
        },
        State {
            name: middleState
        },
        State {
            name: rightState
        }
    ]

    signal leftStateActive()
    signal middleStateActive()
    signal rightStateActive()

    onStateChanged: {
        if (state === leftState) {
            leftStateActive()
        }
        else if (state === middleState) {
            middleStateActive()
        }
        else {
            rightStateActive()
        }
    }

    //TODO - remove isRightDataVisible after complementary change to Fiber Optics is merged
    property bool isRightDataVisible: false
    onIsRightDataVisibleChanged: {
        if (isRightDataVisible) {
            fixedState = true
            state = leftState
        }
        else {
            fixedState = false
        }
    }

    width: toggleSize;
    height: width / 3

    radius: height / 2
    color: ViaviColor.activeColor
    border.color: ViaviColor.borderColor
    border.width: ViaviStyle.layout.borderSmallWidth

    Rectangle {
        id: draggableSwitch

        property double leftPos: horizontalMargins
        property double middlePos: parent.width/2 - width/2
        property double rightPos: parent.width - width - horizontalMargins
        property double horizontalMargins: parent.height * 0.2

        visible: true
        x: getTargetPos()
        anchors.verticalCenter : parent.verticalCenter
        color: ViaviColor.contentBackgroundColor
        height: parent.height * 0.80
        width: height
        z:1
        radius:height / 2


        function getTargetStateFromDragPos() {
            var distLeft = Math.abs(x - leftPos)
            var distMiddle = hasMiddleState ? Math.abs(x - middlePos) : Number.POSITIVE_INFINITY
            var distRight = Math.abs(x - rightPos)
            var closestPos = Math.min(distLeft, distMiddle, distRight)
            if (closestPos === distLeft) {
                return leftState
            }
            else if (closestPos === distMiddle) {
                return middleState
            }
            else {
                return rightState
            }
        }

        function getTargetPosFromState(state) {
            if (state === leftState) {
                return leftPos
            }
            else if (state === middleState) {
                return middlePos
            }
            else {
                return rightPos
            }
        }

        //Provide binding that is reevaluated even when state doesn't change, to avoid leaving the toggle a position where it is moved just a little bit.
        function getTargetPos() {
            if (!mouseArea.drag.active) {
                return getTargetPosFromState(buttonRoot.state)
            }
            else {
                return middlePos
            }
        }

        function chooseStateFromDragPos() {
            buttonRoot.state = getTargetStateFromDragPos()
        }
    }

    //A semi transparent switch, observable while dragging, to hint where the actuall switch will land when released.
    Rectangle {
        id: draggableSwitchGhost
        visible: mouseArea.drag.active
        opacity: 0.5
        x: draggableSwitch.getTargetPosFromState(draggableSwitch.getTargetStateFromDragPos())

        anchors.verticalCenter: parent.verticalCenter

        color: ViaviColor.contentBackgroundColor
        height: parent.height * 0.80
        width: height
        z:1
        radius: height / 2
    }


    function chooseNextState() {
        if (buttonRoot.state === leftState) {
            buttonRoot.state = hasMiddleState ? middleState : rightState
        }
        else if (buttonRoot.state === middleState) {
            buttonRoot.state = rightState
        }
        else {
            buttonRoot.state = leftState
        }
    }

    MouseArea {
        id: mouseArea
        anchors.fill: parent
        z:1
        visible: true
        enabled: !buttonRoot.fixedState
        onClicked: chooseNextState()
        drag.target: draggableSwitch
        drag.axis: Drag.XAndYAxis
        drag.minimumX: draggableSwitch.leftPos
        drag.maximumX: draggableSwitch.rightPos
        property bool dragActive: drag.active
        onDragActiveChanged: {
            if (!drag.active) {
                draggableSwitch.chooseStateFromDragPos()
            }
        }
    }
}
