#!/usr/bin/python3

import os
import ctypes
from ctypes.util import find_library
import fcntl
import struct
import json
import os.path
import subprocess
from functools import lru_cache
from shutil import copyfile, SameFileError

MS_RDONLY = 1
MS_NOSUID = 2
MS_NODEV = 4
MS_NOEXEC = 8
MS_SYNCHRONOUS = 16
MS_REMOUNT = 32
MS_MANDLOCK = 64
MS_DIRSYNC = 128
MS_NOATIME = 1024
MS_NODIRATIME = 2048
MS_BIND = 4096
MS_MOVE = 8192
MS_REC = 16384
MS_SILENT = 32768
MS_POSIXACL = 1 << 16
MS_UNBINDABLE = 1 << 17
MS_PRIVATE = 1 << 18
MS_SLAVE = 1 << 19
MS_SHARED = 1 << 20
MS_RELATIME = 1 << 21
MS_KERNMOUNT = 1 << 22
MS_I_VERSION = 1 << 23
MS_STRICTATIME = 1 << 24
MS_ACTIVE = 1 << 30
MS_NOUSER = 1 << 31


@lru_cache(maxsize=2)
def get_partition_plan():
    hardware_configuration_file = '/etc/hardware.conf'
    if not os.path.exists(hardware_configuration_file):
        print("Cannot found hardware configuration file, critical situation stopping operation.")
        exit(-1)
    with open(hardware_configuration_file) as json_file:
        data = json.load(json_file)
        if 'emmc' not in data:
            print("Cannot found partitioning information in hardware configuration file, stopping operation")
            exit(-1)
        number_of_partitioning_plan = len(data['emmc'])
        if number_of_partitioning_plan > 1:
            blk_device_list = get_block_devices("MMC")
            if len(blk_device_list) == 0:
                print("CRITICAL: no emmc found when looking for partition plan")
                exit(-1)
            if len(blk_device_list) > 1:
                print("Warning: many emmc found when looking for partition plan, using the first one!")
            partition_size = get_blockdevice_size([*blk_device_list][0])
        try:
            for p in data['emmc']:
                if number_of_partitioning_plan == 1 or (p['min_size'] <= partition_size < p['max_size']):
                    return p['partitioning_plan']
        except:
            print("Partition plan information are corrupted, stopping operation")
            exit(-1)
        print("Cannot found partitioning information, critical situation stopping operation.")
        exit(-1)


# Return the size of a block device in Mo
@lru_cache(maxsize=8)
def get_blockdevice_size(blkname):
    buff = ' '*4
    blkcount = 0
    blksize = 0
    with open('/dev/{}'.format(blkname), 'r') as fd:
        buff = fcntl.ioctl(fd.fileno(), 0x1260, buff)
        blkcount = struct.unpack('I', buff)[0]
        buff = fcntl.ioctl(fd.fileno(), 0x1268, buff)
        blksize = struct.unpack('I', buff)[0]
    return blkcount*blksize/(1024*1024)


@lru_cache(maxsize=4)
def get_block_devices(filtertype=None):
    blockdevice = {}
    for d in os.listdir("/sys/block"):
        if not os.path.exists("/sys/block/%s/device/type"%d):
            continue
        with open("/sys/block/%s/device/type"%d) as t:
            typ = t.read().strip()
            if filtertype is None or typ == filtertype:
                blockdevice[d] = typ
    return blockdevice


def mount_bind(source, target):
    """Call mount(2); see the man page for details."""
    libc = ctypes.CDLL(find_library('c'), use_errno=True)
    source = source.encode()
    target = target.encode()
    if libc.mount(source, target, 0, ctypes.c_ulong(MS_BIND), 0) != 0:
        e = ctypes.get_errno()
        raise OSError(e, os.strerror(e))


def mount(source, target, fs, options='', flag=0):
    if not os.path.exists(target):
        os.makedirs(target)
    ret = ctypes.CDLL('libc.so.6', use_errno=True).mount(ctypes.c_char_p(source.encode('ASCII')),
                                                         ctypes.c_char_p(target.encode('ASCII')),
                                                         ctypes.c_char_p(fs.encode('ASCII')),
                                                         ctypes.c_ulong(flag),
                                                         ctypes.c_char_p(options.encode('ASCII')))
    if ret < 0:
        errno = ctypes.get_errno()
        raise RuntimeError("Error mounting {} ({}) on {} with options '{}': {}".format(source, fs, target, options, os.strerror(errno)))
    return ret


def umount(source):
    """Umount a partition (accept only a partition device as input)"""
    ret = ctypes.CDLL('libc.so.6', use_errno=True).umount(ctypes.c_char_p(source.encode('ASCII')))
    if ret < 0:
        errno = ctypes.get_errno()
        raise RuntimeError("Error unmounting {} ({})".format(source, os.strerror(errno)))
    return ret

class MountDisk(object):
    # a simple file writer object
    def __init__(self, source, target, fs, options='', flag=0):
        self.source = source
        self.target = target
        self.fs = fs
        self.options = options
        self.flag = flag

    def __enter__(self):
        mount(self.source, self.target, self.fs, self.options, self.flag)
        return self.target

    def __exit__(self, *args):
        umount(self.target)

class MountBind(object):
    # a simple file writer object
    def __init__(self, source, target):
        self.source = source
        self.target = target

    def __enter__(self):
        mount_bind(self.source, self.target)
        return self.target

    def __exit__(self, *args):
        umount(self.target)

def mounted_point(partition):
    """Return a list of mounted point the partition is mounted to."""
    mounted_path = []
    if os.path.exists('/etc/mtab'):
        mounted_path = [line.split()[1] for line in open('/etc/mtab', 'r') if line.startswith(partition)]
    return mounted_path


def is_mount(path):
    with open("/etc/mtab", "r") as mtab:
        for line in mtab:
            if path == line.split()[1]:
                return True
    return False


CAL_UNMOUNTED = 0
CAL_MOUNTED_RO = 1
CAL_MOUNTED_RW = 2


def mount_cal():
    mmc = list(get_block_devices("MMC").keys())[0]
    if not is_mount("/cal"):
        mount("/dev/%sp%i" % (mmc, 2), '/cal', 'ext4')
        return CAL_UNMOUNTED
    else:
        with open("/proc/mounts","r") as procmount:
            for line in procmount.readlines():
                tline = line.split()
                if tline[1] == "/cal":
                    if "ro" in tline[3].split(","):
                        umount("/cal")
                        mount("/dev/%sp%i" % (mmc, 2), '/cal', 'ext4')
                        return CAL_MOUNTED_RO
                    elif "rw" in tline[3].split(","):
                        return CAL_MOUNTED_RW
                    else:
                        print("UNKNOWN CAL STATE !!!!")
    return -1


def umount_cal(mode):
    os.sync()
    mmc = list(get_block_devices("MMC").keys())[0]
    if mode == CAL_UNMOUNTED or mode == CAL_MOUNTED_RO:
        umount('/cal')
    if mode == CAL_MOUNTED_RO:
        mount("/dev/%sp%i" % (mmc, 2), '/cal', 'ext4', flag = MS_RDONLY)
    return mode


def sync():
    try:
        os.sync()
    except:
        os.system("sync")


def format_device(select):
    format_list = []
    mmc = list(get_block_devices("MMC").keys())[0]
    partition_table = get_partition_plan()
    for partid in range(len(partition_table)):
        part = partition_table[partid]
        if 'label' in part.keys() and not part['protected'] and (select is None or partid == (select - 1)):
            format_list.append(
                "mkfs.%s -F -L %s %s" % (part['format'], part['label'], "/dev/%sp%i" % (mmc, partid + 1)))
    for action in format_list:
        print(action)
        os.system(action)
    sync()

def mount_chroot(path="/mnt/fs"):
    mount_bind("/dev/", "%s/dev/"%path)
    mount("none", "%s/sys"%path, "sysfs")
    mount("none", "%s/proc"%path, "proc")
    try:
        copyfile("/etc/resolv.conf", "%s/etc/resolv.conf"%path)
    except SameFileError:
        pass
    except FileNotFoundError:
        open("%s/etc/resolv.conf"%path, 'a').close()


def umount_chroot(path="/mnt/fs"):
    umount("%s/dev"%path)
    umount("%s/sys"%path)
    umount("%s/proc"%path)


def chroot_exec(path, command):
    real_root = os.open("/", os.O_RDONLY)
    os.chroot(path)
    subprocess.run(command, cwd="/")
    sync()
    os.fchdir(real_root)
    os.chroot(".")
    os.close(real_root)

def get_root_partition():
    with open("/proc/cmdline", "r") as f:
        bootargs = f.readline().split(' ')
        rootpart = [arg for arg in bootargs if arg.startswith("root=")]
        if len(rootpart) < 1:
            print("Cannot determine root partition, fatal error, exiting")
            exit(-1)
        rootpart = rootpart[0].split("=")[1]
        if not "p" in rootpart:
            # Used if reelase-manager is called from pmulti, in this case cmadline contains root=/dev/ram
            rootpart = subprocess.getoutput('lsblk -o MOUNTPOINT,KNAME | grep "^/ " | sed "s|/||g" | sed "s/ //g"')
        return rootpart[:rootpart.rindex("p")]
