

from .json_lite_builder import JsonLiteBuilder, JsonLite
from .specs import CommonSpec, OtdrSpec


_UNITS_TO_SI = {
    "m": lambda value, r: round(value, r),
    "Km": lambda value, r: round(value * 1000., r),
    "ft": lambda value, r: round(value / 3.281, r),
    "mi": lambda value, r: round(value * 1609.34, r),
    "Kft": lambda value, r: round(value / 304.8, r)
}


class OtdrBuilder(JsonLiteBuilder):
    def __init__(self, cdm):
        super().__init__(cdm)

        self.__root_layer = None
        self.__lr_result_schema = None
        self.__element_schema = None
        self.__element_result_schema = None
        self.__prev_section_schema = None
        self.__prev_section_result_schema = None

        self.reset()

    def reset(self) -> None:
        self._json_lite = JsonLite()
        self._json_lite.format = OtdrSpec.format()
        compress_format = [*OtdrSpec.abridgments(), *OtdrSpec.acronyms()]
        self._json_lite.set_compress_format(compress_format)

        self.__root_layer = OtdrSpec.schema()

        lr_results = self.__root_layer.get("LinkResults", dict()).get("Results", list())
        assert len(lr_results) > 0
        self.__lr_result_schema = lr_results.pop(0)

        elements = self.__root_layer.get("Measurement", dict()).get("Elements", list())
        assert len(elements) > 0
        self.__element_schema = elements.pop(0)

        schema_element_results = self.__element_schema.get("Results", list())
        assert len(schema_element_results) > 0
        self.__element_result_schema = schema_element_results.pop(0)

        self.__prev_section_schema = self.__element_schema.pop("PreviousFiberSection", dict())
        prev_section_results = self.__prev_section_schema.get("Results", list())
        assert len(prev_section_results) > 0
        self.__prev_section_result_schema = prev_section_results.pop(0)

    def produce_root_layer(self) -> None:
        self._transform_entries(self.__root_layer)

        self._json_lite.data = self.__root_layer

    def produce_sub_layers(self) -> None:
        lr_results = self.__root_layer.get("LinkResults", dict()).get("Results", list())
        elements = self.__root_layer.get("Measurement", dict()).get("Elements", list())

        elements.extend([dict()] * self._cdm.max_id_slm)

        for test_idx in range(self._cdm.tests_number):
            for result_idx in range(self._cdm.results_number[test_idx]):
                indexes = (test_idx, result_idx)
                result = self._make_sub_layer(self.__lr_result_schema, indexes)
                lr_results.append(result)

                events = self._cdm.get_events(test_idx, result_idx)

                for event_idx, event in enumerate(events):
                    indexes = (test_idx, result_idx, event_idx)

                    id_slm = event.get("idSlm")

                    if elements[id_slm - 1] == dict():
                        elements[id_slm - 1] = self._make_sub_layer(self.__element_schema, indexes)

                    element_result = self._make_sub_layer(self.__element_result_schema, indexes)

                    element_type = elements[id_slm - 1].get("Type")

                    print(element_type)

                    if element_type is not None:
                        elements[id_slm - 1]["Type"] = CommonSpec.convert_event(element_type)

                    elements[id_slm - 1]["Results"].append(element_result)

                    event_type = event.get("eventType")

                    if event_type not in (None, "First Connector"):  # TODO
                        prev_fiber_section = self._make_sub_layer(self.__prev_section_schema, indexes)
                        prev_fiber_section_result = self._make_sub_layer(self.__prev_section_result_schema, indexes)

                        if "PreviousFiberSection" not in elements[id_slm - 1]:
                            prev_fiber_section["Results"].append(prev_fiber_section_result)
                            elements[id_slm - 1]["PreviousFiberSection"] = prev_fiber_section

                        else:
                            elements[id_slm - 1]["PreviousFiberSection"]["Results"].append(prev_fiber_section_result)

    def post_process(self) -> None:
        units = self._cdm.get("$.tests.0.results.data.otdrResults.distanceUnits", "m")

        link_results = self._json_lite.data.get("LinkResults", dict())

        link_length = link_results.get("Length")

        if link_length is not None:
            if units in _UNITS_TO_SI:
                link_results["Length"] = str(_UNITS_TO_SI[units](float(link_length), 2))

    def _transform(self, data, accessor, entry) -> None:
        assert type(accessor) in (int, str), "unexpected accessor type: <%s>" % type(accessor)

        if (type(entry) is str) and entry.startswith("$."):  # entry is a json-path
            content = self._cdm.get(entry)

            if content is None:
                content = "NaN"

            if accessor in ("TestDate",):
                content = content.split("T")[0]  # get only date from ISO 8601 formatted date-time string

            elif accessor in ("LocationDirection",):
                if ">" in content:
                    content = "AB"
                elif "<" in content:
                    content = "BA"

            data[accessor] = str(content)


