import threading
import termios
import sys
import os
from . import *
from .dynmodule import get_dynamic_module


class ConsoleInput(threading.Thread):
    '''
    This class is the class for the console input thread.
    This thread read each char typed on keyboard and send it
    in the flux or parse the command...
    This is also the thread for auto-completion of internal command
    '''

    def __init__(self, mode):
        '''
        Constructor, need session
        '''
        self.__mode = mode
        # mode is True for direct console command
        #         False to use > prefix

        self.__focus = False
        # focus is used only in prefix mode
        # True is when we are in command mode

        self.__navigating = False
        # True indicate we navigate in history

        self.__history = []
        # Command history

        self.__cmd_buf = []
        # in construction current command

        self.__exit = False
        # To leave main loop

        self.__commandmanager = []
        # A command manager is object with list_commands, run_command
        # and get_command_options callback

        self.__commands = []
        # List of commands and callback

        self.__navindex = 0

        self.__prompt = None

        threading.Thread.__init__(self)

        self.__fd = sys.stdin.fileno()

        if os.isatty(self.__fd):
            self.__ttyparam = termios.tcgetattr(self.__fd)
            new = termios.tcgetattr(self.__fd)
            new[3] = new[3] & ~termios.ICANON & ~termios.ECHO
            new[6][termios.VMIN] = 1
            new[6][termios.VTIME] = 0
            termios.tcsetattr(self.__fd, termios.TCSANOW, new)
            termios.tcsendbreak(self.__fd, 0)

    def __getstr(self):
        '''
        __getstr is the same getchar than in C program.
        '''
        try:
            ch = os.read(self.__fd, 1024).decode("utf-8")
        except UnicodeDecodeError:
            pass #0xc3
        r = ch.replace("\r", "")
        return(r)

    def add_command(self, name, fc):
        '''
        Declare a new Command
        '''
        self.__commands.append((name, fc))

    def set_prompt(self, prompt):
        self.__prompt = prompt

    def __run_command(self, name):
        Log("__CMD: %s" % name.strip())
        for cmd, fc in self.__commands:
            if cmd.split()[0] == name.split()[0]:
                O().print_and_return("")
                fc(name)
                return

    def list_commands(self):
        lst = ["Exit"]
        for cmd in self.__commands:
            lst.append(cmd[0])
        return lst

    def get_command(self, cmd):
        for c in self.__commands:
            if c[0] == cmd:
                return c[1]
        print_error("Can't find command %s" % cmd)

    def __send_command(self):
        '''
        Internal command is completed, we send it to session,
        and come back to normal mode
        '''
        O().print_str("\n", False, color="default", stream="console")
        command = "".join(self.__cmd_buf).strip()
        if len(command) == 0:
            if not self.__mode:
                O().unlock_focus()
            self.__focus = False
            self.__navigating = False
            self.__cmd_buf = []
            if self.__prompt is not None:
                O().print_str("%s " % self.__prompt,nline=True)
        elif command == "Exit":
            if not self.__mode:
                O().unlock_focus()
            E().stop()
            E().send_event("EXIT", "")
            self.__exit = True
        else:
            self.__run_command(command)
            if not self.__mode:
                O().unlock_focus()
            self.__history.append("".join(self.__cmd_buf))
            self.__focus = False
            self.__navigating = False
            self.__cmd_buf = []
            if self.__prompt is not None:
                O().print_str("%s " % self.__prompt,nline=True)

    def __set_internal_mode(self):
        '''
        Switch to Internal command mode,
        We stop all write from stream to console and display
        the prompt on console passed to inverse mode video blue/white
        '''
        O().lock_focus()
        self.__focus = True
        O().print_str("\n", lock=False, stream="console",
                      nline=True, color="default")
        O().print_str(">", lock=False, stream="console")

    def __search_complete_command(self):
        '''
        Function Used to AutoComplete the current command
        2 Modes :
        -> 1) we haven't 2 word or a word and a space,
              we print all possible commands with filter
        -> 2) We have a the command, we need argument,
              we ask to command the possibilities
        '''
        command = "".join(self.__cmd_buf)
        possibilities = []
        nbword = len(command.strip().split())
        if (nbword == 0) or ((nbword == 1) and (command[-1] != " ")):
            # First Case, command name
            if len(command.split()) != 0:
                currentWord = command.split()[0].strip()
            else:
                currentWord = ""
            possibilities = self.list_commands()
        else:
            # Second case, parameters
            possibilities = []
            f = None
            for cmd, fc in self.__commands:
                if cmd.split()[0] == command.split()[0]:
                    f = fc
            if f is None:
                return
            params = command.split()
            tree = f.parameters
            if command[-1] == " ":
                currentWord = ""
            elif len(params) > 1:
                currentWord = params.pop()
            else:
                currentWord = ""
            params.reverse()
            params.pop()
            while(len(params) > 0):
                tree = tree[params.pop()]
            for p in tree.keys():
                if p[:len(currentWord)] == currentWord:
                    possibilities.append(p)

        # Remove all impossible possibilities
        removelist = []
        for possibility in possibilities:
            if possibility[0:len(currentWord)] != currentWord:
                removelist.append(possibility)
        for remove in removelist:
            possibilities.remove(remove)
        # If all possibilities begin with sames letters,
        # autocompete this letters
        if len(possibilities) > 1:
            tocontinue = True
            tocomplete = currentWord
            index = len(currentWord)
            while tocontinue:
                if len(possibilities[0]) <= index:
                    tocontinue = False
                    break

                chartest = possibilities[0][index]
                for possibility in possibilities:
                    if len(possibility) <= index:
                        tocontinue = False
                        break
                    if possibility[index] != chartest:
                        tocontinue = False
                        break
                if tocontinue:
                    tocomplete += chartest
                    index += 1
            self.__complete_command(tocomplete[len(currentWord):], False)
            O().print_and_return(possibilities)
        elif len(possibilities) == 1:
            # If they are only once solution, complete it
            self.__complete_command(possibilities[0][len(currentWord):], True)

    def __complete_command(self, word, end):
        '''
        Autocomplete command with word.
        If end == True, we add a space a the end
        '''
        for char in word+(end*" "):
            self.__cmd_buf.append(char)
        O().print_str(word+(end*" "), lock=False, stream="console")

    def __backspace(self):
        '''
        Erase the last character of the command
        '''
        self.__cmd_buf = self.__cmd_buf[:-1]
        O().print_str("\b \b", lock=False, stream="console", color="default")

    def __navigate(self, direction):
        '''
        Navigate in history,
        UP(65) print the previus command,
        DOWN(66) print the next
        '''
        if len(self.__history) == 0:
            return
        command = "".join(self.__cmd_buf)
        if direction == 65:  # UP
            if not self.__navigating:
                self.__navigating = True
                self.__navindex = len(self.__history)
                if len(command.strip()) > 1:
                    self.__history.append(command)
            self.__navindex = self.__navindex - 1
            if self.__navindex < 0:
                self.__navindex = 0
            newtcom = self.__history[self.__navindex]
            O().print_str("\r", False)
            O().print_str(" "*(len(command)+1)+"\r", lock=False,
                          stream="console", color="default")
            O().print_str(">"+newtcom, lock=False, stream="console",
                          nline=True)
            self.__cmd_buf = [newtcom]
        elif direction == 66:  # Down
            if self.__navigating:
                self.__navindex = self.__navindex + 1
                if self.__navindex > len(self.__history) - 1:
                    self.__navindex = len(self.__history) - 1
                O().print_str("\r", False)
                O().print_str(" "*(len(command)+1)+"\r", lock=False,
                              stream="console", color="default")
                self.__cmd_buf = [self.__history[self.__navindex]]
                O().print_str(">"+self.__cmd_buf[0], lock=False,
                              stream="console", nline=True)

    def run(self):
        '''
        Main function, core thread
        '''
        oldchar = "\n"
        if self.__prompt is not None:
            O().print_str("%s " % self.__prompt,nline=True)
        while not self.__exit:
            char = self.__getstr()
            if not self.__mode and not self.__focus:
                # We are in External mode, ">" switch mode;
                # and if we are inline, each char is sent to stream
                if oldchar == "\n" and char == ">":
                    self.__set_internal_mode()
                else:
                    O().write(char)
                    O().flush()
                oldchar = char
            else:
                # We are in Internal Command Mode
                st = list(char)
                st.reverse()
                while len(st) > 0:
                    c = st.pop()
                    if c == "\n":
                        self.__send_command()
                    elif c == "\t":
                        self.__search_complete_command()
                    elif ord(c) == 127:
                        self.__backspace()
                    elif ord(c) == 27:
                        if (len(st) > 0) and (ord(st.pop()) == 91):
                            if (len(st) > 0):
                                self.__navigate(ord(st.pop()))
                    else:
                        self.__cmd_buf.append(c)
                        O().print_str(c, lock=False, stream="console")
        O().print_str("\n", color="default", stream="console")
        O().flush()
        termios.tcsetattr(self.__fd, termios.TCSAFLUSH, self.__ttyparam)
        if env("mode.verbode") == "on":
            print_info("ConsoleInput close", False)

    @classmethod
    def create(cls, locked=False):
        if "cinput" not in __builtins__.keys():
            __builtins__["cinput"] = cls(locked)
        return __builtins__["cinput"]
