<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\http;

use app\settings\SMTSmartOtuSettingsCache;

use app\util\SMTLogger;
use app\admin\SMTUserDto;
use app\settings\SMTSmartOtuSettings;

date_default_timezone_set('UTC');

class SMTSession
{
    /**
     * Last session access time
     * @var integer
     */
    const LAST_ACCESS_TIME = "last_access_time";
    
    /**
     * Period of date-time reset: during that period, no check is performed on session timeout
     * @var integer
     */
    const RESET_PERIOD = 60;
    
    /**
     * Last update time of the session directories
     * 
     * @var integer
     */
    const DIRECTORIES_LAST_UPDATE_TIME = "directories_last_update_time";
    
    /**
     * Token for cross site request forgery
     */
    const USED_TOKEN= "used_tokens";
    
    /**
     * Token for cross site request forgery length
     */
    const TOKEN_SHA_256_LENGTH= 64;
    
    /**
     * Max Token for cross site request forgery length
     */
    const MAX_TOKEN_COUNT= 100;
    
    const CURRENT_OPERATION_NUMBER = "operation_number";
    const OPERATION_DELIMITER = "@@";
    
    /**
     * Register session handler.
     * Start a new http session or restore it.
     * 
     * @param $checkSessionTimedOut boolean whether the session timeout must be checked: TRUE by default
     */
    function __construct( $checkSessionTimedOut = TRUE )
    {
        //register session handler
        $handler = new SMTFileSessionHandler();
        session_set_save_handler(
                                    array($handler, 'open'),
                                    array($handler, 'close'),
                                    array($handler, 'read'),
                                    array($handler, 'write'),
                                    array($handler, 'destroy'),
                                    array($handler, 'gc')
                                 );
        
        //Mandatory to use objects (SMTFileSessionHandler) as session manager.
        register_shutdown_function('session_write_close');

        $protocol = apache_getenv('HTTPS') ? 'https:' : 'http:';
        $sec_protocol = 'https';
        if(strpos($protocol, $sec_protocol) !== false){
           @ini_set('session.cookie_secure', '1');
        } else {
           @ini_set('session.cookie_secure', '0');
        }

        //start or restore php session
    	session_start();
    	
    	//check session timeout
    	if ( $checkSessionTimedOut )
    	{
    	    $this->checkSessionTimedOut();
    	}
    }
    
    /**
     * Check session timout 
     * If timeout occured, close the session and indirectly unlog the current user because its identifiers are registered in session; then, create a new empty Session.
     */
    public function checkSessionTimedOut()
    {
    	$timeout = $this->hasSessionTimedOut();
    	//if timeout occured, release the session (unlog indirectly user because user idenitfiers are registered in session) and create a new Session
    	if ( $timeout )
    	{
    		$this->restartSession();
    	}
    }
    
    /**
     * Destroy previous running session and start a new one.
     * 
     */
    function restartSession()
    {
        //test if a session is already running
        $this->closeSession("restart session");

        $protocol = apache_getenv('HTTPS') ? 'https:' : 'http:';
        $sec_protocol = 'https';
        if(strpos($protocol, $sec_protocol) !== false){
           @ini_set('session.cookie_secure', '1');
        } else {
           @ini_set('session.cookie_secure', '0');
        } 

        //start session
        session_start();                
        //update last application access to track timeout
        $this->updateLastAccess();        
    }
    
    /**
     * Destroy running session.
     *
     */
    function closeSession($cause)
    {        
    	//test if a session is already running
    	if ( self::session_is_active() )
    	{
    	    SMTLogger::getInstance()->trace("Close session: ".$cause);
    	    self::destroySession();
    	}    	
    	else
    	{
            //Unset all the session variables.
    	    $_SESSION = array();
    	}
    }    
    
    /**
     * close, cleanup and destroy the current session
     */
    static function destroySession()
    {   
    	// To kill the session, also delete the session cookie.
    	if (ini_get("session.use_cookies"))
    	{
    		$params = session_get_cookie_params();
    		setcookie(session_name(), '', time() - 42000,
    		$params["path"], $params["domain"],
    		$params["secure"], $params["httponly"]);
    	}
    	
    	// Unset all of the session variables.
    	$_SESSION = array();
    	
    	// Finally, destroy the session.
    	session_destroy();
    }    
    
    /**
     * Whether a user session exists
     *
     * @return bool
     */
    static function session_is_active()
    {
    	//disable warning logging
    	$currentErrorLevel = error_reporting();
    	error_reporting(E_ERROR);
    
    	$setting = 'session.use_trans_sid';
    	$current = ini_get($setting);
    	if (FALSE === $current)
    	{
    		//throw new UnexpectedValueException(sprintf('Setting %s does not exists.', $setting));
    		return false;
    	}
    	$result = @ini_set($setting, $current);
    
    	//enable warning logging
    	error_reporting( $currentErrorLevel );
    
    	return $result !== $current;
    }    
    
    /**
     * Last access time to the session
     * @return int
     */
    public function getLastAccess()
    {
        //Lorsque plusieurs process php tournent dans la même session (plusieurs requêtes simultanées depuis un même client), 
        //cette variable peut avoir des valeurs différentes dans les process php.
        //Attention ces variables $_SESSION ne sont enregistrées que lors de la fin du script.
        return isset($_SESSION[self::LAST_ACCESS_TIME])?$_SESSION[self::LAST_ACCESS_TIME]:time();
    }
    
    /**
     * Last update time of the session directories
     * @return int
     */
    public function getSessionDirectoriesLastUpdate()
    {
        //Lorsque plusieurs process php tournent dans la même session (plusieurs requêtes simultanées depuis un même client),
        //cette variable peut avoir des valeurs différentes dans les process php.
        //Attention ces variables $_SESSION ne sont enregistrées que lors de la fin du script.        
    	return isset($_SESSION[self::DIRECTORIES_LAST_UPDATE_TIME])?$_SESSION[self::DIRECTORIES_LAST_UPDATE_TIME]:time();
    }
    
    /**
     * Update last access time to the session and returns the value.
     * Update sessions directories last update value.
     * 
     */
    public function updateLastAccess()
    {
    	//update last access
    	$_SESSION[self::LAST_ACCESS_TIME] = time();
    	
        //Lorsque plusieurs process php tournent dans la même session (plusieurs requêtes simultanées depuis un même client),
        //cette variable peut avoir des valeurs différentes dans les process php.
        //Attention ces variables $_SESSION ne sont enregistrées que lors de la fin du script.
                
        //update session directory
        $_SESSION[self::DIRECTORIES_LAST_UPDATE_TIME] = SMTSessionDirectoryUtil::touchSessionDirectoryFiles( $this->getSessionDirectoriesLastUpdate());
    }
    
    /**
     * Reset last access time to the session and returns the value.
     * Reset sessions directories last update value.
     *
     */
    public function resetLastAccess()
    {
    	//update reset date
    	self::setLastResetDateTime( time() );
    	 
    	//update last access
    	$_SESSION[self::LAST_ACCESS_TIME] = time();
    	
    	$_SESSION[self::DIRECTORIES_LAST_UPDATE_TIME] = time();
    	
    	session_commit();
    	
        //Lorsque plusieurs process php tournent dans la même session (plusieurs requêtes simultanées depuis un même client),
        //cette variable peut avoir des valeurs différentes dans les process php.
        //Attention ces variables $_SESSION ne sont enregistrées que lors de la fin du script.
                
        //update session directory
        SMTSessionDirectoryUtil::touchSessionDirectoryFiles( time() );  	
    }
    
    /**
     * Whether the session has timed-out.
     * Update the last session access.
     * 
     * return boolean
     */
    public function hasSessionTimedOut()
    {
        $hasSessionTimedOut = false;
        
        //during one minute period after date time update, session musn't timeout
        if ( (self::getLastResetDateTime() > 0) &&  ( ( time() - self::getLastResetDateTime() ) < self::RESET_PERIOD) )
        {
        	$hasSessionTimedOut = false;
        }        
        else if ( ( time() - $this->getLastAccess() ) < SMTSmartOtuSettings::getSessionTimeOut() * 60 )
        {
            $this->updateLastAccess();
        }
        //timeout
        else
        {
            $hasSessionTimedOut = true;
        }                
        
        return $hasSessionTimedOut;
    }
    
    /**
     * Set user in session.
     *
     * @param SMTUserDto
     */
    public function setUser( SMTUserDto $user = NULL )
    {
    	$_SESSION[SMTUserDto::USER] = $user;
    }
    
    /**
     * Retrieve user from session or null if none is set.
     *
     * @return SMTUserDto
     */
    public function getUser()
    {
    	return isset($_SESSION[SMTUserDto::USER])? $_SESSION[SMTUserDto::USER]: NULL;
    }
    
    /**
     *
     * @param string $address
     */
    public function setOtuAddress($address)
    {
        $_SESSION[SMTSmartOtuSettings::OTU_ADDRESS] = $address;
    }
    
    /**
     * Retrieve the current otu address from session or from default settings if none is set.
     *
     * @return string
     */
    public function getOtuAddress()
    {
        return isset($_SESSION[SMTSmartOtuSettings::OTU_ADDRESS]) ? $_SESSION[SMTSmartOtuSettings::OTU_ADDRESS] : SMTSmartOtuSettings::getOtuAddress();
    }
    
    /**
     * Store used token to avoid replay attack
     * @param string $token
     */
    public static function addToken($token)
    {
    	if (isset($_SESSION[self::USED_TOKEN]) && strlen($_SESSION[self::USED_TOKEN]) >= (self::TOKEN_SHA_256_LENGTH + 1)*self::MAX_TOKEN_COUNT)
    	{
    		$_SESSION[self::USED_TOKEN] = substr($_SESSION[self::USED_TOKEN], self::TOKEN_SHA_256_LENGTH + 1);
    	}
    	if (!isset($_SESSION[self::USED_TOKEN]) )
    	{
    		$_SESSION[self::USED_TOKEN] = $token.',';
    	}
    	else 
    	{
    		$_SESSION[self::USED_TOKEN] = $_SESSION[self::USED_TOKEN].$token.',';
    	}
    }
    
    /**
     * Whether the given token has alreay been used
     *
     * @return boolean
     */
    public static function hasToken($token)
    {
    	return isset($_SESSION[self::USED_TOKEN]) && strpos($_SESSION[self::USED_TOKEN],$token) !== false;
    }

    /**
     * Get the date of the last date-time reset
     *
     * @return int
     */
    public static function getLastResetDateTime()
    {
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	return ( $settings != NULL )? $settings->getLastResetDateTime() : 0;
    }
    
    /**
     * Set the date of the last date-time reset in application scope memory
     *
     * @param int $lastResetDateTime
     */
    private static function setLastResetDateTime( $lastResetDateTime )
    {
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setLastResetDateTime( $lastResetDateTime );
    	$settings->save();
    }    
}

?>
