<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\events\operations;

use app\events\operations\SMTIOperationEventDto;

use app\events\operations\SMTIReplaceableEventDto;

use app\events\SMTEventDto;

/**
 * Operation event
 * 
 * @author Sylvain Desplat
 *
 */
class SMTOperationEventDto extends SMTEventDto implements SMTIReplaceableEventDto, SMTIOperationEventDto
{       
    const INIT = 'INIT';
    const WAITING = 'WAITING';    
    const IN_PROGRESS = 'IN_PROGRESS';
    const AVAILABLE = 'AVAILABLE';
    const ENDED = 'ENDED';
    const INVALID_STATUS = 'INVALID_STATUS';
    const NO_PROGRAMMED = 'NO_PROGRAMMED';
    const FAILED = 'FAILED';    
    const BAD_TIMESLOT = 'BAD_TIMESLOT';
        
    const ERROR_ONE_TEST_ON_DEMAND = 'ERROR_ONE_TEST_ON_DEMAND';
    const ERROR_ONE_MEASURE_ON_DEMAND = 'ERROR_ONE_MEASURE_ON_DEMAND';
    
    const TEST_ON_DEMAND = 'TEST_ON_DEMAND';
    const MEASURE_ON_DEMAND = 'MEASURE_ON_DEMAND';
    const MEASURE_INSTRUMENT = 'MEASURE_INSTRUMENT';
    const MEASURE_INSTRUMENT_PORT = 'MEASURE_INSTRUMENT_PORT_%d_';    
    
    /**
     * Operation status
     * 
     * @var string
     */
    protected $status = self::INVALID_STATUS;
    
    /**
     * Operation status decoded into a localized message
     * @var unknown
     */
    protected $localizedMessage = NULL;   
      
    function setOperationId( $operationId )
    {
    	$this->setId( $operationId );
    }        
    
    function getOperationId()
    {
    	return $this->getId();
    }    

    function setLocalizedMessage( $localizedMessage )
    {
    	$this->localizedMessage = $localizedMessage;
    }
    
    function getLocalizedMessage()
    {
    	return $this->localizedMessage;
    }    
    
    function hasOperationEnded()
    {
        return !( $this->isOperationRunning() );
    }
    
    function getStatus()
    {
        return $this->status;
    }
    
    function setStatus( $status )
    {
        if ( self::isOperationStatus( $status ) )
        {
    	    $this->status = $status;
    	    $this->setLocalizedMessage( self::decodeMessage( $status ) );
        }
        else
        {
            throw new \InvalidArgumentException( MSG_ERROR_INVALID_OPERATION_STATUS );
        }
    }
    
    function canReplace( SMTIReplaceableEventDto $eventDto)
    {
        if ( $eventDto != NULL && $eventDto instanceof SMTOperationEventDto )
        {
            return ( $this->getOperationId() == $eventDto->getOperationId() && 
                    $this->getStatus() == $eventDto->getStatus()
                      );
        }
        return FALSE;
    }    
    
    /**
     * Whether the status is in waiting, init or in_progress
     *
     *
     * @return boolean
     */
    public function isOperationRunning()
    {
    	$running = FALSE;
    	switch( $this->status )
    	{
    		case self::INIT:
    		case self::IN_PROGRESS:
    		case self::WAITING:
    			$running = TRUE;
    			break;
    
    		default:
    			$running = FALSE;
    			break;
    	}
    	return $running;
    }    
    
    /**
     * Whether the operation has successfully ended and whether the result is available
     *
     * @param the status to check
     * 
     * @return boolean  Whether the operation has successfully ended and the whether the result is available
     */
    public static function isResultAvailable( $status )
    {
    	$available = FALSE;
    	switch( $status )
    	{
    		case self::AVAILABLE:
    			$available = TRUE;
    			break;
    
    		default:
    			$available = FALSE;
    			break;
    	}
    	return $available;
    }    
    
    /**
     * Whether the status given is an operation status
     * 
     * @param string $code operation status
     * 
     * @return boolean
     */
    public static function isOperationStatus( $code )
    {
        $valid = FALSE;
        switch( $code )
        {
        	case self::INIT:
        	case self::IN_PROGRESS:
        	case self::AVAILABLE:
        	case self::WAITING:
        	case self::ENDED:
        	case self::INVALID_STATUS:
        	case self::NO_PROGRAMMED:
        	case self::BAD_TIMESLOT:
        	case self::FAILED:
        	case self::ERROR_ONE_TEST_ON_DEMAND:
        	case self::ERROR_ONE_MEASURE_ON_DEMAND:
        		$valid = TRUE;
        		break;
        
        	default:
        	    $valid = FALSE;
        		break;
        }
        return $valid;
    }           
    
    /**
     * Translates status code. If the code is not found, returns an empty string.
     *
     * @param string $code The status code to translate
     *
     * @return string the status message or an empty string if the status couldn't be found.
     */
    public static function decodeMessage( $code )
    {
    	$message = MSG_INVALID_STATUS.$code;
    
    	switch( $code )
    	{
    	    case self::INIT:
    	    	$message = MSG_MEASURE_INIT;
    	    	break;
    	    case self::IN_PROGRESS:
    	    	$message = MSG_MEASURE_IN_PROGRESS;
    	    	break;
    	    case self::WAITING:
    	    	$message = MSG_MEASURE_WAITING;
    	    	break;
    	    case self::AVAILABLE:
    	    case self::ENDED:
    	    	$message = MSG_MEASURE_AVAILABLE;
    	    	break;    	    
    		case self::INVALID_STATUS:
    			$message = MSG_INVALID_STATUS;
    			break;
			case self::BAD_TIMESLOT:
				$message = MSG_ERROR_MEASURE_BAD_TIMESLOT;
				break;
			case self::NO_PROGRAMMED:
				$message = MSG_ERROR_MEASURE_NOT_PROGRAMMED;
				break;
			case self::FAILED:
				$message = MSG_ERROR_MEASURE_FAILED;
				break;    			
    		case self::ERROR_ONE_TEST_ON_DEMAND:
    			$message = MSG_ERROR_ONE_TEST_ON_DEMAND;
    			break;
			case self::ERROR_ONE_MEASURE_ON_DEMAND:
				$message = MSG_ERROR_ONE_MEASURE_ON_DEMAND;
				break;    			    			
    		default:
    			break;
    	}
    	return $message;
    }    
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }        
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }    
}

?>