import QtQuick 2.1
import QtQml.Models 2.1
import QtQuick.Layouts 1.0

import common 1.0
import graph_plugin 1.0

Item { id: label

    property alias pressed: mouseArea.pressed
    property real minimumWidth: 136
    default property alias values: valueRow.children
    property bool posLabelVisible: true
    property real cursorLabelPosition: -1.0
    readonly property bool cursorLabelPositionSet: label.cursorLabelPosition !== -1.0
    property bool canDragY: false
    property int titleNumberOfLines: 1

    property int minDragPosX: 0
    property int maxDragPosX: Math.max(parent.width-1, 0)

    signal clicked(var mouse)
    signal released()
    signal longPress()

    property string __xValue: GraphCursor.formattedXValue
    property real __yPos: 0.0;

    width: Math.max(minimumWidth, Math.ceil(valueRow.implicitWidth), Math.ceil(xValue.implicitWidth) + 10)
    height: column.height
    smooth: false

    property bool largeFont: false

    property alias yPosition: label.y

    Column { id: column
        spacing: 0

        BorderImage { id: posLabel
            border { left: 8; right: 8; top: 0; bottom: 0 }
            height: 20 * label.titleNumberOfLines
            width: label.width
            source: theme.dynamicDirPrefix + "cursor-label-top"
            smooth: false
            visible: label.posLabelVisible

            Text { id: xValue
                color: "#FFFFFF"
                font: largeFont ? systemFont.LargeTitleFont : systemFont.HeaderFont
                anchors.centerIn: parent
                text: label.__xValue
            }
        }

        RowLayout { id: valueRow
            width: label.width
            height: 30
            spacing: 0
            smooth: false
        }
    }

    RangeModel { id: range
        minimumValue: label.minDragPosX
        maximumValue: label.maxDragPosX
        value: !label.cursorLabelPositionSet ? 0 : label.cursorLabelPosition
        stepSize: 1
        positionAtMinimum: label.minDragPosX
        positionAtMaximum: label.maxDragPosX - label.width
        onValueChanged:
        {
            if(!label.cursorLabelPositionSet && mouseArea.drag.active)
            {
                label.GraphCursor.cursor.setPosOnScreen(value);
            }
        }
    }

    GraphCursor.onXPosOnScreenChanged: {
        if(!label.cursorLabelPositionSet) range.value = GraphCursor.xPosOnScreen;
    }

    MouseArea { id: mouseArea
        anchors.fill: parent
        drag.target: label
        drag.axis: label.canDragY ? Drag.XAndYAxis : Drag.XAxis
        drag.minimumX: range.positionAtMinimum
        drag.maximumX: range.positionAtMaximum
        drag.minimumY: label.canDragY ? 0 : 0
        drag.maximumY: label.canDragY ? label.parent.height - label.height : 0
    }

    Binding {
        when: !mouseArea.drag.active
        target: label
        property: "x"
        value: Math.round(range.position)
    }

    // when the slider is dragged, the value binds to the handle's position
    Binding {
        when: mouseArea.drag.active
        target: range
        property: "position"
        value: label.x
    }

    Connections {
        target: mouseArea
        onClicked: clicked(mouse)
        onReleased: {
            if (label.parent) {
                var pos = mapToItem(label.parent, mouse.x, mouse.y)
                __yPos = pos.y
            }
            released()
        }
        onPressAndHold: longPress()
    }

    Connections {
        target: label.parent ? label.parent : null
        onHeightChanged: {
            if (label.parent.height > 0 && ((__yPos + label.height) > label.parent.height)) {
                label.y = (label.canDragY ? label.parent.height - label.height : 0)
            }
        }
    }
}
