#!/usr/bin/python3

import parted
import os
import sys
import tarfile
import subprocess
import traceback
import linecache
from pathlib import Path
import xml.etree.ElementTree as etree
from urllib.request import urlopen
from viavi.stdhw.blockutils import sync, format_device, mount_chroot, umount_chroot, chroot_exec, get_block_devices, get_partition_plan, mount, umount, mount_bind, get_partition_plan

GRAY = (0xBB, 0xBB, 0xBB)
BLACK = (0x0, 0x0, 0x0)
BLUE = (0x0, 0x0, 0xFF)

last_pbar = 0


def progress_bar(gui, value):
    global last_pbar
    value = int(value)
    if gui is not None:
        if value != last_pbar:
            last_pbar = value
            gui.rectangle(50, 250, int((value * 700) / 100), 50, BLUE)
            print("Install %i%%"%value)
            sys.stdout.flush()



def part_type(typ):
    if typ == 'p':
        return parted.PARTITION_NORMAL
    if typ == 'x':
        return parted.PARTITION_EXTENDED
    if typ == 'l':
        return parted.PARTITION_LOGICAL


def create_partitions_table():
    mmc = list(get_block_devices("MMC").keys())[0]
    partition_table = get_partition_plan()
    device = parted.getDevice("/dev/%s"%mmc)
    disk = parted.freshDisk(device, 'msdos')
    format_list = []
    for part in partition_table:
        if 'end' in part.keys():
            geometry = parted.Geometry(device=device, start=part['start'], end=part['end'])
        else:
            free_space_regions = disk.getFreeSpaceRegions()
            geometry = free_space_regions[-1]
            geometry.start += 1
        if part['format'] != 'raw':
            filesystem = parted.FileSystem(type=part['format'], geometry=geometry)
        else:
            filesystem = None
        partition = parted.Partition(disk=disk, type=part_type(part['type']), fs=filesystem, geometry=geometry)
        disk.addPartition(partition=partition, constraint=device.minimalAlignedConstraint)
        if 'label' in part.keys():
            if part['format'].startswith('fat'):
                format_list.append("mkfs.vfat -F %s -n %s %s"%(part['format'][3:], part['label'], partition.path))
            else:
                format_list.append("mkfs.%s -F -L %s %s"%(part['format'], part['label'], partition.path))

    disk.commit()
    sync()
    for action in format_list:
        print(action)
        os.system(action)
    sync()



def mount_target(mmc, fs):
    mount("/dev/%sp%i"%(mmc, fs), "/mnt/fs/", "ext4")
    mount("/dev/%sp%i"%(mmc, 2), '/mnt/fs/cal', 'ext4')
    mount("/dev/%sp%i"%(mmc, 3), '/mnt/fs/user/disk', 'ext4')


def umount_target():
    umount('/mnt/fs/user/disk')
    umount('/mnt/fs/cal')
    umount('/mnt/fs')


def StreamDecorator(original, tarsize, notify):
    setattr(original.__class__, "oldread", original.read)
    setattr(original.__class__, "notify", notify)
    original.lastpercent = 0
    original.tarsize = int(tarsize)
    original.currentsize = 0

    def myread(self, size):
        percent = int(self.currentsize * 100 / self.tarsize)
        if percent != self.lastpercent:
            self.notify(percent)
            self.lastpercent = percent
        self.currentsize += size
        return self.oldread(size)

    myread.__doc__ = "My read"
    myread.__name__ = "read"
    setattr(original.__class__, "read", myread)
    return original


def printme(self, percent):
    percent = int(percent)
    if percent % 5 != 0:
        return

    if percent != printme.last:
        printme.last = percent
        print("Installation %i%%"%percent)


printme.last = 120


def install_image(uri, notifier=printme, gui=None):
    import os.path
    sync()
    if gui:
        def print_gui(self, percent):
            progress_bar(gui, percent)
        notifier = print_gui
    f = urlopen(uri)
    xmlstring = f.read()
    releasexml = etree.fromstring(xmlstring)
    archive = releasexml.findall("./release/repositories/archive")[0]
    tarsize = archive.attrib['size']
    tarfilename = archive.attrib['file']
    taruri = uri.replace('release.xml', tarfilename)
    f = StreamDecorator(urlopen(taruri), tarsize, notifier)
    tar = tarfile.open(fileobj=f, mode='r|gz', errorlevel=1)
    tar.extractall("/mnt/fs")
    sync()
    if not os.path.isdir("/mnt/fs/etc/release/"):
        os.mkdir("/mnt/fs/etc/release/")
    with open("/mnt/fs/etc/release/current.xml", "wb") as currentrelease:
        currentrelease.write(xmlstring)
    sync()



def run_hook(hook, name):
    print("Run hook %s" % name)
    result = True
    if hook is None:
        return True
    try:
        if name in list(hook.__dict__.keys()):
            function = hook.__dict__[name]
            result &= function()
    except Exception:
        exc_type, exc_obj, tb = sys.exc_info()
        f = tb.tb_frame
        lineno = tb.tb_lineno
        filename = f.f_code.co_filename
        linecache.checkcache(filename)
        line = linecache.getline(filename, lineno, f.f_globals)
        print('EXCEPTION IN ({}, LINE {} "{}"): {}'.format(filename, lineno, line.strip(), exc_obj))
        traceback.print_exception(exc_type, exc_obj, tb, limit=5, file=sys.stdout)
        return False
    return result

def get_uboot_env_partition():
    idx = 1
    for part in get_partition_plan():
        if part.get("uboot-env"):
            return idx
        idx = idx + 1
    return None

def install_release(releasexml, firstinstall=True, part=5, gui=None):
    import os.path
    result = False
    print("Starting software installation")
    mmc = list(get_block_devices("MMC").keys())[0]
    if firstinstall:
        print("Partitioning flash and formatting")
        create_partitions_table()
    else:
        print("Formatting flash")
        format_device(part)
    try:
        mount_target(mmc, part)
        if gui:
            progress_bar(gui, 2)
        print("Image download and installation")
        install_image(releasexml, gui=gui)
        print("Post installation steps")
        try:
            uboot_env_part = get_uboot_env_partition()
            mount_chroot()

            chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])

            if uboot_env_part != None:
                mount("/dev/%sp%i"%(mmc, uboot_env_part), "/mnt/fs/boot", "vfat")
                # make sure that uboot.env exists. otherwise fw_setenv will not work
                Path("/mnt/fs/boot/uboot.env").touch()
            if not os.path.exists("/mnt/fs/var/run/lock"):
                os.makedirs("/mnt/fs/var/run/lock")
            if not os.path.exists("/mnt/fs/boot/env.d"):
                os.makedirs("/mnt/fs/boot/env.d")
            chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])
            if os.path.exists("/mnt/fs/sbin/fw_setenv"):
                chroot_exec("/mnt/fs", ["/sbin/fw_setenv", "mmc_part", "%i"%part])
            else:
                chroot_exec("/mnt/fs", ["/usr/bin/fw_setenv", "mmc_part", "%i"%part])
            result = True
        finally:
            if uboot_env_part != None:
                umount("/mnt/fs/boot")
            umount_chroot()
    finally:
        umount_target()
    return result


def install_pmulti(uri, firstinstall=True, part=7):
    status = False
    mmc = list(get_block_devices("MMC").keys())[0]
    if firstinstall:
        create_partitions_table()
    else:
        format_device(part)
    try:
        mount("/dev/%sp%i"%(mmc, part), "/mnt/fs/", "ext4")
        sync()
        f = urlopen(uri)
        xmlstring = f.read()
        releasexml = etree.fromstring(xmlstring)
        pmulti = releasexml.findall("./release/pmulti")[0]
        pmultiname = pmulti.attrib['file']
        pmultiuri = uri.replace('release.xml', pmultiname)
        r = urlopen(pmultiuri)
        with open("/mnt/fs/pmulti.itb", "wb") as pm:
            while True:
                stream = r.read(1*1024*1024)  # Read 1MB
                if len(stream) == 0:
                    break
                pm.write(stream)
        sync()
        status = True
    finally:
        umount("/mnt/fs")
    return status
