
from .json_lite_builder import JsonLiteBuilder, JsonLite
from .specs import OltsSpec


class OltsBuilder(JsonLiteBuilder):
    def __init__(self, cdm) -> None:
        super().__init__(cdm)

        self.__root_layer = None
        self.__measurement_schema = None
        self.__meas_result_schema = None

        self.reset()

    def reset(self) -> None:
        self._json_lite = JsonLite()
        self._json_lite.format = OltsSpec.format()
        compress_format = [*OltsSpec.abridgments(), *OltsSpec.acronyms()]
        self._json_lite.set_compress_format(compress_format)

        self.__root_layer = OltsSpec.schema()
        self.__measurement_schema = self.__root_layer.pop("Measurement")
        assert self.__measurement_schema is not None

        schema_results = self.__measurement_schema.get("Results", [])
        assert len(schema_results) == 1

        self.__meas_result_schema = schema_results.pop(0)
        assert self.__meas_result_schema is not None

    def produce_root_layer(self) -> None:
        self._transform_entries(self.__root_layer)

        self._json_lite.data = self.__root_layer

    def produce_sub_layers(self) -> None:
        for test_idx in range(self._cdm.tests_number):
            indexes = (test_idx, )
            measurement = self._make_sub_layer(self.__measurement_schema, indexes)

            if "Identifiers" in measurement:
                measurement["Identifiers"] = "{}_{}".format(self._cdm.get_cable_id(test_idx),
                                                            self._cdm.get_fiber_number(test_idx))

            results = measurement.get("Results")

            for result_idx in range(self._cdm.results_number[test_idx]):
                indexes = (test_idx, result_idx)
                result = self._make_sub_layer(self.__meas_result_schema, indexes)
                results.append(result)

            self._json_lite.data["Measurement%i" % (test_idx + 1)] = measurement

    def post_process(self) -> None:
        self._json_lite.data = {
            "brief": self._json_lite.data
        }

    def _transform(self, data, accessor, entry) -> None:
        assert type(accessor) in (int, str), "unexpected accessor type: <%s>" % type(accessor)

        if (type(entry) is str) and entry.startswith("$."):  # entry is a json-path
            content = self._cdm.get(entry)

            if content is None:
                content = "NaN"

            if accessor in ("TestDate", "ReferenceDate"):
                content = content.split("T")[0]  # get only date from ISO 8601 formatted date-time string

            data[accessor] = str(content)


