
from abc import ABC, abstractmethod
from .json_lite import JsonLite


class JsonLiteBuilder(ABC):
    def __init__(self, cdm):
        self._cdm = cdm
        self._json_lite: JsonLite = JsonLite()

    @abstractmethod
    def reset(self) -> None:
        pass

    @abstractmethod
    def produce_root_layer(self) -> None:
        pass

    @abstractmethod
    def produce_sub_layers(self) -> None:
        pass

    @abstractmethod
    def post_process(self) -> None:
        pass

    @abstractmethod
    def _transform(self, data, accessor, entry):
        pass

    def _transform_entries(self, entry):
        if type(entry) in (list, dict):
            pair_data = list(entry.items() if type(entry) is dict else enumerate(entry))

            for accessor, sub_entry in pair_data:
                self._transform(entry, accessor, sub_entry)

                if type(sub_entry) in (dict, list):
                    self._transform_entries(sub_entry)

    def _make_sub_layer(self, schema, indexes) -> dict:
        sub_layer = dict()
        for key, entry in schema.items():
            if (type(entry) is str) and entry.startswith("$."):  # entry is a json-path
                content = self._cdm.get(entry.format(*indexes))

                if content is None:
                    content = "NaN"

                sub_layer[key] = str(content)

            elif entry in ([], {}):  # entry is sub-layer
                sub_layer[key] = type(entry)()  # must copy the entry or create a new instance of same type as entry

            else:  # entry is const default value
                sub_layer[key] = entry

        return sub_layer

    def get_json_lite(self) -> JsonLite:
        json_lite = self._json_lite
        self.reset()

        return json_lite
