#-*-coding:Utf-8 -*
from viavi.evm import *
from viavi.diag import *
import viavi.mts.uts_spi as utsspi
import time, os
import subprocess
#
#   reg_name             reg_addr         reg_access    value @ reset
#
#
#**********************************************************************************
# OSB board register mapping
#**********************************************************************************
#
#---------------------------------------------------------------------------------
# OSB SPI MASTER IP
#---------------------------------------------------------------------------------
#
# REG_OSB_SPI_BASE_ADDR   = 0x0C00
#
REG_OSB_SPI_SLV0        = 0x0C00
REG_OSB_SPI_SLV1        = 0x0C04
REG_OSB_SPI_CTRL        = 0x0C08
REG_OSB_SPI_TRANSMIT    = 0x0C0C
#
#---------------------------------------------------------------------------------
# OSB GLU_IP
#---------------------------------------------------------------------------------
#
REG_FPGA_VERSION        = 0x1400            # RO  (slv0)    <current_version>
REG_FPGA_SYNTH_DATE     = 0x1404            # RO  (slv1)    <current_synth_date>
REG_PSU_CTRL            = 0x1408            # R/W (slv2)    0x0FF
REG_SHIFTER_CTRL        = 0x140C            # R/W (slv3)    0x0
REG_MUX_CTRL            = 0x1410            # RO  (slv4)    Na
REG_SYS_FAN1_CTRL       = 0x1418            # R/W (slv6)    0x00090032
REG_SYS_FAN2_CTRL       = 0x141C            # R/W (slv7)    0x00090019
REG_SYS_FAN3_CTRL       = 0x1420            # R/W (slv8)    0x00090014
REG_OSB_FAN1_CTRL       = 0x1424            # R/W (slv9)    0x0013001E
REG_OSB_FAN2_CTRL       = 0x1428            # R/W (slv10)   0x00130028
REG_DBG_CTRL            = 0x1438            # R/W (slv14)
REG_MSEL                = 0x143C            # R/W (slv15)   0x0
REG_SYS_FAN1_TACHI_VAL  = 0x1440            # RO  (slv16)   Na
REG_SYS_FAN2_TACHI_VAL  = 0x1444            # RO  (slv17)   Na
REG_SYS_FAN3_TACHI_VAL  = 0x1448            # RO  (slv18)   Na
REG_OSB_FAN1_TACHI_VAL  = 0x144C            # RO  (slv19)   Na
REG_OSB_FAN2_TACHI_VAL  = 0x1450            # RO  (slv20)   Na
REG_OSB_PUSHB_VAL       = 0x1454            # RO  (slv21)   Na
REG_SYS_PSYNC_700K_MEAS = 0x1458            # RO  (slv22)   Na
REG_SYS_PSYNC_640K_MEAS = 0x145C            # RO  (slv23)   Na
REG_SYS_PSYNC_320K_MEAS = 0x1460            # RO  (slv24)   Na
REG_CPLD_33M_FREQ_MEAS  = 0x1464            # RO  (slv25)   Na
REG_DTSS_FAN_TACHI_VAL  = 0x1468            # RO  (slv26)   Na

REG_MOD_FAN_RDBCK       = 0x1474            # RO  (slv29)   Na
REG_MOD_PRESENCE_RDBCK  = 0x1478            # RO  (slv30)   Na
REG_PERIPH_RST_CTRL     = 0x147C            # R/W (slv31)   0xFFFFFFFF
#
# URJTAG_IP
#
URJTAG_BASE_ADDR        = 0x1800
REG_URJTAG_FREQ         = URJTAG_BASE_ADDR + 0x0
REG_URJTAG_IO           = URJTAG_BASE_ADDR + 0x4
REG_URJTAG_STATUS       = URJTAG_BASE_ADDR + 0x8
REG_URJTAG_DUMMY        = URJTAG_BASE_ADDR + 0xC
#
# UIC IP (User Intyerrupts Controler)
#
UIC_BASE_ADDR       = 0x2800
REG_IT_STATUS       = UIC_BASE_ADDR             # RO ; value@rst = 0x0
REG_IT_CLEAR        = UIC_BASE_ADDR + 0x4       # RW ; value@rst = 0xffffffff
REG_IT_MASK         = UIC_BASE_ADDR + 0x8       # RW ; value@rst = 0xffffffff
REG_IT_TYPE         = UIC_BASE_ADDR + 0xC       # RO ; value@rst = 0x0
#
# UART IP
#
UART_MOD_BASE_ADDR      = 0x10000
UART_RS485_BASE_ADDR    = 0x19000

UART_RBR_DLL            = 0x01000
UART_DLM                = 0x01004
UART_FCR                = 0x01008
UART_LCR                = 0x0100C
UART_MCR                = 0x01010
UART_MSR                = 0x01018
#
#---------------------------------------------------------------------------------
# OSB SEQUENCER_CORE IP
#---------------------------------------------------------------------------------
#
SEQ_CORE_BASE_ADDR           = 0x0

REG_BDT_SEQ_SLV1             = SEQ_CORE_BASE_ADDR + 0x4   # slv1      DEBUG ONLY 
REG_BDT_SEQ_MISC             = SEQ_CORE_BASE_ADDR + 0x48  # slv18 (RW)

REG_BDT_SEQ_MMCM_CLK_IN_FREQ = SEQ_CORE_BASE_ADDR + 0x64  # slv25 (RO)
REG_BDT_SEQ_PLAS_DIAG        = SEQ_CORE_BASE_ADDR + 0x68  # slv26 (RW)
REG_BDT_SEQ_LVDS_FREQ        = SEQ_CORE_BASE_ADDR + 0x6C  # slv27 (RO)

REG_BDT_SEQ_VERSION          = SEQ_CORE_BASE_ADDR + 0x7C  # slv31
#
#---------------------------------------------------------------------------------
AXIS_BASE_ADDR           = 0x300
#---------------------------------------------------------------------------------
#
REG_BDT_IP_VERSION       = AXIS_BASE_ADDR + 0    # RO (slv0)     : 
REG_BDT_CTRL             = AXIS_BASE_ADDR + 4    # RW (slv1)
REG_BDT_FIFO_RD_CNT      = AXIS_BASE_ADDR + 8    # RO (slv2) : no more used
REG_BDT_FIFO_OUT         = AXIS_BASE_ADDR + 0xC  # RO (slv3) : no more used

REG_BDT_DBG_CTRL         = AXIS_BASE_ADDR + 0x14    # slv_reg5 : RW : no more used
REG_BDT_DBG_DATA         = AXIS_BASE_ADDR + 0x18    # slv_reg6 : RW : no more used
REG_BDT_DBG_STATUS       = AXIS_BASE_ADDR + 0x1C    # slv_reg7 : RW : no more used
REG_AXIS_CLK_FREQ_MEAS   = AXIS_BASE_ADDR + 0x20    # slv_reg8 : RO

REG_DMA_NUMBER_OF_POINTS = AXIS_BASE_ADDR + 0x24    # slv_reg9   : RW : Nombre de points attendus pour l'acquisition en cours
REG_DMA_DBG_CTRL         = AXIS_BASE_ADDR + 0x28    # slv_reg10 : RW
REG_DMA_DBG_STATUS       = AXIS_BASE_ADDR + 0x2C    # slv_reg11 : RO
REG_DMA_DBG_DATA         = AXIS_BASE_ADDR + 0x30    # slv_reg12 : RW : for debug purpose ONLY
#
#---------------------------------------------------------------------------------
# OSB XADC IP
#---------------------------------------------------------------------------------
#
XADC_BASE_ADDR          = 0x2000
#
# XADC Status Registers (Read Only) :
#"""""""""""""""""""""""""""""""""""""
#
XADC_TEMP_REG           = XADC_BASE_ADDR + 0x200    # RO
#
XADC_VAUX_0_REG         = XADC_BASE_ADDR + 0x240     # RO
XADC_VAUX_1_REG         = XADC_BASE_ADDR + 0x244     # RO
XADC_VAUX_2_REG         = XADC_BASE_ADDR + 0x248     # RO
XADC_VAUX_3_REG         = XADC_BASE_ADDR + 0x24C     # RO
#XADC_VAUX_4_REG         = XADC_BASE_ADDR + 0x250    # RO : not available in Artix7
#XADC_VAUX_5_REG         = XADC_BASE_ADDR + 0x254    # RO : not available in Artix7
#XADC_VAUX_6_REG         = XADC_BASE_ADDR + 0x258    # RO : not available in Artix7
#XADC_VAUX_7_REG         = XADC_BASE_ADDR + 0x25C    # RO : not available in Artix7
XADC_VAUX_8_REG         = XADC_BASE_ADDR + 0x260     # RO
XADC_VAUX_9_REG         = XADC_BASE_ADDR + 0x264     # RO
XADC_VAUX_10_REG        = XADC_BASE_ADDR + 0x268     # RO
XADC_VAUX_11_REG        = XADC_BASE_ADDR + 0x26C     # RO
#XADC_VAUX_12_REG        = XADC_BASE_ADDR + 0x270    # RO : not available in Artix7
#XADC_VAUX_13_REG        = XADC_BASE_ADDR + 0x274    # RO : not available in Artix7
#XADC_VAUX_14_REG        = XADC_BASE_ADDR + 0x278    # RO : not available in Artix7
#XADC_VAUX_15_REG        = XADC_BASE_ADDR + 0x27C    # RO : not available in Artix7
#
#**********************************************************************************
# B200 Tool board registers mapping
#**********************************************************************************
#
SPI_ADDR_VERSION        = 0x0   # RO
SPI_ADDR_REGLVDS_7_0    = 0x1   # R/W
SPI_ADDR_REGLVDS_15_8   = 0x2   # R/W
SPI_ADDR_REGLVDS_20_16  = 0x3   # R/W
SPI_ADDR_MISC           = 0x4   # R/W
SPI_ADDR_SYNC_MEAS1_LSB = 0x5   # RO
SPI_ADDR_SYNC_MEAS1_MSB = 0x6   # RO
SPI_ADDR_SYNC_MEAS2_LSB = 0x7   # RO
SPI_ADDR_SYNC_MEAS2_MSB = 0x8   # RO
SPI_ADDR_SYNC_MEAS3_LSB = 0x9   # RO
SPI_ADDR_SYNC_MEAS3_MSB = 0xa   # RO
SPI_ADDR_SYNC_MEAS4_LSB = 0xb   # RO
SPI_ADDR_SYNC_MEAS4_MSB = 0xc   # RO
SPI_ADDR_SYNC_MEAS5_LSB = 0xd   # RO
SPI_ADDR_SYNC_MEAS5_MSB = 0xe   # RO
SPI_ADDR_SYNC_MEAS6_LSB = 0xf   # RO
SPI_ADDR_SYNC_MEAS6_MSB = 0x10  # RO

SPI_ADDR_BOARD_PRES     = 0x11 # R/W
SPI_ADDR_UART           = 0x12 # R/W
SPI_ADDR_DTSS_REG       = 0x13 # R/W
SPI_ADDR_MSEL_REG       = 0x14 # RO
SPI_ADDR_EXPCB          = 0x15 # RO
SPI_ADDR_CPLD33_REG     = 0x16 # RO

SPI_ADDR_SYNC_MEAS7_LSB = 0x17  # RO
SPI_ADDR_SYNC_MEAS7_MSB = 0x18  # RO

SPI_ADDR_CCAN_PLAS      = 0x19  # RO

SPI_ADDR_SYNC_MEAS8_LSB = 0x1a  # RO
SPI_ADDR_SYNC_MEAS8_MSB = 0x1b  # RO

#**********************************************************************************
# DTSS Module board registers mapping
#**********************************************************************************
#
# DTSS FPGA SPI register mapping
#
DTSS_SPI_XVERSION       = 0x7F
#
# DTSS BDT SPI register mapping
#
DTSS_SPI_TNUMBER        = 0x20
DTSS_SPI_NDURATION      = 0x23
DTSS_SPI_ANUMBER        = 0x24
DTSS_SPI_ARMAX          = 0x25
DTSS_SPI_ARINC          = 0x26
DTSS_SPI_CNUMBER        = 0x27
DTSS_SPI_PULSE          = 0x2A
DTSS_SPI_CRNUMBER       = 0x2B
DTSS_SPI_MISC           = 0x30
DTSS_SPI_STATUS         = 0x31
DTSS_SPI_OFFSET_TM      = 0x34
DTSS_SPI_OFFSET_TE      = 0x35
#
# DTSS HW SPI register mapping
#
DTSS_SPI_ON_ALIM        = 0x41
DTSS_SPI_LAS_FURU_SL    = 0x42
DTSS_SPI_LAS_FURU_OL    = 0x43
DTSS_SPI_AMPLI_RF       = 0x44
DTSS_SPI_RS232          = 0x48
DTSS_SPI_DAC_SL         = 0x49
DTSS_SPI_DAC_OL         = 0x4A
DTSS_SPI_FREQUENCE      = 0x4B
DTSS_SPI_FREQ_WIN       = 0x4C
DTSS_SPI_OPTICAL_SW     = 0x4D
DTSS_SPI_DATA_LMK       = 0x4E
DTSS_SPI_FAN_PWM_DUTY   = 0x4F 
DTSS_SPI_FREQUENCE_READ = 0x50
DTSS_SPI_FAN_PWM_DIV    = 0x51
DTSS_SPI_PT100_CMDE     = 0x52
DTSS_SPI_PT100_INT      = 0x53
DTSS_SPI_PT100_EXT      = 0x54
DTSS_SPI_PT100_REG      = 0x55
DTSS_SPI_FREQ_REGUL     = 0x56
DTSS_SPI_FREQ_REGUL_WIN = 0x59
#
# DTSS DMA SPI register mapping
#
DTSS_SPI_DMA_DBG        = 0x5a  # For DEBUG Only

DTSS_bdt_reg_name_list = ["DTSS_TNUMBER", "DTSS_NDURATION", "DTSS_ANUMBER", "DTSS_ARMAX",
                          "DTSS_ARINC", "DTSS_CNUMBER", "DTSS_PULSE", "DTSS_CRNUMBER",
                          "DTSS_MISC", "DTSS_STATUS", "DTSS_OFFSET_TM", "DTSS_OFFSET_TE"]

DTSS_bdt_reg_addr_list = [DTSS_SPI_TNUMBER, DTSS_SPI_NDURATION, DTSS_SPI_ANUMBER, DTSS_SPI_ARMAX,
                          DTSS_SPI_ARINC, DTSS_SPI_CNUMBER, DTSS_SPI_PULSE, DTSS_SPI_CRNUMBER,
                          DTSS_SPI_MISC, DTSS_SPI_STATUS, DTSS_SPI_OFFSET_TM, DTSS_SPI_OFFSET_TE]

DTSS_bdt_reg_range_list = ['(24:0)', '(16:0)', '(17:0)', '(7:0)',
                           '(3:0)', '(2:0)', '(31:0)', '(7:0)',
                           '(23:0)', '(7:0)', '(11:0)', '(11:0)']

DTSS_hw_reg_list   = ['DTSS_SPI_ON_ALIM', 'DTSS_SPI_LAS_FURU_SL', 'DTSS_SPI_LAS_FURU_OL', 'DTSS_SPI_AMPLI_RF',
                      'DTSS_SPI_RS232', 'DTSS_SPI_DAC_SL', 'DTSS_SPI_DAC_OL', 'DTSS_SPI_FREQUENCE',
                      'DTSS_SPI_FREQ_WIN', 'DTSS_SPI_OPTICAL_SW', 'DTSS_SPI_DATA_LMK', 'DTSS_SPI_FAN_PWM_DUTY',
                      'DTSS_SPI_FREQUENCE_READ', 'DTSS_SPI_FAN_PWM_DIV']

DTSS_hw_reg_addr_list = [DTSS_SPI_ON_ALIM, DTSS_SPI_LAS_FURU_SL, DTSS_SPI_LAS_FURU_OL, DTSS_SPI_AMPLI_RF,
                         DTSS_SPI_RS232, DTSS_SPI_DAC_SL, DTSS_SPI_DAC_OL, DTSS_SPI_FREQUENCE,
                         DTSS_SPI_FREQ_WIN, DTSS_SPI_OPTICAL_SW, DTSS_SPI_DATA_LMK, DTSS_SPI_FAN_PWM_DUTY,
                         DTSS_SPI_FREQUENCE_READ, DTSS_SPI_FAN_PWM_DIV]

DTSS_hw_reg_range_list = ['(24:0)', '(24:0)', '(24:0)', '(24:0)',
                          '(24:0)', '(24:0)', '(24:0)', '(24:0)',
                          '(24:0)', '(24:0)', '(24:0)', '(24:0)',
                          '(24:0)', '(24:0)']
#
#**********************************************************************************
# Z112 module board registers mapping
#**********************************************************************************
#
Z112_SPI_XVERSION        = 0x0
Z112_ADDR_REGLVDS_7_0    = 0x1   # R/W
Z112_ADDR_REGLVDS_15_8   = 0x2   # R/W
Z112_ADDR_REGLVDS_20_16  = 0x3   # R/W
#
#**********************************************************************************
# TACB board registers mapping
#**********************************************************************************
#
#---------------------------------------------------------------------------------
# TACB GLU_IP (BAR1_addr)
#---------------------------------------------------------------------------------
#
TACB_GLU_IP_BASE_ADDR       = 0xA000
#
REG_TACB_GLU_IP_VERSION     = TACB_GLU_IP_BASE_ADDR + 0x0
REG_TACB_GLU_SYNTH_DATE     = TACB_GLU_IP_BASE_ADDR + 0x4
REG_TACB_GLU_MTS_PS         = TACB_GLU_IP_BASE_ADDR + 0x8
REG_TACB_GLU_LED            = TACB_GLU_IP_BASE_ADDR + 0xC
REG_TACB_GLU_MSEL           = TACB_GLU_IP_BASE_ADDR + 0x10
REG_TACB_MTS_PWR_DIAG       = TACB_GLU_IP_BASE_ADDR + 0x7C

tacb_glu_reg_name_list      = ['TACB_GLU_IP_VERSION', 'TACB_GLU_SYNTH_DATE', 'TACB_GLU_MTS_PS',
                               'TACB_GLU_LED', 'TACB_GLU_MSEL', 'TACB_MTS_PWR_DIAG']

tacb_glu_reg_addr_list      = [REG_TACB_GLU_IP_VERSION, REG_TACB_GLU_SYNTH_DATE, REG_TACB_GLU_MTS_PS,
                               REG_TACB_GLU_LED, REG_TACB_GLU_MSEL, REG_TACB_MTS_PWR_DIAG]

tacb_glu_reg_range_list     = ['(31:0)', '(31:0)', '(31:0)', '(31:0)', '(31:0)', '(31:0)']

#
#---------------------------------------------------------------------------------
# TACB BDT (BAR3_addr)
#---------------------------------------------------------------------------------
#
TACB_SEQ_CORE_IP_BASE_ADDR   = 0x0
#
REG_TACB_TEETH_NUMBER        = TACB_SEQ_CORE_IP_BASE_ADDR + 0x0
REG_TACB_XOFFSET_DURATION    = TACB_SEQ_CORE_IP_BASE_ADDR + 0x4
REG_TACB_MASK_DURATION       = TACB_SEQ_CORE_IP_BASE_ADDR + 0x8
REG_TACB_WAIT_DURATION       = TACB_SEQ_CORE_IP_BASE_ADDR + 0xC
REG_TACB_NOISE_DURATION      = TACB_SEQ_CORE_IP_BASE_ADDR + 0x10
REG_TACB_AVG_NUMBER          = TACB_SEQ_CORE_IP_BASE_ADDR + 0x14
REG_TACB_RAMP_INCR           = TACB_SEQ_CORE_IP_BASE_ADDR + 0x18
REG_TACB_RAMP_RST_VAL        = TACB_SEQ_CORE_IP_BASE_ADDR + 0x1C
REG_TACB_COMB_NUMBER         = TACB_SEQ_CORE_IP_BASE_ADDR + 0x20
REG_TACB_COMB_INCR           = TACB_SEQ_CORE_IP_BASE_ADDR + 0x24

REG_TACB_MISC                = TACB_SEQ_CORE_IP_BASE_ADDR + 0x48
REG_TACB_HW_CFG              = TACB_SEQ_CORE_IP_BASE_ADDR + 0x4C
REG_TACB_TOT_TEETH_NB_CALC   = TACB_SEQ_CORE_IP_BASE_ADDR + 0x50 # RO
REG_TACB_FORCE_COMB_ID       = TACB_SEQ_CORE_IP_BASE_ADDR + 0x54
REG_TACB_RECUR_PRIME_NB      = TACB_SEQ_CORE_IP_BASE_ADDR + 0x60
REG_TACB_FSM_STATE_DBG       = TACB_SEQ_CORE_IP_BASE_ADDR + 0x64
REG_TACB_SPI_STEP_VAL        = TACB_SEQ_CORE_IP_BASE_ADDR + 0x68

REG_TACB_SEQ_CORE_IP_VERSION = TACB_SEQ_CORE_IP_BASE_ADDR + 0x7C

TACB_ACCU_CORE_IP_BASE_ADDR     = 0x100
#
REG_TACB_ACCU_AXI_ADDR_OFFSET   = TACB_ACCU_CORE_IP_BASE_ADDR
REG_TACB_ACCU_CRUNCH_NB         = TACB_ACCU_CORE_IP_BASE_ADDR + 0x4
REG_TACB_ACCU_OFFSET_CONT       = TACB_ACCU_CORE_IP_BASE_ADDR + 0x8
REG_TACB_ACCU_STATE_MEM_DBG     = TACB_ACCU_CORE_IP_BASE_ADDR + 0x10

TACB_LASER_CTRL_BASE_ADDR       = 0x200
#
REG_TACB_LAS_WIDTH_ENT          = TACB_LASER_CTRL_BASE_ADDR
REG_TACB_LAS_WIDTH_DEC          = TACB_LASER_CTRL_BASE_ADDR + 0x4
REG_TACB_LAS_EN                 = TACB_LASER_CTRL_BASE_ADDR + 0x8
REG_TACB_LAS_SRC_EN             = TACB_LASER_CTRL_BASE_ADDR + 0xC
REG_TACB_LAS_SRC_FREQ_MOD       = TACB_LASER_CTRL_BASE_ADDR + 0x10
REG_TACB_LAS_SRC_FREQ_FAST_MOD  = TACB_LASER_CTRL_BASE_ADDR + 0x14

TACB_AXIS_BASE_ADDR          = 0x300
#

tacb_seq_core_reg_name_list = ['TACB_TEETH_NUMBER', 'TACB_XOFFSET_DURATION', 'TACB_MASK_DURATION', 'TACB_WAIT_DURATION',
                               'TACB_NOISE_DURATION', 'TACB_AVG_NUMBER', 'TACB_RAMP_INCR', 'TACB_RAMP_RST_VAL', 
                               'TACB_COMB_NUMBER', 'TACB_COMB_INCR', 'TACB_MISC', 'TACB_HW_CFG',
                               'TACB_TOT_TEETH_NB_CALC', 'TACB_FORCE_COMB_ID', 'TACB_RECUR_PRIME_NB', 'TACB_FSM_STATE_DBG',
                               'TACB_SPI_STEP_VAL', 'TACB_SEQ_CORE_IP_VERSION']

tacb_seq_core_reg_addr_list = [REG_TACB_TEETH_NUMBER, REG_TACB_XOFFSET_DURATION, REG_TACB_MASK_DURATION, REG_TACB_WAIT_DURATION,
                               REG_TACB_NOISE_DURATION, REG_TACB_AVG_NUMBER, REG_TACB_RAMP_INCR, REG_TACB_RAMP_RST_VAL, 
                               REG_TACB_COMB_NUMBER, REG_TACB_COMB_INCR, REG_TACB_MISC, REG_TACB_HW_CFG,
                               REG_TACB_TOT_TEETH_NB_CALC, REG_TACB_FORCE_COMB_ID, REG_TACB_RECUR_PRIME_NB, REG_TACB_FSM_STATE_DBG,
                               REG_TACB_SPI_STEP_VAL, REG_TACB_SEQ_CORE_IP_VERSION]

tacb_seq_core_reg_range_list = ['(17:0)', '(9:0)', '(17:0)', '(17:0)',
                                '(13:0)', '(15:0)', '(3:0)', '(7:0)',
                                '(8:0)', '(7:0)', '(15:0)', '(23:0)',
                                '(17:0)', '(9:0)', '(17:0)','(?:?)',
                                '(7:0)', '(31:0)']
#
#**********************************************************************************
# Constants definition
#**********************************************************************************
#
DTSS_SLICE             = 4
Z112_SLICE             = 3
B200_SLICE             = 3

B200_DTSS_MOD          = 1
B200_SWITCH_MOD        = 2

cRead                  = 1
cWrite                 = 0

#
#**********************************************************************************
# ****************** OSB functions definition *************************************
#**********************************************************************************
#
#************************
def get_glu_base_addr():
#************************
#
    debug = False

    FileName = "/usr/share/pydiag/osb_glu_base_addr_tmp.txt"

    os_base_cmd = "lspci -s 06:00.0 -v | grep size=128K"
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()

    if debug:
        print("File %s: line is %s" %(FileName, line))

    tmp_list = line.split()

    if debug == True:
        print("tmp_list is" , tmp_list)

    base_addr = tmp_list[2]

    base_address = int(base_addr, base=16)

    if debug == True:
        print("base_address is 0x%x" %base_address)
        print("base_address is d'%d\n" %base_address)

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return base_address
#
#****************************
def get_pci_bdt_base_addr():
#****************************
#
    debug = False

    # OSB is a 1 functions pcie :
    #
    # 0000:06:00.0 Communication controller: Device 1d7d:2104
    # OSB is a 1 functions pcie :
    #
    # Memory at 18720000 (32-bit, non-prefetchable) [size=64K]  <- bar0 
    # Memory at 18700000 (32-bit, non-prefetchable) [size=128K] <- bar1 <- glu_ip
    # Memory at 18730000 (32-bit, non-prefetchable) [size=64K]  <- bar2 <- unused
    # Memory at 18740000 (32-bit, non-prefetchable) [size=64K]  <- bar3 <- bdt_seq_core_ip
    # Kernel modules: pci_glu2
    #
    FileName = "/usr/share/pydiag/osb_bdt_base_addr_tmp.txt"

    os_base_cmd = "lspci -s 06:00.0 -v | grep size=64K"
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()

    if debug:
        print("File %s: line is \n %s" %(FileName, line))

    tmp_list = line.split()

    if debug == True:
        print("tmp_list is" , tmp_list)

    base_addr = tmp_list[14]

    base_address = int(base_addr, base=16)

    if debug == True:
        print("base_address is 0x%x (d'%d)" %(base_address, base_address))

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return base_address
#
#****************************
def get_reg_value(reg_addr):
#****************************

    debug = False

    var = str(reg_addr)
    #FileName = "/usr/share/pydiag/oth/"+var+"_tmp.txt"
    FileName = "/usr/share/pydiag/"+var+"_tmp.txt"

    os_base_cmd = "mem 0x%x" %reg_addr+" w"
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_base_cmd is", os_base_cmd)
        print("os_cmde is ", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()

    tmp_list = line.split()

    if debug == True:
        print("File %s: line is %s" %(FileName, line))
        print("tmp_list is" , tmp_list)

    reg_val = tmp_list[1]

    register_value = int(reg_val, base=16)

    if debug == True:
        print("register_value is 0x%x %d" %register_value)
        print("register_value is d'%d\n" %register_value)

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return register_value

#*************************************
def reg_write(reg_addr, data_value):
#*************************************

    debug = False

    var = "%x" %reg_addr

    #FileName = "/usr/share/pydiag/oth/"+var+"_tmp.txt"
    FileName = "/usr/share/pydiag/"+var+"_tmp.txt"

    if debug == True:
        print("reg_addr is 0x%x ; %d" %(reg_addr, reg_addr))
        print("data_value is 0x%x ; %d" %(data_value, data_value))

    os_base_cmd = "mem 0x%x" %reg_addr+" w "+"0x%x" %data_value
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))
    os_cmde = "rm -f "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    return 0
#
#********************************************
def periph_reset_gen(flag, verbose_flag):
#
# OSB FPGA has a PERIPH_HW_RSTn output going
# to SWITCH and DTSS module
#********************************************
#
    glu_reg_base_addr = get_glu_base_addr()

    if verbose_flag:
        print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PERIPH_RST_CTRL)
    if verbose_flag:
        print("reset_ctrl_value = 0x%x" % reset_ctrl_value)

    data_val =  0x0
    reg_write(glu_reg_base_addr + REG_PERIPH_RST_CTRL, data_val)

    if verbose_flag:
        print ("\n     * Enable reset")
    reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PERIPH_RST_CTRL)

    if verbose_flag:
        print("         => reset_ctrl_value = 0x%x" % reset_ctrl_value)

    if flag == "release":
        if verbose_flag:
            print("     * Wait for 0.1s")
        time.sleep(0.1)

        if verbose_flag:
            print("     * Disable reset")
        data_val =  0x1
        reg_write(glu_reg_base_addr + REG_PERIPH_RST_CTRL, data_val)

        reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PERIPH_RST_CTRL)

        if verbose_flag:
            print("         => reset_ctrl_value = 0x%x" % reset_ctrl_value)

    return True
#
#****************************************
def show_uic_reg_contents(reg_type):
#****************************************
#
    glu_reg_base_addr = get_glu_base_addr()

    if reg_type == 'ALL' or reg_type == 'IT_TYPE':
        it_type_reg = get_reg_value(glu_reg_base_addr + REG_IT_TYPE)

        print("\n* REG_IT_TYPE (RO) read value = 0x%x" %it_type_reg)
        print("  Note : '0' for IT on falling edge ; '1' for IT on Low level")
        
        print("     > REG_IT_TYPE(0)  = PERIPH_UART_IRQ  = %d" %(it_type_reg & 0x1))
        print("     > REG_IT_TYPE(1)  = BINDER_UART_IRQ  = %d" %((it_type_reg & 0x2) >> 1))
        print("     > REG_IT_TYPE(2)  = PUSHB_IRQ        = %d" %((it_type_reg & 0x4) >> 2))
        print("     > REG_IT_TYPE(3)  = DBG_GPIO_IRQ     = %d" %((it_type_reg & 0x8) >> 3))
        print("     > REG_IT_TYPE(4)  = DTSS_IRQ         = %d" %((it_type_reg & 0x10) >> 4))
        
    if reg_type == 'ALL' or reg_type == 'IT_STATUS':

        it_status_reg = get_reg_value(glu_reg_base_addr + REG_IT_STATUS)
        print("\n* REG_IT_STATUS (RO) read value = 0x%x" %it_status_reg)
        print("  Note : '0' for it not detected ; '1' for it detected")

        print("     > REG_IT_STATUS(0)  = PERIPH_UART_IRQ  = %d" %(it_status_reg & 0x1))
        print("     > REG_IT_STATUS(1)  = BINDER_UART_IRQ  = %d" %((it_status_reg & 0x2) >> 1))
        print("     > REG_IT_STATUS(2)  = PUSHB_IRQ        = %d" %((it_status_reg & 0x4) >> 2))
        print("     > REG_IT_STATUS(3)  = DBG_GPIO_IRQ     = %d" %((it_status_reg & 0x8) >> 3))
        print("     > REG_IT_STATUS(4)  = DTSS_IRQ         = %d" %((it_status_reg & 0x10) >> 4))
 
    if reg_type == 'ALL' or reg_type == 'IT_CLEAR':
        it_clear_reg = get_reg_value(glu_reg_base_addr + REG_IT_CLEAR)
        print("\n* REG_IT_CLEAR (R/W) read value = 0x%x" %it_clear_reg)
        print("  Note : '1' : clear bit ; '0' no action")

        print("     > REG_IT_CLEAR(0)  = PERIPH_UART_IRQ  = %d" %(it_clear_reg & 0x1))
        print("     > REG_IT_CLEAR(1)  = BINDER_UART_IRQ  = %d" %((it_clear_reg & 0x2) >> 1))
        print("     > REG_IT_CLEAR(2)  = PUSHB_IRQ        = %d" %((it_clear_reg & 0x4) >> 2))
        print("     > REG_IT_CLEAR(3)  = DBG_GPIO_IRQ     = %d" %((it_clear_reg & 0x8) >> 3))
        print("     > REG_IT_CLEAR(4)  = DTSS_IRQ         = %d" %((it_clear_reg & 0x10) >> 4))

    if reg_type == 'ALL' or reg_type == 'IT_MASK':
        it_mask_reg = get_reg_value(glu_reg_base_addr + REG_IT_MASK)
        print("\n* REG_IT_MASK (R/W) read value = 0x%x" %it_mask_reg)
        print("  Note : '1' : bit is masked ; '0' no action")

        print("     > REG_IT_MASK(0)  = PERIPH_UART_IRQ  = %d" %(it_mask_reg & 0x1))
        print("     > REG_IT_MASK(1)  = BINDER_UART_IRQ  = %d" %((it_mask_reg & 0x2) >> 1))
        print("     > REG_IT_MASK(2)  = PUSHB_IRQ        = %d" %((it_mask_reg & 0x4) >> 2))
        print("     > REG_IT_MASK(3)  = DBG_GPIO_IRQ     = %d" %((it_mask_reg & 0x8) >> 3))
        print("     > REG_IT_MASK(4)  = DTSS_IRQ         = %d" %((it_mask_reg & 0x10) >> 4))

    return None
#
#=============================================================================================
def show_irq_dbg_ctrl_contents():
#=============================================================================================
#
    glu_reg_base_addr = get_glu_base_addr()
    
    dbg_reg_value = get_reg_value(glu_reg_base_addr + REG_DBG_CTRL)
    print("\n OSB board : Read REG_DBG_CTRL reg. value @ addr 0x%1x = 0x%x" %(REG_DBG_CTRL, dbg_reg_value))

    bit_0_write_bit0  = dbg_reg_value & 0x1
    Wrdbgdetect_bit1  = (dbg_reg_value & 0x2) >> 1
    nWrdbgdetect_bit2 = (dbg_reg_value & 0x4) >> 2
    dbg_gpio_i_3_bit3 = (dbg_reg_value & 0x8) >> 3
    
    dbg_rst_bit15           = (dbg_reg_value & 0x8000) >> 15
    bit_0_write_mask_bit16  = (dbg_reg_value & 0x10000) >> 16
    Wrdbgdetect_mask_bit17  = (dbg_reg_value & 0x20000) >> 17
    nWrdbgdetect_mask_bit18 = (dbg_reg_value & 0x40000) >> 18
    dbg_gpio_i_3_mask_bit19 = (dbg_reg_value & 0x80000) >> 19

    # RO bits
    gpio_IRQ_o_bit27  = (dbg_reg_value & 0x8000000) >> 27
    PUSHB_IRQ_bit28   = (dbg_reg_value & 0x10000000) >> 28
    dtss_IRQ_bit29    = (dbg_reg_value & 0x20000000) >> 29
    switch_IRQ_bit30  = (dbg_reg_value & 0x40000000) >> 30
    ADP5050_IRQ_bit31 = (dbg_reg_value & 0x80000000) >> 31
    
    print("         - bit_0_write (bit0)           = %d" %bit_0_write_bit0)
    print("         - Wr_fct_positive_pulse (bit1) = %d" %Wrdbgdetect_bit1)
    print("         - Wr_fct_negative_pulse (bit2) = %d" %nWrdbgdetect_bit2)
    print("         - dbg_gpio_i_3 (bit3)          = %d" %dbg_gpio_i_3_bit3)

    print("\n         - dbg_rst_bit15 (active high)  = %d" %dbg_rst_bit15)

    print("\n         - bit_0_write mask (bit16)           = %d" %bit_0_write_mask_bit16)
    print("         - Wr_fct_positive_pulse mask (bit17) = %d" %Wrdbgdetect_mask_bit17)
    print("         - Wr_fct_negative_pulse mask (bit18) = %d" %nWrdbgdetect_mask_bit18)
    print("         - dbg_gpio_i_3 mask (bit19)          = %d" %dbg_gpio_i_3_mask_bit19)
    
    print("\n         - gpio_IRQ_o (bit27)  = %d" %gpio_IRQ_o_bit27)

    print("\n         - PUSHB_IRQ (bit28)   = %d" %PUSHB_IRQ_bit28)
    print("         - dtss_IRQ  (bit29)   = %d" %dtss_IRQ_bit29)
    print("         - switch_IRQ (bit30)  = %d" %switch_IRQ_bit30)
    print("         - ADP5050_IRQ (bit31) = %d" %ADP5050_IRQ_bit31)
    
    return None

#*********************************************
def show_bdt_ctrl_reg_contents():
#*********************************************

    bdt_reg_base_addr = get_pci_bdt_base_addr()

    bdt_ctrl_value = get_reg_value(bdt_reg_base_addr + REG_BDT_CTRL)

    status_reg = (bdt_ctrl_value & 0xf0000000) >> 28
    ctrl_reg   = (bdt_ctrl_value & 0x3)
    
    fifo_reset_bit0  = bdt_ctrl_value & 0x1
    OsanotOtdr_bit1  = (bdt_ctrl_value & 0x2) >> 1

    deser_enable_bit28     = status_reg & 0x1
    mmcm_locked_bit29      = (status_reg & 0x2) >> 1
    mmcm_resetn_bit30      = (status_reg & 0x4) >> 2
    Axis_tready_rdbk_bit31 = (status_reg & 0x8) >> 3

    if fifo_reset_bit0:
        fifo_rst_pin = 0
    else:
        fifo_rst_pin = 1

    print("\nLVDS_TO AXIS IP : REG_BDT_CTRL reg. readback = 0x%x (@ addr 0x%x)" %(bdt_ctrl_value, REG_BDT_CTRL))
    print("     * fifo_reset  (bit0)  = %d : fifo reset pin = %d (active high reset)" %(fifo_reset_bit0,\
                  fifo_rst_pin))
    print("     * OsanotOtdr  (bit1)  = %d" %OsanotOtdr_bit1)
    print("     * LVDS_deserializer_deser_enable (bit28) = %d" %deser_enable_bit28)
    print("     * LVDS_deserializer_mmcm_locked  (bit29) = %d" %mmcm_locked_bit29)
    print("     * LVDS_deserializer_mmcm_resetn  (bit30) = %d" %mmcm_resetn_bit30)
    print("     * M_AXIS_TREADY_RDBK             (bit31) = %d" %Axis_tready_rdbk_bit31)

    return None

#***************************************************************************************
def show_seq_core_misc_reg_contents():
#
# REG_BDT_SEQ_MISC readback = dcm_speed_reg(1:0) & dcm_reconfig_reg &
#                             status_reg(7:0) & misc(15:0)
#
# status_reg(0) = DESER_RDY_RDBCK
# status_reg(1) = LVDS_CCAN_LOCKED
# status_reg(2) = CLK_25M_LOCKED
# status_reg(3) = lvds_clk_running
# RESET_PLL     = REG_BDT_SEQ_MISC(2) in write access
#**************************************************************************************

    bdt_reg_base_addr = get_pci_bdt_base_addr()

    pci_bdt_seq_core_misc  = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_MISC)
    reset_pll_bit2         = (pci_bdt_seq_core_misc & 0x4) >> 2
    
    deser_rdy_bit16        = (pci_bdt_seq_core_misc & 0x10000) >> 16
    lvds_ccan_locked_bit17 = (pci_bdt_seq_core_misc & 0x20000) >> 17                       
    clk_25M_locked         = (pci_bdt_seq_core_misc & 0x40000) >> 18
    lvds_clk_running       = (pci_bdt_seq_core_misc & 0x80000) >> 19
    lvds_cnter             = (pci_bdt_seq_core_misc & 0xf00000) >> 20
    dcm_rcfg_bit29         = (pci_bdt_seq_core_misc & 0x20000000) >> 29
    dcm_speed_bit31_30     = (pci_bdt_seq_core_misc & 0xd0000000) >> 30

    if reset_pll_bit2:
        msg_0 = "deserializer : mmcm in reset state, fsm in iddle state"
        msg_1 = "deser_en = 0"
    else:
        msg_0 = "deserializer : mmcm in running state, fsm in running state"
        msg_1 = "deser_en = 1"
    
    print("\nSEQ_CORE IP : SEQ_CORE_MISC readback : 0x%x" % pci_bdt_seq_core_misc)
    print("     * reset_pll (bit2)         = %d : %s" %(reset_pll_bit2, msg_0))
    print("                                    : %s" %msg_1)
    
    print("     * deser_rdy (bit16)        = %d" %deser_rdy_bit16)
    print("     * lvds_ccan_locked (bit17) = %d" %lvds_ccan_locked_bit17)
    print("     * clk_25M_locked (bit18)   = %d" %clk_25M_locked)
    print("     * lvds_clk_running (bit19) = %d" %lvds_clk_running)
    print("     * lvds_cnter               = 0x%x" %lvds_cnter)
    
    print("     * bit(28:18) unused        = 0x0")
    print("     * dcm_rcfg  (bit29)        = %d" %dcm_rcfg_bit29)
    print("     * dcm_speed (bit31:30)     = 0x%x" %dcm_speed_bit31_30)

    return None
#
#******************************************************
def show_mux_ctrl_reg_contents():
#****************************************************** 
#
    glu_reg_base_addr = get_glu_base_addr()

    mux_ctrl_value = get_reg_value(glu_reg_base_addr + REG_MUX_CTRL)
    mux_sel_bit = mux_ctrl_value & 0x1
    mux_oen_bit = (mux_ctrl_value & 0x2) >> 1
    print("* OSB mux_ctrl_value = 0x%x" % mux_ctrl_value)
    print("     - MUX_CTRL(0) = mux_sel = %d (0 : switch selected ; 1 : DTSS selected)" %(mux_sel_bit))
    print("     - MUX_CTRL(1) = mux_oen  = %d\n" %mux_oen_bit)

    return None
#
#**********************************************************************************************
#************************ OSB spi functions ***************************************************
#**********************************************************************************************
#
#******************************************************
def spi_master_transfer(readNwrite, address, wr_data):
#******************************************************

    debug_flag = False

    if readNwrite == 1:
        if debug_flag:
            print("readNwrite = Read")
        header = 0xf0
    else:
        if debug_flag:
            print("readNwrite = Write")
        header = 0xe0

    if debug_flag:
        print("address = 0x%x" %address)
        print("wr_data = 0x%x" %wr_data)

    # header trfer :
    #================

    glu_reg_base_addr = get_glu_base_addr()

    if debug_flag:
        print("\n** Header : set OSB SPI Master SPI_TRANSMIT value = 0x%x" % header)

    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, (header * 65536))

    if debug_flag:
        header_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        print("            => OSB SPI Master SPI_TRANSMIT reg. read value = 0x%x" % header_value)

    time.sleep(0.5)

    mod_spi_sel16 = 0x0
    mod_spi_cpol  = 0x0
    mod_spi_cpha  = 0x0
    mod_spi_freq  = 0x12

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    if debug_flag:
        print("            Launch SPI transfer with OSB SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    # reg_addr trfer :
    #==================

    if debug_flag:
        input("\nAddress transfer : Enter")
        print("")

    reg_addr_value = address * 2

    if debug_flag:
        print("** Register Address : Set OSB SPI Master SPI_TRANSMIT value = 0x%x" % reg_addr_value)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, ( reg_addr_value * 65536))

    if debug_flag:
        addr_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        print("                   => OSB SPI Master SPI_TRANSMIT reg. read value = 0x%x" % addr_value)

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    if debug_flag:
        print("                   Launch SPI transfer with OSB SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    # reg_addr_value = SPI_ADDR_VERSION

    if debug_flag:
        input("\nData transfer : Enter")
        print("")

    # data trfer :
    #==============

    if debug_flag:
        print("** Write_data : OSB SPI Master SPI_TRANSMIT value = 0x%x" % wr_data)

    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, (wr_data * 65536))

    if debug_flag:
        data_wr_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        print("             => OSB SPI Master SPI_TRANSMIT reg. read value = 0x%x" % data_wr_value)

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    if debug_flag:
        print("             Launch SPI transfer with OSB SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    time.sleep(0.5)

    if readNwrite == 1:
        if debug_flag:
            print("\nOSB SPI Master SPI_TRANSMIT Receive value = 0x%x" % wr_data)

        reg_data_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        return_value = reg_data_value & 0xff
        if debug_flag:
            print("             OSB SPI Master SPI_TRANSMIT read value = 0x%x" % return_value)

    if readNwrite == 1:
        return return_value
    else:
        return None
#
#**********************************************************************************************
#************************ OSB fans functions **************************************************
#**********************************************************************************************
#
#***********************
def OSB_fan_topology():
#***********************
#
    debug = False

    # os_cmde : ls -ltr /sys/class/hwmon | grep -i "osb_fpga:fan"

    FileName = "/usr/share/pydiag/oth/fan_topology.txt"

    os_base_cmd = "ls -l /sys/class/hwmon | grep -i 'osb_fpga:fan'"
    os_cmde = os_base_cmd+" > "+FileName

    if debug:
        print("\n * os_cmde is :", os_cmde, "\n")

    (os.system(os_cmde))

    tmp_list = []
    drv_vs_pos_list = []

    line_num = 1
    with open(FileName,'r') as tmpfile:
        for line in tmpfile.readlines():
            #print("****** line is ", line)

            tmp_list = line.split()

            if debug == True:
                print(" * line_num = %d :" %line_num)
                print(" ==================\n")
                print("   => tmp_list is", tmp_list)

            hwmon_flag = False
            fan_nb_flag = False

            for elt in enumerate(tmp_list):
                if 'hwmon' in elt[1]:
                    if '/' not in elt[1]:
                        if debug:
                            print("\n   * elt is ", elt, ": hwmon string found")
                        hmon_str = elt[1]
                        hwmon_flag = True
                    else:
                        if 'osb_fpga:fan' in elt[1]:
                            my_list1 = elt[1].split('/')
                            if (debug):
                                print("\n   * my_list1 is", my_list1)
                            for el in enumerate(my_list1):
                                if 'osb_fpga:fan' in el[1]:
                                    my_list2 = el[1].split(':')
                                    if (debug):
                                        print("   * my_list2 = ", my_list2)
                                    for e in enumerate(my_list2):
                                        if 'fan' in e[1]:
                                            fan_nb = e[1]
                                            if (debug):
                                                print("   * fan_nb = ", fan_nb)
                                            fan_nb_flag = True

                    if hwmon_flag == True and fan_nb_flag == True:
                        drv_vs_pos_list.append(hmon_str)
                        drv_vs_pos_list.append(fan_nb)

                        hwmon_flag = False
                        fan_nb_flag = False
            if debug:
                print("\n   * line_num = %d :" %line_num,"drv_vs_pos_list is", drv_vs_pos_list, "\n")

            line_num = line_num + 1

    tmpfile.close()

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return drv_vs_pos_list
#
#*****************************************************************************************
#************** B200 functions ***********************************************************
#*****************************************************************************************
#
#=============================================================================================
def osb200_cpld_detect():
#=============================================================================================
#
    FileName = "/usr/share/pydiag/oth/cpld_load.txt"

    f = open(FileName, "w")
    f.write('cable osb\n')
    f.write('detect\n')
    f.close()

    print()
    
    cmde = "jtag "+FileName

    res = False
    
    p = subprocess.Popen(cmde, shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                         stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            if ((columns[0] == "Part(0)") and (columns[1] == "10M04SCU169 (0x318A)")):
                res = True

    if  res:
        print_ok("\n=> CPLD Detected : PASS\n")
        return True
    else:
        print_info("\n=> CPLD Detected : FAIL")
        return False
#
#=============================================================================================
def show_z112_B200_misc_reg_contents():
#=============================================================================================
#
    side = 0
    slice = DTSS_SLICE
    if (os.path.exists("/dev/uts_spi")):
        Utsspi = utsspi.SpiModule(side, slice, "/dev/uts_spi")
    else:
        print("Utsspi driver (/dev/uts_spi) not found")
        return False

    value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
    print("\n Z112 board : Read cAddress_Misc reg. value @ 0x%1x : 0x%x" %(SPI_ADDR_MISC, value))

    enable_loop_uart_bit0       = value & 0x1
    select_2V5_VIN_bit1         = (value & 0x2) >> 1
    nIRQ_SX_bit2                = (value & 0x4) >> 2
    nIRQ_Sn_bit3                = (value & 0x8) >> 3
    synchro_meas_start_bit4     = (value & 0x10) >> 4
    synchro_meas_ok_bit5        = (value & 0x80) >> 5

    print("         - REGISTER_MISC(0) = ENABLE_LOOP_UART = %d" %enable_loop_uart_bit0)
    print("         - REGISTER_MISC(1) = SELECT_2V5_VIN   = %d" %select_2V5_VIN_bit1)
    print("         - REGISTER_MISC(2) = nIRQ_SX_en       = %d" %nIRQ_SX_bit2)
    print("         - REGISTER_MISC(3) = nIRQ_Sn_en       = %d" %nIRQ_Sn_bit3)
    print("         - REGISTER_MISC(4) = SYNCHRO_MEASURE_START = %d" %synchro_meas_start_bit4)
    print("")
    print("         - REGISTER_MISC(7) = SYNCHRO_MEASURE_OK = %d" %synchro_meas_ok_bit5)

    return None
#
#**************************************************************
def B200_info(verbose_flag):
#**************************************************************
#
    side = 0
    slice = DTSS_SLICE
    if (os.path.exists("/dev/uts_spi")):
        Utsspi = utsspi.SpiModule(side, DTSS_SLICE, "/dev/uts_spi")
    else:
        print("Utsspi driver (/dev/uts_spi) not found")
        return False

    value = Utsspi.read_mod_register8(SPI_ADDR_BOARD_PRES)

    switch_pres_b_bit = value & 0x1
    dtss_pres_b_bit   = (value & 0x2)/2
    toggle_bit        = (value & 0x4)/4
    board_id          = (value & 0xc0)/64

    if verbose_flag:
        print("* B200 CPLD module_presence_rdback register details :")
        print("     * module_presence_rdback read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_BOARD_PRES, value))
        print("         - BOARD_ID           = %d (2 = SWITCH ; 1 = DTSS)" %board_id)
        print("         - toggle_board_state = %d " %toggle_bit)
        print("         - DTSS_PRES_B        = %d ('0' = Selected ; '1' = DeSelected)" %dtss_pres_b_bit)
        print("         - SWITCH_PRES_B      = %d ('0' = Selected ; '1' = DeSelected)\n" %switch_pres_b_bit)

    return (board_id, toggle_bit, dtss_pres_b_bit, switch_pres_b_bit)
#
#******************************************************
def b200_module_type_set(module_to_set, verbose_flag):
#******************************************************
#
    glu_reg_base_addr = get_glu_base_addr()

    side = 0
    slice = B200_SLICE
    if (os.path.exists("/dev/uts_spi")):
        Utsspi = utsspi.SpiModule(side, slice, "/dev/uts_spi")
    else:
        print("Utsspi driver (/dev/uts_spi) not found")
        return False

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    if verbose_flag:
        print("* B200 CPLD_version readback value is 0x%2x" % cpld_version)

    (board_id, toggle_bit, dtss_pres_b_bit, switch_pres_b_bit) = B200_info(verbose_flag)

    if verbose_flag:
        show_mux_ctrl_reg_contents()

    if module_to_set == "DTSS":
        if dtss_pres_b_bit == 0:
            if verbose_flag:
                print(" %s module (%d) allready selected : no action" %(module_to_set, board_id))
            return True
        else:
            if verbose_flag:
                print(" Select %s module (Apply a RESET) : " %module_to_set)
    else:
        if switch_pres_b_bit == 0:
            if verbose_flag:
                print("%s module (%d) allready selected : no action" %(module_to_set, board_id))
            return True
        else:
            if verbose_flag:
                print("Select %s module (Apply a RESET) : " %module_to_set)

    if verbose_flag:
        print ("\n NOW WRITE TO TOGGLE\n")

    wr_data = 0x0

    if verbose_flag:
        print("     * Write 0x%x into B200 REGISTER_MOD (@ addr 0x%1x)\n" %(wr_data, SPI_ADDR_BOARD_PRES))

    Utsspi.write_mod_register8(SPI_ADDR_BOARD_PRES, wr_data)

    (dtss_fpga_rdy, dtss_presence, switch_presence) = module_presence_info(verbose_flag)

    if verbose_flag:
        show_mux_ctrl_reg_contents()

    my_verbose = False
    periph_reset_gen("release", my_verbose)

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("* B200 CPLD version is 0x%2x\n" % cpld_version)

    (board_id, toggle_bit, dtss_pres_b_bit, switch_pres_b_bit) = B200_info(verbose_flag)

    return None
#
#*****************************************************************************************
#************** OSB DTSS functions *******************************************************
#*****************************************************************************************
#
#**************************************************************
def module_presence_info(verbose_flag):
#**************************************************************
#
    glu_reg_base_addr = get_glu_base_addr()

    module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)

    switch_bit        = module_presence_rdback_value & 0x1
    dtss_bit          = (module_presence_rdback_value & 0x2)/2
    dtss_fpga_rdy_bit = (module_presence_rdback_value & 0x4)/4

    if verbose_flag:
        print("\n* OSB module_presence_rdback register details :")
        print("     * module_presence_rdback_value read = 0x%x" % module_presence_rdback_value)
        print("         - module_presence_rdback(2) = DTSS_FPGA_RDY = %d ('0' = FPGA NOT loaded ; '1' = FPGA Loaded)" %dtss_fpga_rdy_bit)
        print("         - module_presence_rdback(1) = DTSS_PRES     = %d ('0' = ABSENT ; '1' = present)" %dtss_bit)
        print("         - module_presence_rdback(0) = SWITCH_PRES   = %d ('0' = ABSENT ; '1' = present)" %switch_bit)

    return (dtss_fpga_rdy_bit, dtss_bit, switch_bit)
#
#**************************************************************
def is_dtss_fpga_present():
#**************************************************************
#    
    verbose = False
    (dtss_fpga_rdy, dtss_presence, switch_presence) = module_presence_info(verbose)
    
    if dtss_fpga_rdy == 0:
        print()
        print_error ("* Error : DTSS module FPGA is not loaded")
        return False
    else:
        return True
#
#**********************************************************************************
# ***************** TACB functions definition *************************************
#**********************************************************************************
#
#*********************************
def get_tacb_pci_glu_base_addr():
#*********************************
#
    debug = False
    
    # bar1_addr for TACB GLU : base_address = 0x18100000

    tacb_in_ona800 = True
    if tacb_in_ona800:
        FileName = "/usr/share/pydiag/tacb_pci_glu_base_tmp.txt"
    else:
        FileName = "/usr/share/pydiag/oth/tacb_pci_glu_base_tmp.txt"

    os_base_cmd = "lspci -s 03:00.0 -v | grep size=128K"
    os_cmde = os_base_cmd+" > "+FileName

    if debug:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()
    
    if debug:
        print("File %s: line is %s" %(FileName, line))
    
    tmp_list = line.split()

    if debug == True:
        print("tmp_list is" , tmp_list)

    base_addr = tmp_list[2]

    base_address = int(base_addr, base=16)

    if debug == True:
        print("base_address is 0x%x (d'%d)" %(base_address, base_address))

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return base_address
#
#*********************************
def get_tacb_pci_bdt_base_addr():
#*********************************

    # tacb is a 2 functions pcie :
    #
    # fct_0 (03:00.0) with 3 bars (from bar0 to bar2) : controlled by pci_glu driver
    # fct_1 (03:00.1) with 2 bars (from bar3 to bar4) : controlled by pci_bdt driver
    #
    # 0000:03:00.0 Communication controller: Device 1d7d:2102
    # Memory at 18120000 (32-bit, non-prefetchable) [size=64K]  <- bar0
    # Memory at 18100000 (32-bit, non-prefetchable) [size=128K] <- bar1 : glu
    # Memory at 18130000 (32-bit, non-prefetchable) [size=64K]  <- bar2 : ethernet
    # Kernel modules: pci_glu
    #
    # 0000:03:00.1 Communication controller: Device 1d7d:2102   
    # Memory at 18140000 (32-bit, non-prefetchable) [size=64K]  <- bar3 : bdt
    # Memory at 18200000 (32-bit, non-prefetchable) [size=2M]   <- bar4 : zinq_ps
    # Kernel modules: pci_bdt
    
    debug = False

    # bar3_addr for TACB BDT : base_address = 0x18140000

    tacb_in_ona800 = True
    tacb_1fct = True

    if tacb_in_ona800:
        FileName = "/usr/share/pydiag/tacb_pci_bdt_base_tmp.txt"
    else:
        FileName = "/usr/share/pydiag/oth/tacb_pci_bdt_base_tmp.txt"

    if tacb_1fct:
        os_base_cmd = "lspci -s 03:00.0 -v | grep size=64K"
    else:
        os_base_cmd = "lspci -s 03:00.1 -v | grep size=64K"
    
    os_cmde = os_base_cmd+" > "+FileName

    if debug:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()

    if debug:
        print("File %s: line is \n %s" %(FileName, line))

    tmp_list = line.split()

    if debug:
        print("tmp_list is" , tmp_list)

    if tacb_1fct:
        base_addr = tmp_list[14]
    else:
        base_addr = tmp_list[2]

    base_address = int(base_addr, base=16)

    if debug:
        print("base_address is 0x%x (d'%d)" %(base_address, base_address))

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return base_address
#
#====================================================================================
#
if __name__ == '__main__':
    print('osb_reg_def module ok')
