#!/usr/bin/python3

import os
import ctypes
from ctypes.util import find_library
import fcntl
import struct
import json
import os.path
from functools import lru_cache

MS_RDONLY = 1
MS_NOSUID = 2
MS_NODEV = 4
MS_NOEXEC = 8
MS_SYNCHRONOUS = 16
MS_REMOUNT = 32
MS_MANDLOCK = 64
MS_DIRSYNC = 128
MS_NOATIME = 1024
MS_NODIRATIME = 2048
MS_BIND = 4096
MS_MOVE = 8192
MS_REC = 16384
MS_SILENT = 32768
MS_POSIXACL = 1 << 16
MS_UNBINDABLE = 1 << 17
MS_PRIVATE = 1 << 18
MS_SLAVE = 1 << 19
MS_SHARED = 1 << 20
MS_RELATIME = 1 << 21
MS_KERNMOUNT = 1 << 22
MS_I_VERSION = 1 << 23
MS_STRICTATIME = 1 << 24
MS_ACTIVE = 1 << 30
MS_NOUSER = 1 << 31


@lru_cache(maxsize=2)
def get_partition_plan():
    hardware_configuration_file = '/etc/hardware.conf'
    if not os.path.exists(hardware_configuration_file):
        print("Cannot found hardware configuration file, critical situation stopping operation.")
        exit(-1)
    with open(hardware_configuration_file) as json_file:
        data = json.load(json_file)
        if 'emmc' not in data:
            print("Cannot found partitioning information in hardware configuration file, stopping operation")
            exit(-1)
        number_of_partitioning_plan = len(data['emmc'])
        if number_of_partitioning_plan > 1:
            blk_device_list = get_block_devices("MMC")
            if len(blk_device_list) == 0:
                print("CRITICAL: no emmc found when looking for partition plan")
                exit(-1)
            if len(blk_device_list) > 1:
                print("Warning: many emmc found when looking for partition plan, using the first one!")
            partition_size = get_blockdevice_size([*blk_device_list][0])
        try:
            for p in data['emmc']:
                if number_of_partitioning_plan == 1 or (p['min_size'] <= partition_size < p['max_size']):
                    return p['partitioning_plan']
        except:
            print("Partition plan information are corrupted, stopping operation")
            exit(-1)
        print("Cannot found partitioning information, critical situation stopping operation.")
        exit(-1)


# Return the size of a block device in Mo
@lru_cache(maxsize=8)
def get_blockdevice_size(blkname):
    buff = ' '*4
    blkcount = 0
    blksize = 0
    with open('/dev/{}'.format(blkname), 'r') as fd:
        buff = fcntl.ioctl(fd.fileno(), 0x1260, buff)
        blkcount = struct.unpack('I', buff)[0]
        buff = fcntl.ioctl(fd.fileno(), 0x1268, buff)
        blksize = struct.unpack('I', buff)[0]
    return blkcount*blksize/(1024*1024)


@lru_cache(maxsize=4)
def get_block_devices(filtertype=None):
    blockdevice = {}
    for d in os.listdir("/sys/block"):
        if not os.path.exists("/sys/block/%s/device/type"%d):
            continue
        with open("/sys/block/%s/device/type"%d) as t:
            typ = t.read().strip()
            if filtertype is None or typ == filtertype:
                blockdevice[d] = typ
    return blockdevice


def mount_bind(source, target):
    """Call mount(2); see the man page for details."""
    libc = ctypes.CDLL(find_library('c'), use_errno=True)
    source = source.encode()
    target = target.encode()
    if libc.mount(source, target, 0, ctypes.c_ulong(MS_BIND), 0) != 0:
        e = ctypes.get_errno()
        raise OSError(e, os.strerror(e))


def mount(source, target, fs, options='', flag=0):
    if not os.path.exists(target):
        os.makedirs(target)
    ret = ctypes.CDLL('libc.so.6', use_errno=True).mount(ctypes.c_char_p(source.encode('ASCII')),
                                                         ctypes.c_char_p(target.encode('ASCII')),
                                                         ctypes.c_char_p(fs.encode('ASCII')),
                                                         ctypes.c_ulong(flag),
                                                         ctypes.c_char_p(options.encode('ASCII')))
    if ret < 0:
        errno = ctypes.get_errno()
        raise RuntimeError("Error mounting {} ({}) on {} with options '{}': {}".format(source, fs, target, options, os.strerror(errno)))
    return ret


def umount(source):
    """Umount a partition (accept only a partition device as input)"""
    ret = ctypes.CDLL('libc.so.6', use_errno=True).umount(ctypes.c_char_p(source.encode('ASCII')))
    if ret < 0:
        errno = ctypes.get_errno()
        raise RuntimeError("Error unmounting {} ({})".format(source, os.strerror(errno)))
    return ret


def mounted_point(partition):
    """Return a list of mounted point the partition is mounted to."""
    mounted_path = []
    if os.path.exists('/etc/mtab'):
        mounted_path = [line.split()[1] for line in open('/etc/mtab', 'r') if line.startswith(partition)]
    return mounted_path


def is_mount(path):
    with open("/etc/mtab", "r") as mtab:
        for line in mtab:
            if path == line.split()[1]:
                return True
    return False


CAL_UNMOUNTED = 0
CAL_MOUNTED_RO = 1
CAL_MOUNTED_RW = 2


def mount_cal():
    mmc = list(get_block_devices("MMC").keys())[0]
    if not is_mount("/cal"):
        mount("/dev/%sp%i" % (mmc, 2), '/cal', 'ext4')
        return CAL_UNMOUNTED
    else:
        with open("/proc/mounts","r") as procmount:
            for line in procmount.readlines():
                tline = line.split()
                if tline[1] == "/cal":
                    if "ro" in tline[3].split(","):
                        umount("/cal")
                        mount("/dev/%sp%i" % (mmc, 2), '/cal', 'ext4')
                        return CAL_MOUNTED_RO
                    elif "rw" in tline[3].split(","):
                        return CAL_MOUNTED_RW
                    else:
                        print("UNKNOWN CAL STATE !!!!")
    return -1


def umount_cal(mode):
    os.sync()
    mmc = list(get_block_devices("MMC").keys())[0]
    if mode == CAL_UNMOUNTED or mode == CAL_MOUNTED_RO:
        umount('/cal')
    if mode == CAL_MOUNTED_RO:
        mount("/dev/%sp%i" % (mmc, 2), '/cal', 'ext4', flag = MS_RDONLY)
    return mode
