<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\util\upload;

use app\sharedmemory\SMTMemoryManager;

use app\services\maintenance\SMTUploadStatusDto;

use app\util\SMTLogger;

/**
 * Handles upload function
 *
 * @author Sylvain Desplat
 */
class SMTUploadStatus
{
    const UPLOAD_PROGRESS_IN_APC = "upload_";
    
    const OTU_UPLOAD_PROGRESS_KEY = "otu-8kv2";
    
    const UPLOAD_RELEASE_IN_APC = "upload_release_running";
        
    public static function isUploadRelaseRunning()
    {
    	return apc_fetch(self::UPLOAD_RELEASE_IN_APC);
    }
    
    public static function setUploadRelaseRunning($running)
    {
    	apc_store(self::UPLOAD_RELEASE_IN_APC, $running);
    	SMTLogger::getInstance()->trace("setUploadRelaseRunning: ".$running, SMTLogger::DEBUG);
    	if (!$running)
    	{
    		//ensure progress is 100%
    		$cacheDto = self::getCurrentUploadInfoFromCache();
    		if ( $cacheDto != NULL )
    		{
    			$cacheDto->setProgress( SMTUploadStatusDto::UPLOAD_COMPLETE);
    			$cacheDto->save();
    		}
    	}
    }
    
    /**
     * Retrieve current upload info from APC cache
     *
     * @return SMTUploadStatusDto or NULL if no upload is in progress or was performed
     */
    public static function getCurrentUploadInfoFromCache()
    {
    	$currentUploadInfo = NULL;
    	//retrieve current Upgrade from memory:
    	$dtos = SMTMemoryManager::fetchAll( SMTUploadStatusDto::getClass() );
    	if ( is_array( $dtos ) && count( $dtos ) > 0 )
    	{
    		reset($dtos);
    		$first_key = key($dtos);
    		$currentUploadInfo = $dtos[$first_key];
    	}
    
    	return $currentUploadInfo;
    }    
    
    /**
     * Check whether an upload is in progress
     *
     * @return boolean
     */
    public static function checkNoUploadRunning()
    {
    	$noUploadRunning = TRUE;
    
    	if ( PHP_MAJOR_VERSION >= 6 )
    	{
    		$key = ini_get("session.upload_progress.prefix").self::OTU_UPLOAD_PROGRESS_KEY;//$_POST[ini_get("session.upload_progress.name")];
    		
    		SMTLogger::getInstance()->trace("Get upload info with key ".$key, SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
    		
    		$apcStatus = (isset($_SESSION[$key])) ? $_SESSION[$key] : FALSE;
    	}
    	else
    	{
	    	//retrieve php upload progress from apc
    		$apcStatus = apc_fetch( self::UPLOAD_PROGRESS_IN_APC.self::OTU_UPLOAD_PROGRESS_KEY );
    	}
    
    	//an upload is running
    	if ( $apcStatus != FALSE && ( array_key_exists('done', $apcStatus) && (($apcStatus['done'] != "1") && ($apcStatus['done'] != "true") ) && ( array_key_exists('cancel_upload', $apcStatus) && ($apcStatus['cancel_upload'] != "1") ) ) )
    	{
    		SMTLogger::getInstance()->trace('apcStatus[done]='.$apcStatus['done'].' apcStatus[cancel_upload]='.$apcStatus['cancel_upload'] );
    		$noUploadRunning = FALSE;
    	}
    	else if ( self::isUploadRelaseRunning() )
    	{
    		SMTLogger::getInstance()->trace("Upload release still running: checking release");
    		$noUploadRunning = FALSE;
    	}
        else
        {
        	SMTLogger::getInstance()->trace("No Upload running");
        }
    	return $noUploadRunning;
    }    
    
    /**
     *
     * @param string $id Upload identifier ( should be OTU_UPLOAD_PROGRESS_KEY)
     *
     * @return SMTUploadStatusDto The upload progress
     */
    public static function getUploadProgress( $id )
    {
    	$uploadProgress = SMTUploadStatusDto::NO_UPLOAD;
    	$uploadRate = SMTUploadStatusDto::NO_RATE;
    	$errorStatus = SMTUploadStatusDto::NO_ERROR;
    	$cacheDto = self::getCurrentUploadInfoFromCache();
    
    	if ( PHP_MAJOR_VERSION >= 6 )
    	{
    		$key = ini_get("session.upload_progress.prefix") . $id;//$_POST[ini_get("session.upload_progress.name")];
    		
//     		SMTLogger::getInstance()->trace(ini_get("session.upload_progress.prefix")."  ".ini_get("session.upload_progress.name"), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
//     		SMTLogger::getInstance()->trace("Get upload info with key ".$key, SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
//     		SMTLogger::getInstance()->trace("Session info: ".print_r($_SESSION, TRUE), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
    		
    		$uploadInfo = (isset($_SESSION[$key])) ? $_SESSION[$key] : NULL;
    		if ( $uploadInfo != NULL )
    		{
//      			SMTLogger::getInstance()->trace("Upload info: ".print_r($_SESSION[$key], TRUE), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
    			$uploadProgress = round( ($uploadInfo['bytes_processed']/$uploadInfo['content_length'])*100.0 );
    			$bytesUploaded = intval( $uploadInfo['bytes_processed']);
    			$uploadRate = 0;
    			if ( floatval( $uploadInfo['start_time'] ) > 0 )
    			{
    				$elapseTime = microtime(true) - floatval( $uploadInfo['start_time'] );
    				$uploadRate = $bytesUploaded / $elapseTime;
    				$uploadRate = round( $uploadRate/1000000.0, 2);
    			}
    		}
    		//no upload is running, retrieve error status from cache ( it can only be computed by upload process (SMTUpload.php) at the end of the upload)
    		else if( $cacheDto != NULL )
    		{
    			$errorStatus = $cacheDto->getUploadErrorStatus();
    		}
    		else if ( self::isUploadRelaseRunning() )
    		{
    			SMTLogger::getInstance()->trace("Upload release still running: checking release");
    		}
    		else
    		{
    			SMTLogger::getInstance()->trace("Couldn't get upload info with key ".$key, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		}
    	}
    	else
    	{
	    	//retrieve php upload progress from apc
	    	$apcStatus = apc_fetch( self::UPLOAD_PROGRESS_IN_APC.$id );
	    
	    	//an upload is running
	    	if ( $apcStatus != FALSE )
	    	{
	    		$uploadProgress = round( $apcStatus['current']/$apcStatus['total']*100.0 );
	    		$bytesUploaded = intval( $apcStatus['current'] );
	    		$uploadRate = 0;
	    		if ( floatval( $apcStatus['start_time'] ) > 0 )
	    		{
	    			$elapseTime = microtime(true) - floatval( $apcStatus['start_time'] );
	    			$uploadRate = $bytesUploaded / $elapseTime;
	    			$uploadRate = round( $uploadRate/1000000.0, 2);
	    		}
	    	}
	    	//no upload is running, retrieve error status from cache ( it can only be computed by upload process (SMTUpload.php) at the end of the upload)
	    	else if( $cacheDto != NULL )
	    	{
	    		$errorStatus = $cacheDto->getUploadErrorStatus();
	    	}
    	}
    
    	$dto = new SMTUploadStatusDto();
    	$dto->setProgress( $uploadProgress );
    	$dto->setRateMBs( $uploadRate );
    	$dto->setUploadErrorStatus( $errorStatus );
    	if ($dto->isUploadInProgress() == FALSE )
    	{
    		//sleep 1s if upload is finished to let the "upload release running" status to be updated  
    		sleep(1);
    	}
    	$dto->setCheckingUpload(self::isUploadRelaseRunning());
    	if ( $cacheDto != NULL )
    	{
    	    $dto->setFileName( $cacheDto->getFileName() );
    	}
    
    	//if an upload is in progress, updates its upload progress in cache
    	if ( $apcStatus != FALSE )
    	{
    		if( $cacheDto != NULL )
    		{
    			//don't set error status in cache; it can only be computed by upload process (SMTUpload.php) at the end of the upload
    			$cacheDto->setProgress( $uploadProgress );
    			$cacheDto->setRateMBs( $uploadRate );
    			$cacheDto->save();
    		}
    		//no upload status exists in cache, create it
    		else
    		{
    			$dto->save();
    		}
    	}
    
    	return $dto;
    }    
    
    /**
     * If upload is finished, remove the php upload progress object from APC
     *
     */
    public static function cleanPhpUploadProgressFromApcCache()
    {
    	if ( PHP_MAJOR_VERSION >= 6 )
    	{
    	    $sessionKey = ini_get("session.upload_progress.prefix").self::OTU_UPLOAD_PROGRESS_KEY;//$_POST[ini_get("session.upload_progress.name")];
    		
    	    SMTLogger::getInstance()->trace("Get upload info with key ".$sessionKey, SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
    		
    		$uploadInfo = (isset($_SESSION[$sessionKey])) ? $_SESSION[$sessionKey] : NULL;
    		if ( $uploadInfo != NULL )
    		{
    			unset($_SESSION[$sessionKey]);
    		}
    	}
    	else
    	{
	    	$key = self::UPLOAD_PROGRESS_IN_APC.self::OTU_UPLOAD_PROGRESS_KEY;
	    	$success = FALSE;
	    	$deleteTrialCount = 0;
	    	$maxDeleteTrialCount = 50;
	    	if ( apc_exists( $key ) )
	    	{
	    		while ( !$success && $deleteTrialCount < $maxDeleteTrialCount )
	    		{
	    			$deleteTrialCount++;
	    			usleep(rand(1000, 10000));
	    			$success = apc_delete( $key );
	    		}
	    		if ( !$success )
	    		{
	    			SMTLogger::getInstance()->trace("Couldn't delete ".$key." from APC store.".$key, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
	    		}
	    	}
    	}
    }
}


?>