<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otdr;

use app\events\SMTEventFactory;
use app\events\operations\SMTOperationEventDto;
use app\otuMediation\SMTOtuMeasureOnDemandStatus;
use app\parser\SMTOtuApi;
use app\parser\SMTOtuSocket;
use app\parser\SMTOtuSocketManager;
use app\services\api\SMTApiMeasureInstrumentStatusDto;
use app\services\api\SMTApiMeasurementDto;
use app\serviceshelper\module\SMTModule;
use app\serviceshelper\monitoring\SMTMeasureOnDemandEventDto;
use app\serviceshelper\monitoring\SMTMeasureOnDemandException;
use app\util\SMTLogger;
use app\util\SMTUtil;
use app\http\SMTContext;

/**
 * Handles measure operations: 
 * - Fetch measure parameters
 * - Start a measurement on a port.
 * - utility functions for a measure
 * 
 * @author Sylvain Desplat
 */
class SMTMeasurementInstrument extends SMTCommonMeasurement
{      
    const OTU_SET_UP_DIR = 'harddisk/otu/setup/';
    const OTU_RESULT_DIR = 'harddisk/otu/result/measure_on_demand';
    const OTU_SET_UP_FILE = 'otuConfig_%d.%s.fo_cfg';
    const OTU_RESULT_FILE_PREFIX = 'port%d';
    const OTU_RESULT_FILE = self::OTU_RESULT_FILE_PREFIX.'_[Date_Time]';
    const MAX_ACQ_FILE_LIFE_TIME = 600; //10min
    const DEFAULT_MAX_ACQUISITION_COUNT = 6; // 2x3 max acq count per port = 2 (trace, CDM, pdf)
    
    
    /**
     * Check whether requested function is available and returns the module found
     * 
     * @param SMTApiMeasurementDto $measurementDto
     * @throws SMTMeasureOnDemandException
     * @return \app\serviceshelper\module\app\serviceshelper\otdr\SMTModuleDto
     */
    private function checkRequestedFunctionAvailable( SMTApiMeasurementDto $measurementDto )
    {
        //retrieve OTDR module functions, wavelengths...
        $module = new SMTModule();
        $module->setContext( $this->getContext() );
        $otdrModule = $module->fetchOtdrModuleFunctions();
        $functions = $otdrModule->getFunctions();
        $functionFound = FALSE;
        
        // check that requested function is available in the list
        foreach ($functions as $function)
        {
            if ( strcasecmp($function->getName(), $measurementDto->getConfigType()) == 0 )
            {
                $functionFound = TRUE;
            }
        }
        if (!$functionFound)
        {
            $this->getContext()->getLogger()->trace("Couldn't find requested function: ".$measurementDto->getConfigType(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, MSG_ERROR_START_MEASURE ) ;
        }
        
        return $otdrModule;
    }
    
    /**
     * Open socket on FO app
     * 
     * @throws SMTMeasureOnDemandException
     * @return \app\parser\SMTOtuSocket
     */
    private function getFoSocket()
    {
        // retrieve FO connection
        $portFO = SMTOtuSocketManager::retrieveFOPort( $this->getContext() );
        if ( $portFO !== NULL && SMTUtil::isInt($portFO) && ($portFO > 0) && ($portFO <= 65535) )
        {
            $socketManager = new SMTOtuSocketManager( $this->getContext() );
            //retrieve connection from the socket manager
            $socketFO = $socketManager->openSocket( $portFO );
        }
        else
        {
            $this->getContext()->getLogger()->trace("Couldn't open FO socket on port: ".$portFO, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, 'Failed to access FO application' ) ;
        }
        SMTLogger::getInstance()->trace( sprintf("Retrieved connection on FO on port %d", $portFO));
        
        return $socketFO;
    }
    
    
    /**
     * Start measurement on port
     *
     * @param SMTApiMeasurementDto $measurementDto
     * @param int $portNumber
     *
     * @return SMTMeasureOnDemandEventDto;
     *
     * @throws SMTMeasureOnDemandException if measurement couldn't be launched
     */
    function startMeasure( SMTApiMeasurementDto $measurementDto, $portNumber )
    {        
        // 1 check that requested function is available
        $otdrModule = $this->checkRequestedFunctionAvailable( $measurementDto );

        if ( $this->acquireMeasureOnDemandLock() )
        {
            // 2 - check that no measurement is running
            //if a measurement is already running, abort start measurement operation
            try
            {
                $operationId = SMTMeasureOnDemandEventDto::generateMeasureInstrumentOperationId($portNumber);
                $measurementStatus = $this->checkNoMeasurementRunning( NULL , $operationId );
            }
            catch ( \Exception $e )
            {
                SMTLogger::getInstance()->trace( "Measurement already running: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                $this->releaseMeasureOnDemandLock();
                $this->getContext()->getLogger()->traceException($e);
                
                throw $e;
            }
            
            // keep only 2 measurements (2x3) on that port
            self::cleanupAcquisitions($portNumber);
            
            // 3 - interrupt current measurement
            $retry = FALSE;
            try
            {
                $this->send( SMTOtuApi::getInterruptOtuSequencerCommand() );
            }
            catch ( \Exception $e )
            {
                $this->getContext()->getLogger()->traceException($e);
                
                //retry once:
                $retry = TRUE;
            }
            
            if ($retry)
            {
                try
                {
                    //interrupt current measurement
                    $this->send( SMTOtuApi::getInterruptOtuSequencerCommand() );
                }
                catch ( \Exception  $e)
                {
                    $this->getContext()->getLogger()->traceException($e);
                    //restart sequencer
                    $this->send( $this->send( SMTOtuApi::setMonitoringSequensorSuspend(FALSE) ) );
                    
                    throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, 'Measurement cannot be started on port  '.$portNumber ) ;
                }
            }
            
            // 4 - ask switch commutation
            try
            {
                //interrupt current measurement
                $this->send( SMTOtuApi::setSwitchCommuteCommand(0, 1, $portNumber) );
            }
            catch ( \Exception  $e)
            {
                $this->getContext()->getLogger()->traceException($e);
                
                //restart sequencer
                $this->send( $this->send( SMTOtuApi::setMonitoringSequensorSuspend(FALSE) ) );
                
                throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, 'Optical Switch cannot switch to port '.$portNumber ) ;
            }
            
            
            // 5 - Reset preivious setup
            try
            {
                // retrieve FO connection
                $socketFO = $this->getFoSocket();
                
                //reset previous ref
                $socketFO->send( SMTOtuApi::CMD_Fo_Reset_Ref );
                $socketFO->send( SMTOtuApi::CMD_Fo_Unlock_Marker );
                $socketFO->send( SMTOtuApi::CMD_Fo_Result ); //to close setup
            }
            catch ( \Exception $e )
            {
                $socketFO->close();
                $this->getContext()->getLogger()->traceException($e);
            }
            $socketFO->close();
            
            // 6 - select module and function
            SMTLogger::getInstance()->trace( sprintf("Select module %s and function %s", $otdrModule->getPosition(), $measurementDto->getConfigType()));
            $this->send( SMTOtuApi::getModuleSelectFunctionCommand( $otdrModule->getPosition(), $measurementDto->getConfigType() ) );
            
            
            $fileName = sprintf(self::OTU_SET_UP_FILE, $portNumber, $measurementDto->getConfigType() );
            $dir = '/user/'.self::OTU_SET_UP_DIR;
            $measurementDto->saveContent($dir, $fileName);
            
            SMTLogger::getInstance()->trace( sprintf("Save config file %s in %s", $fileName, $dir) );
            
            $measurementStatus = SMTEventFactory::createMeasureOnDemandEvent( NULL, $operationId, SMTOperationEventDto::NO_PROGRAMMED );

            SMTLogger::getInstance()->trace( sprintf("Start measurement on demand with id: ",  $operationId ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    
            // 7 - Start measure
            try
            {                
                // retrieve FO connection
                $socketFO = $this->getFoSocket();
                
                //setup measurement
                SMTLogger::getInstance()->trace( sprintf("Set measurement with module %s and function", $otdrModule->getPosition(), $measurementDto->getConfigType()), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);                
                $answer = $socketFO->sendReceive( SMTOtuApi::getLoadMeasurementCommand(self::OTU_SET_UP_DIR, $fileName ), 
                    SMTOtuSocket::DEFAULT_SLEEP, SMTOtuSocket::LONGER_TIME_OUT_READING, TRUE);
                SMTLogger::getInstance()->trace( sprintf("Measurement setup loaded:%s ", $answer), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
                
                //set directory output and file naming                
                $file = sprintf(self::OTU_RESULT_FILE, $portNumber);
                SMTLogger::getInstance()->trace( sprintf("Set measurement dir %s and file %s", self::OTU_RESULT_DIR, $file), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
                $socketFO->send( SMTOtuApi::getMeasurementFileNamingCommand(self::OTU_RESULT_DIR, $file ) ); 
                
                //start measurement
                SMTLogger::getInstance()->trace( sprintf("Start measurement with module %s and function", $otdrModule->getPosition(), $measurementDto->getConfigType()), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
                $socketFO->send( SMTOtuApi::getStartMeasurementCommand() );   
            }
            catch ( \Exception $e )
            {
                $socketFO->close();
                //nothing to do, try to get info with getMeasurementStatus
                $this->getContext()->getLogger()->traceException($e);
            }
            $socketFO->close();
            
            // 8 - check measurement status
            try
            {
                $measurementStatus = $this->getMeasurementStatus( $operationId );
                SMTLogger::getInstance()->trace( sprintf("Measurement added status %s", $measurementStatus->getStatus() ), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
                $this->releaseMeasureOnDemandLock();
    
                //1 -measurement was started: save some measure features
                $this->cacheMeasureInfo($operationId, $portNumber, '/user/'.self::OTU_RESULT_DIR , $file );
                
                //2 - measurement was started: start the polling on measure in an asynchronous process
                $measureOnDemandClass = SMTMeasureOnDemandEventDto::getClass();
                $measureOnDemandClass = str_replace("\\", "\\\\", $measureOnDemandClass);
                
                self::executeAsynchronousMeasureProcess( __DIR__.'/measurementProcess.php', $measureOnDemandClass, $operationId );
            }
            catch( \Exception $e )
            {
                $this->releaseMeasureOnDemandLock();
                throw  $e;
            }
        }
        else
        {
            $this->getContext()->getLogger()->trace("Couldn't acquire lock for SMTMeasurement to start measure with operation Id:".$operationId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, MSG_ERROR_START_MEASURE ) ;
        }
    
        return $measurementStatus;
    }               
    
    
    /**
     * Get measure instrument status from FO parser
     *
     * @param number $portNumber
     * @return SMTApiMeasureInstrumentStatusDto
     *
     * @throws SMTMeasureOnDemandException if instrument measurement fails
     */
    public function getMeasurementStatusFromPort( $portNumber )
    {
        // retrieve FO connection
        $socketFO = $this->getFoSocket();
        
        //get FO measurement status:
        $otuStatus = $socketFO->sendReceive( SMTOtuApi::getFoMeasurementStatus() );
        $socketFO->close();
        
        SMTLogger::getInstance()->trace( sprintf("Measurement status %s", $otuStatus ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    
        $measurementStatusApi = SMTApiMeasureInstrumentStatusDto::createMeasureInstrumentStatusApi( SMTOperationEventDto::FAILED );
        if ( SMTOtuMeasureOnDemandStatus::isValidMeasureStatus( $otuStatus ) )
        {
            $decodedStatus = SMTOtuMeasureOnDemandStatus::decodeOtuStatus( $otuStatus );
            
            SMTLogger::getInstance()->trace( sprintf("Measurement decoded status %s", $decodedStatus ), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
            if ( strcasecmp($decodedStatus, SMTOperationEventDto::AVAILABLE) == 0 )
            {                
                //search result files
                $fileName = sprintf('/user/'.self::OTU_RESULT_DIR.'/port%d*',$portNumber);
                foreach ( glob( $fileName ) as $file )
                {
                    if ( file_exists($file)  )
                    {
                        SMTLogger::getInstance()->trace( sprintf("Measurement file found: %s", $file ), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
                        $measurementStatusApi = SMTApiMeasureInstrumentStatusDto::createMeasureInstrumentStatusApi( SMTOperationEventDto::AVAILABLE );
                        break;
                    }                    
                }
            }
            else 
            {
                $measurementStatusApi = SMTApiMeasureInstrumentStatusDto::createMeasureInstrumentStatusApi( $decodedStatus );
            }
        }
        
        return $measurementStatusApi;
    }   
    
    public static function cleanupOldFiles($portNumber)
    {
        $fileName = sprintf('/user/'.self::OTU_RESULT_DIR.'/port%d*',$portNumber);
        foreach ( glob($fileName) as $file )
        {
            if (file_exists($file) && ( filemtime($file) + self::MAX_ACQ_FILE_LIFE_TIME < time() ) )
            {
                unlink($file);
            }
        }
    }
    
    
    
    
    public static function getCDMFileName( $portNumber)
    {
        $fileName = sprintf('/user/'.self::OTU_RESULT_DIR.'/port%d*.json',$portNumber);
        $found = FALSE;
        foreach ( glob( $fileName ) as $file )
        {
            if ( file_exists($file)  )
            {
                $found = TRUE;
                break;
            }
        }
        return $found? $file : '';
    }
    
    public static function getPdfFileName( $portNumber)
    {
        $fileName = sprintf('/user/'.self::OTU_RESULT_DIR.'/port%d*.pdf',$portNumber);
        $found = FALSE;
        foreach ( glob( $fileName ) as $file )
        {
            if ( file_exists($file)  )
            {
                $found = TRUE;
                break;
            }
        }
        return $found? $file : '';
    }
    
    public static function getTraceFileName( $portNumber)
    {
        $fileName = sprintf('/user/'.self::OTU_RESULT_DIR.'/port%d*sor',$portNumber);
        $found = FALSE;
        foreach ( glob( $fileName ) as $file )
        {
            if ( file_exists($file)  )
            {
                $found = TRUE;
                break;
            }
        }
        
        if (!$found)
        {
            SMTLogger::getInstance()->trace("file pattern not found ".$fileName, SMTLogger::ERROR);
        }
        return $found? $file : '';
    }
    
    public static function deleteFile( $fileName)
    {
        if ( !unlink($fileName) )
        {
            SMTLogger::getInstance()->trace("delete measure File>> file not found ".$fileName, SMTLogger::ERROR);
        }
    }
    
    /**
     * Launch asynchronous SCPI command
     *
     * @param string $phpFileToExecute
     * @param string $scpiCommands

     */
    static function executeAsynchronousSCPI( $phpFileToExecute, $scpiCommands, $sleepBeforeStart = 0 )
    {
        SMTLogger::getInstance()->trace(SMTUtil::PHP_CLIENT_EXECUTABLE.$phpFileToExecute.' @@'.$scpiCommands.'@@'.$sleepBeforeStart, SMTLogger::INFO);
        
        $output = array();
        passthru(SMTUtil::PHP_CLIENT_EXECUTABLE.$phpFileToExecute.' @@'.$scpiCommands.'@@'.$sleepBeforeStart.' >> '.SMTLogger::getLogFilePath().' 2>&1 & ', $output );
    }
    
    /**
     * Restart sequencer
     *
     * @param string $context
     * @param string $sleepBeforeCommand
     
     */
    static function restartSequencer(SMTContext $context, $sleepBeforeCommand)
    {
        $measurement = new SMTMeasurementInstrument();
        $measurement->setContext( $context );
        try 
        {
            if ( $measurement->acquireMeasureOnDemandLock() )  //forbid to restart sequencer whereas a measurement is setup
            {
                sleep($sleepBeforeCommand);
                
                // stop acquisition in FO
                $socketFO = $measurement->getFoSocket();
                $socketFO->send(SMTOtuApi::CMD_FO_acq_stop);
                $socketFO->close();
                
                //restart OTU sequencer
                sleep(1);
                $socket = new SMTOtuSocket( $context );
                $socket->open( SMTOtuSocket::OTU_EXTERNAL_PORT);
                
                $socket->send(SMTOtuApi::setMonitoringSequensorSuspend(FALSE));
                $socket->send(SMTOtuApi::setMonitoringSequensorStatus('SEQU'));
                
                $socket->close();
                $measurement->releaseMeasureOnDemandLock();
            }
        } 
        catch (\Exception $e) 
        {
            SMTLogger::getInstance()->traceException($e);
            $measurement->releaseMeasureOnDemandLock();
            throw $e;
        }
    }

    /**
     * Clean up measurement acquisition files (sor... pdf, json) on given port.
     * keep only 2 measurement (sor, CDM, pdf) per port + current
     *
     * 
     */
    public static function cleanupAcquisitions($portNumber)
    {
        $count = 0;
        $fileNamePrefix = sprintf(self::OTU_RESULT_FILE_PREFIX.'*', $portNumber);
        
        //retrieve files
        $resultArray = glob( '/user/'.self::OTU_RESULT_DIR.'/'.$fileNamePrefix);
        if ( $resultArray != FALSE && !empty( $resultArray ) )
        {
            $acquisitionFiles = array();
            foreach ( $resultArray as $file )
            {
                $acquisitionFiles[ $file ] = filemtime( $file );
                $count++;
            }
            
            //remove old files
            if ( $count > self::DEFAULT_MAX_ACQUISITION_COUNT )
            {
                $filesCountToRemove =  $count - self::DEFAULT_MAX_ACQUISITION_COUNT;
                asort( $acquisitionFiles, SORT_NUMERIC );
                for ( $i = 0; $i < $filesCountToRemove; $i++)
                {
                    reset( $acquisitionFiles );
                    $fileName = key( $acquisitionFiles );
                    //SMTLogger::getInstance()->trace("files to delete: $filesCountToRemove, delete file ".$fileName, SMTLogger::INFO);
                    unlink($fileName);
                    array_shift( $acquisitionFiles );
                }
            }
        }
        
    }
}

?>