<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\autotest;

use app\util\SMTInvalidValueException;

use app\services\autotest\SMTAutotestSetupDto;

use app\services\autotest\SMTAutotestDto;

use app\services\autotest\otu\SMTOtuAutotestDto;

use app\util\SMTUtil;

use app\parser\SMTOtuApi;

use app\serviceshelper\SMTServiceHelper;

use app\util\SMTLogger;


/**
 * Autotest processing
 *
 * @author Sylvain Desplat
*/
class SMTAutotest extends SMTServiceHelper
{   
    /**
     * Process otu autotest dto and save it in database
     *  
     * @param SMTOtuAutotestDto $autotestOtuDto
     */
    public function processAutotest( SMTOtuAutotestDto $autotestOtuDto )
    {
        //retrieve database connection
        $dbConnection = $this->getContext()->getDatabase();
        
        //create autotest dto from otu autotest dto:
        $autotestDto = new SMTAutotestDto();
        $autotestDto->setDurationSec( $autotestOtuDto->getDurationSec() );
        $autotestDto->setSequencerLastRunTimestamp( $autotestOtuDto->getSequencerLastRunDateSec() );
        $autotestDto->setStartTimestamp( $autotestOtuDto->getEndDate() - $autotestOtuDto->getDurationSec() );
        $autotestDto->setStatusOK( $autotestOtuDto->isAutotestSuccessful() );
        $autotestDto->setAutotestSummaryFileName( $autotestOtuDto->getAutotestSummaryFileName() );
        $autotestDto->save($dbConnection);
        
        //purge autotests older than 1 year
        SMTAutotestDto::purgeOldAutotests($dbConnection);
    }    
    
    /**
     * fetch autotests from database
     *
     * @return array
     */
    public function fetchAutotests()
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    
    	return SMTAutotestDto::fetchAll($dbConnection);
    }
    
    /**
     * Process otu autotest dto and save it in database
     *
     * @param $autotestTimeStamp
     * @return SMTOtuAutotestDto
     */
    public function fetchAutotest($autotestTimeStamp)
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    	
    	return SMTAutotestDto::fetchAutotest($dbConnection, $autotestTimeStamp);
    }
    
    
    /**
     * Update autotest settings
     * 
     * @param SMTAutotestSetupDto $autotestSetUpDto
     * 
     * @throws SMTSocketException
     * @throws SMTInvalidValueException in case of error.
     */
    public function updateAutotestSettings( SMTAutotestSetupDto $autotestSetUpDto )
    {
        $this->enableAutotest( $autotestSetUpDto->isEnabled(), $autotestSetUpDto->getAutotestDateMinutes() );
    }
        
    /**
     * Enable/disable autotest
     * In addition, if the autotest is enabled, update the autotest period to its default value
     * 
     * @param boolean $enable
     * @param integer $autotestDateMin autotest date in a day in minutes [0-1439]
     * 
     * @throws SMTSocketException
     * @throws SMTInvalidValueException in case of error.
     */
    public function enableAutotest( $enable, $autotestDateMin )
    {
        $invalidAutotestDate = FALSE;
        
        //update autotest period and autotest date:
        if ( $enable )
        {
            $this->send( SMTOtuApi::setAutotestPeriodInHourCommand( SMTAutotestSetupDto::DEFAULT_AUTOTEST_PERIOD_H ) );
            
            if ( SMTUtil::isInt( $autotestDateMin ) && $autotestDateMin >=0 && $autotestDateMin <= SMTAutotestSetupDto::AUTOTEST_MAX_DATE_MIN )
            {
                $this->send( SMTOtuApi::setAutotestDateInMinuteCommand( $autotestDateMin ) );
            }
            else
            {
                //don't throw exception here to allows to enable/disable autotest with current date value
                $invalidAutotestDate = TRUE;
                SMTLogger::getInstance()->trace( sprintf( "Invalid autotest date: %s, should be between 0 and %s.", $autotestDateMin, SMTAutotestSetupDto::AUTOTEST_MAX_DATE_MIN ), SMTLogger::ERROR );
                //$this->send( SMTOtuApi::setAutotestDateInMinuteCommand( self::DEFAULT_AUTOTEST_DATE_MIN ) );                
            }
        }
        
        //enable/disable autotest
        $this->send( SMTOtuApi::enableAutotestCommand( $enable ) );        
        
        //throw exception if autotest date is invalid but allows to enable/disable autotest with current value
        if ( $invalidAutotestDate )
        {
            throw new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "autotestDateMin", $autotestDateMin);
        }
    }
    
    /**
     * Autotest date in a day in minutes
     *
     * @return integer
     */
    public function getAutotestDateInDayInMinutes()
    {
    	return $this->sendReceive( SMTOtuApi::getAutotestDateInMinuteCommand() );
    }
    
    /**
     * Whether Autotest is activated
     * 
     * @return boolean
     */
    public function isAutotestOn()
    {
        $res = $this->sendReceive( SMTOtuApi::isAutotestEnabledCommand() );
        return strcmp( SMTOtuApi::RES_ON, $res ) == 0;
    }
}
?>