<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\admin\SMTOtuType;
use app\admin\SMTOtuVersion;

use app\serviceshelper\maintenance\SMTLicence;

use app\util\SMTInvalidValueException;

use app\events\SMTEventMessageManager;

use app\serviceshelper\system\SMTSystemConfigurationDto;

use app\serviceshelper\system\SMTDateTimeDto;

use app\SMTApplication;

use app\util\SMTUtil;

use app\serviceshelper\monitoring\SMTLinkTest;

use app\serviceshelper\system\SMTOtu;

use app\services\monitoring\SMTOtuMonitoringTestThresholdDto;
use app\serviceshelper\media\SMTSms;
use app\settings\SMTSmartOtuSettings;
use app\util\SMTLogger;

use app\admin\SMTOtuMonitoringOptions;
use app\parser\SMTOtuApi;
use app\services\SMTIService;
use app\services\SMTService;
use app\view\SMTTestRestApiPage;

date_default_timezone_set('UTC');

/**
 * Service to setup the OTU module features and display its configuration.
 * 
 * @author Sylvain Desplat
 */
class SMTSmartOtuSettingsService extends SMTService implements SMTIService
{                
    /**
     * Retrieve OTU general status (options activated, OTU status, smartOTU application status...)
     *
     * @url POST /status
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getOtuStatus()
    {            
        try
        {
    		$status = new SMTOtuStatusDto();   
    		
    		$smsOptionValid = SMTSmartOtuSettings::isSmsOptionValid();
    		if ( $smsOptionValid == NULL )
    		{    		
    		    try 
    		    {
        		    $sms = new SMTSms();
        		    $sms->setContext( $this->getContext() );
        		    $smsOptionValid = $sms->isModemOption();
        		    $smsOptionValid = $smsOptionValid && $sms->isValidModemType();
        		    //save sms option availability in cache
        		    SMTSmartOtuSettings::setSmsOptionValid( $smsOptionValid );
    		    }
    		    catch ( \Exception $e )
    		    {
    		        //catch exception to retrieve settings and status even if an error occurs
    		        $this->getContext()->getLogger()->traceException( $e );
                    //set by default sms option as not available in case of error (OTU application not available...)
    		        $smsOptionValid = FALSE;
    		    }
    		}
    		$status->setSmsOption( $smsOptionValid );
    		
    		$relayOptionValid = SMTSmartOtuSettings::isRelayOptionValid();
    		if ( $relayOptionValid== NULL )
    		{
    			try
    			{
    				$otu = new SMTOtu();
    				$otu->setContext( $this->getContext() );
    				$relayOptionValid = $otu->isRelayOption();
    				//save relay option availability in cache
    				SMTSmartOtuSettings::setRelayOptionValid( $relayOptionValid);
    			}
    			catch ( \Exception $e )
    			{
    				//catch exception to retrieve settings and status even if an error occurs
    				$this->getContext()->getLogger()->traceException( $e );
    				//set by default sms option as not available in case of error (OTU application not available...)
    				$smsOptionValid = FALSE;
    			}
    		}
    		$status->setRelayOption( $relayOptionValid );
    		
            try 
		    {
        		$linkTest = new SMTLinkTest();
        		$linkTest->setContext( $this->getContext() );   
    		    $status->setSequensorOn( $linkTest->isSequensorOn() );    
		    }
		    catch ( \Exception $e )
		    {
		        //catch exception to retrieve status even if an error occurs
		        $this->getContext()->getLogger()->traceException( $e );
		    }
		    
		    try
		    {
		    	$status->setModuleConfigOK( $this->getContext()->getModuleConfigOK() );
		    }
		    catch ( \Exception $e )
		    {
		    	//catch exception to retrieve status even if an error occurs
		    	$this->getContext()->getLogger()->traceException( $e );
		    }

        	try
		    {		        
		    	$status->setSwitchConfigOK( $this->getContext()->getSwitchConfigOK() );
		    }
		    catch ( \Exception $e )
		    {
		    	//catch exception to retrieve status even if an error occurs
		    	$this->getContext()->getLogger()->traceException( $e );
		    }
		    
    		//Retrieve SmartOTU status even if other commands fails
    		$status->setSmartOTUStatus( $this->getContext()->getSmartOTUStatus() );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $status->getJsonData();
    }    
    
    /**
     * Retrieve OTU general settings (loglevel, sequensor...)
     *
     * @url GET /
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getSmartOtuSettings()
    {
        try
        {
        	$setup = new SMTSmartOtuSettingsDto();
        
        	$linkTest = new SMTLinkTest();
        	$linkTest->setContext( $this->getContext() );
        	
        	$setup->setSequensorOn( $linkTest->isSequensorOn() );
        	$setup->setLogLevel( SMTSmartOtuSettings::getLogLevel() );
        	
        	$otuRftyMode = $this->sendReceive( SMTOtuApi::getOtuRFTYMode() );
        	$setup->setOtuRftyMode( $otuRftyMode );
        	
        	$otuInfo = $this->sendReceive( SMTOtuApi::getOtuInfo() );
        	list( $company, $otuType, $serialNumber, $otuMode, $otuReleaseVersion ) = explode(",", $otuInfo);
        	   	
        	$setup->setOtuSerialNumber( trim($serialNumber) );
        	$setup->setOtuReleaseVersion( SMTOtuVersion::getOtuVersion($this->getContext()));
        	$setup->setOtuTypeLabel( trim($otuType) );
        	$setup->setOtuType( SMTOtuType::getOtuType() );
        	
        	if (SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2())
        	{
        		if ( strcasecmp( trim( $this->sendReceive( SMTOtuApi::getModuleName( SMTModuleDto::MOD3 ) )," \t\n\r\0\"" ), SMTOtuApi::RES_NONE ) != 0 )
            	{
            		$setup->setDTSSModule( TRUE );
            	}
        	}
        	
        	try 
        	{
        		$hddStatus = $this->sendReceive( SMTOtuApi::get_harddisk_status() );
        		$setup->setHddStatus( $hddStatus );
        	}
        	catch(\Exception $ex)
        	{
        		//can fail if Reboot on HDD alarm is set to YES
        		$this->getContext()->getLogger()->traceException( $ex);
        	}
        	
        	if (SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2())
        	{
	        	try
	        	{
	        		$isBackup = $this->sendReceive(SMTOtuApi::CMD_is_Backup_Hard_Disk_Option);
	        		$isBackup = str_replace("\"","",$isBackup);
	        		$isBackupHardDiskOption = ( strcmp($isBackup, 'TRUE') == 0)? TRUE : FALSE;
	        		$setup->setHddBackup($isBackupHardDiskOption);
	        	}
	        	//catch the exception
	        	catch ( \Exception $e )
	        	{
	        		$this->getContext()->traceException($e);
	        	}
        	}
        	
        	try
        	{
        		//retrieve monitoring options
        		$monitoringOptions = SMTOtuMonitoringOptions::getOtuMonitoringOptions($this->getContext() );
        		$setup->setMonitoringOptions($monitoringOptions);
        	}
        	catch ( \Exception $e )
        	{
        		$this->handleException($e);
        	}
        	
//         	try
//         	{
//         		//retrieve monitoring mode
//         		$mode = trim( $this->sendReceive( SMTOtuApi::get_otu_mon_mode( ) ) );
//         		$setup->setMonitoringMode($mode);
//         	}
//         	catch ( \Exception $e )
//         	{
//         		//don't throw exception here, either SmartOTU will not be initialized
//         		SMTLogger::getInstance()->traceException($e);
//         	}
        	
        	$setup->setSoftwareOption(SMTLicence::retrieveSoftwareOptions($this->getContext()));
        	
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }
        
        return $setup->getJsonData();        
    }

    /**
     * Update OTU general settings (loglevel, sequensor...)
     *
     * @url PUT /
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */    		
    function setSmartOtuSettings( $request_data )    
    {
    	try
    	{
    	    $setup = SMTSmartOtuSettingsDto::getInstance( $request_data );
    	    
    	    SMTSmartOtuSettings::setLogLevel( $setup->getLogLevel(), $this->getContext() );
    	    
    	    $linkTest = new SMTLinkTest();
    	    $linkTest->setContext( $this->getContext() );    
    	    //if sequencer is not ON on OTU, put it on.	    
    	    if ( $setup->isSequensorOn() && !$linkTest->isSequensorOn() )
    	    {
    	        $linkTest->setSequensorOn();
    	        
    	        //notify link-test full refresh needed
    	        SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateAllEvent();
    	    }    	    
    	    //if OTU mode changed, update it
    	    $otuRftyMode = $this->sendReceive( SMTOtuApi::getOtuRFTYMode() );
    	    if ( $otuRftyMode !== $setup->getOtuRftyMode() )
    	    {
    	        $this->changeOtuMode( $setup->getOtuRftyMode() );
    	        
    	        //OTU is going down to reboot...
        		//set the OTU status to reboot and close the session
        		SMTApplication::handleOTUReboot( $this->getContext() );    	        
    	    }    	    
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $setup->getJsonData();
    }    
    
    /**
     * Start OTU sequencer
     *
     * @url PUT /sequencer/start
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function startOtuSequencer()
    {
    	try
    	{
    			
    		$linkTest = new SMTLinkTest();
    		$linkTest->setContext( $this->getContext() );
    		//if sequencer is not ON on OTU, put it on.
    		if ( !$linkTest->isSequensorOn() )
    		{
    			$linkTest->setSequensorOn();
    			 
    			//notify link-test full refresh needed
    			SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateAllEvent();
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $this->getSmartOtuSettings();
    }
    
    /**
     * Reset Hdd
     *
     * @url PUT /hdd/reset
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function resetHdd()
    {
    	try
    	{
    		$this->send( SMTOtuApi::get_reset_harddisk() );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	return $this->getSmartOtuSettings();
    }  
    
    
    /**
     * Update OTU date-time
     *
     * @url PUT /date
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateOtuDateTime( $request_data )
    {
    	$ntp = FALSE;
    	
    	try
    	{
    	    $dateTimeDto = SMTDateTimeDto::getInstance( $request_data );
    	    
    	    try
    	    {
    	    	$ntp = ( strcasecmp($this->sendReceive( SMTOtuApi::getNtpActive() ), 'TRUE' ) == 0 );
    	    }
    	    catch ( \Exception $e )
    	    {
    	    	$this->getContext()->getLogger()->traceException( $e);
    	    }
    	    
    	    if ( $ntp )
    	    {
    	    	$this->send( SMTOtuApi::getNtpSync() );
    	    	
    	    	//retrieve database connection
    	    	$dbConnection = $this->getContext()->getDatabase();
    	    	
    	    	//update timezone in database settings
    	    	$dto = SMTSystemConfigurationDto::fetch($dbConnection);
    	    	$dto->setTimezoneSec( 0 );
    	    	$dto->setTimezoneName( 'UTC' );
    	    	$dto->setTimezoneDst( FALSE );
    	    	$dto->save($dbConnection);
    	    	$dbConnection->close();
    	    }
    	    else 
    	    {
    	    	//update OTU date time and SmartOtu timezone
    	    	$otu = new SMTOtu();
    	    	$otu->setContext( $this->getContext() );
    	    	$otu->setOtuTimeStamp( $dateTimeDto->getTimestamp(), $dateTimeDto->getTimezoneSec(), $dateTimeDto->getTimezoneName(), $dateTimeDto->isTimezoneDst()  );
    	    }
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return self::RESULT_OK;
    }    
    
    /**
     * Update OTU date-time
     *
     * @url PUT /checkdate
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function checkOtuDateTime( $request_data )
    {
        $result = self::RESULT_OK;
        $ntp = FALSE;
    	try
    	{
    		$dateTimeDto = SMTDateTimeDto::getInstance( $request_data );
    			
            if ( $dateTimeDto != NULL )
            {
                $otu = new SMTOtu();
                $otu->setContext( $this->getContext() );
                
        		//retrieve current OTU date:
//         		$currentTimeStamp = time();//OTU timestamp with timezone offset (not in UTC)
        		
        		
        		$currentTimeStamp = $otu->getOtuTimeStamp();
        		$currentTimeZone = $otu->getOtuDateTz();
        		
        		$clientTimeStamp = SMTUtil::convertToInt( $dateTimeDto->getTimestamp() );//Client timestamp with timezone offset (not in UTC)
        		$systemConfiguration = SMTSystemConfigurationDto::fetch( $this->getContext()->getDatabase() );
        		$currentTimezoneName = $systemConfiguration->getTimezoneName();
        		$currentTimeZoneDst = $systemConfiguration->isTimezoneDst();
        		        		
        		SMTLogger::getInstance()->trace( "currentTimeStamp: ".$currentTimeStamp);
        		SMTLogger::getInstance()->trace( "currentTimezoneName: ".$currentTimezoneName);
        		SMTLogger::getInstance()->trace( sprintf("currentTimezoneDst: %b", $currentTimeZoneDst) );
        		SMTLogger::getInstance()->trace( "clientTimeStamp: ".$clientTimeStamp);
        		SMTLogger::getInstance()->trace( "clientTimezoneSec: ".$dateTimeDto->getTimezoneSec() );
        		SMTLogger::getInstance()->trace( "clientTimezoneName: ".$dateTimeDto->getTimezoneName());
        		SMTLogger::getInstance()->trace( sprintf("clientTimezoneDst: %b",$dateTimeDto->isTimezoneDst() ));
        		SMTLogger::getInstance()->trace( "currentTimezone OTU: ".$currentTimeZone * 60);
        		if ( ($currentTimeZone * 60) != $dateTimeDto->getTimezoneSec() )
        		{
        		    SMTLogger::getInstance()->trace( "currentTimezone OTU: TimeZone changed" );
        		}

        		//update OTU date time:
        		// - if it is not already set
        		// - or if timezone are the same and if the drift is > 15mn 
        		// and ntp is not activated
        		if ( !isset( $currentTimezoneName ) || ($currentTimezoneName === NULL) ||
        		     ( (abs( $currentTimeStamp - $clientTimeStamp ) >  SMTSmartOtuSettings::getDateTimeMaxDrift() )
        		       && ( strcasecmp($currentTimezoneName, $dateTimeDto->getTimezoneName() ) == 0 ) 
        		         ) || ( ($currentTimeZone * 60) != $dateTimeDto->getTimezoneSec())
        		   )
        		{
        			try
        			{
        				$ntp = ( strcasecmp($this->sendReceive( SMTOtuApi::getNtpActive() ), 'TRUE' ) == 0 );
        			}
        			catch ( \Exception $ex )
        			{
        				$this->getContext()->getLogger()->traceException( $ex);
        			}
        			
        			if ( !$ntp )
        			{
	            		//update OTU date time and SmartOtu timezone

	            		$otu->setOtuTimeStamp( $dateTimeDto->getTimestamp(), $dateTimeDto->getTimezoneSec(), $dateTimeDto->getTimezoneName(), $dateTimeDto->isTimezoneDst() );
	            		//check failed: a new date was applied on OTU
	            		$result = self::RESULT_KO;
        			}
        		}
        		$this->getContext()->getDatabase()->close();
            }
            else
            {
                throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "Invalid date time dto", "SMTDateTimeDto" );
            }
    	}
    	catch ( \Exception $e )
    	{
    		//$this->handleException($e);  don't throw exception
    	    $this->getContext()->getLogger()->traceException( $e );
    	    $result = self::RESULT_KO;
    	}
    	return $result;
    }    
    
    /**
     * Change logging level
     * 
     * @url PUT /loglevel/{logLevel}
     * 
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateLogLevel( $logLevel )
    {
        try
        {
            SMTSmartOtuSettings::setLogLevel( $logLevel, $this->getContext() );
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }
        return self::RESULT_OK;
    }        
    
    /**
     *
     * @url GET /loglevel
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchLogLevel()
    {
    	try
    	{
    	    $logLevel = SMTSmartOtuSettings::getLogLevel();  	   
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return json_encode($logLevel);
    }
    
    
    /**
     * Change OTU mode: OTU, ROTAU, OEM, ONMSI or SMARTOTU
     *
     * @url PUT /mode/{mode}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function changeOtuMode( $mode )
    {
    	try
    	{
    	    SMTLogger::getInstance()->trace("TRYING TO SWITCH FROM SMARTOTU MODE TO: ".$mode, SMTLogger::INFO );
    	    $this->send( SMTOtuApi::setOtuRFTYMode($mode) );
    	    SMTLogger::getInstance()->trace("SWITCHED SUCCESSFULLY FROM SMARTOTU MODE TO: ".$mode, SMTLogger::INFO );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return self::RESULT_OK;
    }
    
    /**
     *
     * @url GET /mode
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchOtuMode()
    {
        $otuRftyMode = SMTSmartOtuSettingsDto::OTU_MODE;
    	try
    	{
    	    $otuRftyMode = $this->sendReceive( SMTOtuApi::getOtuRFTYMode() );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return json_encode($otuRftyMode);
    }
    
    
    
    /**
     *
     * @url GET /releaseversion
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getVersion()
    {        
    	try
    	{
    	    $about = new SMTAboutDto();
    	    $smartOtuVersion = SMTSmartOtuSettings::getSmartOtuVersion();
    	    $about->setReleaseVersion( $smartOtuVersion );
    	    
    	    $otuReleaseVersion = SMTSmartOtuSettings::getOtuVersion();
    	    $about->setOtuReleaseVersion( $otuReleaseVersion );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $about->getJsonData();
    }    
    
    
    /**
     * Retrieve current memory cache info: no security control
     *
     * @url GET /apcinfo
     *
     */
    function fetchAPCInfo()
    {            
    	return apc_sma_info();
    }
    
    /**
     *
     * @url GET /budgetthresholds
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchDefaultBudgetVariationThresholds()
    {
    	try
    	{
    		$thresholds = SMTSmartOtuSettings::getDefaultBudgetVariationThresholds();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $thresholds->getJsonData();
    }    

    /**
     *
     * @url GET /tappingthresholds
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchDefaultTappingThresholds()
    {
    	try
    	{
    		$thresholds = SMTSmartOtuSettings::getDefaultTappingThresholds();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	return $thresholds->getJsonData();
    }
    
    /**
     *
     * @url GET /firstmarkerattenuation
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchDefaultFirstMarkerAttenuationThresholds()
    {
    	try
    	{
    		$thresholds = SMTSmartOtuSettings::getDefaultFirstMarkerAttenuationThresholds();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $thresholds->getJsonData();
    }    
    
    /**
     *
     * @url POST /budgetthresholds
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setDefaultBudgetVariationThresholds( $request_data )
    {
    	try
    	{
    	    $budgetVariationThresholds = SMTOtuMonitoringTestThresholdDto::getInstance( $request_data );    	    
    		SMTSmartOtuSettings::setDefaultBudgetVariationThresholds( $budgetVariationThresholds );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
    
    /**
     *
     * @url POST /firstmarkerattenuation
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setDefaultFirstMarkerAttenuationThresholds( $request_data )
    {
    	try
    	{
    	    $firstMarkerThresholds = SMTOtuMonitoringTestThresholdDto::getInstance( $request_data );    	    
    		SMTSmartOtuSettings::setDefaultFirstMarkerAttenuationThresholds( $firstMarkerThresholds );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }    
    
    /**
     * @url GET /test/updateloglevel
     */
    function testUpdateLogLevel()
    {    
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/settings/loglevel/'.SMTLogger::INFO;
        	$page = new SMTTestRestApiPage($url, "", 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }  
      
    /**
     * @url GET /test/updatefirstmarkerthreshold
     */
    function testUpdateFirstMarkerAttenuation()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/settings/firstmarkerattenuation/';
        	$thresholds = SMTSmartOtuSettings::getDefaultFirstMarkerAttenuationThresholds();
        	$json_string = json_encode( $thresholds->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     * @url GET /test/budgetthresholds
     */
    function testUpdateBudgetThresholds()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/settings/budgetthresholds/';
        	$thresholds = SMTSmartOtuSettings::getDefaultBudgetVariationThresholds();
        	$json_string = json_encode( $thresholds->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     * @url GET /test/sequencer/start
     */
    function testStartOtuSequencer()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/settings/sequencer/start/';
        	$page = new SMTTestRestApiPage($url, "", 'PUT');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
}

?>