<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\util\SMTUtil;

use app\sharedmemory\SMTIMemorySupport;

use app\sharedmemory\SMTMemoryManager;

use app\util\SMTRwDto;
use app\services\monitoring\SMTCurrentPortDto;
use app\http\SMTContext;

/**
 * Handles OTU current port when OTU is used with an external switch managed by an external application
 * 
 * @author Sylvain Desplat
 */
class SMTOtuCurrentPortDto extends SMTRwDto implements SMTIMemorySupport
{   
    /**
     *  Current port dto Identifier: return always 0 because we have only one OTU current port in cache
     */
    const CURRENT_PORT_DTO_ID = 0;
    
    /**
     * Last access to update the OTU current port
     *
     * @var float
     */
    private $lastAccess = 0;
    
    /**
     * OTU timestamp in seconds and in current timezone
     *
     * @var integer/float OTU timestamp in seconds and in current timezone
     */
    protected $date;
    
    /**
     * Alarm creation date as a string (not used just added to map info sent by OTU)
     *
     * @var string
     */
    protected $dateHuman;
    
    /**
     * Operation Id (not used just added to map info sent by OTU)
     *
     * @var string
     */
    protected $operationId;
    
    /**
     * 
     * @var integer OTU current port when used with external switch managed by external application
     */
    protected $currentPort = SMTCurrentPortDto::NO_PORT_SELECTED;
	
	/**
	 * @param string $releaseVersion
	 */
	public function setCurrentPort( $currentPort )
	{		
	    if ( self::isValid($currentPort) )
	    {
	    	$this->currentPort = intval($currentPort);
	    }	    
	    else
	    {
	    	$this->currentPort = SMTCurrentPortDto::NO_PORT_SELECTED;
	    }	    
	}
	
	/**
	 * @return int currentPort
	 */
	public function getCurrentPort()
	{
		return self::isValid($this->currentPort)? intval($this->currentPort) : SMTCurrentPortDto::NO_PORT_SELECTED;
	}
	
	/**
	 * Whether the value is a valid number
	 *
	 * @param string $value
	 * @return boolean
	 */
	public static function isValid( $value )
	{
		if ( SMTUtil::isInt( $value ) && ($value > 0) )
		{
			return TRUE;
		}
		return FALSE;
	}	
	
	/**
	 * OTU timestamp in seconds and in current timezone
	 *
	 * @return integer/float OTU timestamp in seconds and in current timezone
	 */
	public function getDate()
	{
		return ( isset( $this->date) && SMTUtil::isInt( $this->date) )? $this->date : 0;
	}
	
	/**
	 * OTU activity timestamp in UTC
	 * @param \SQLite3 $dbConnection
	 *
	 * @return integer/float otu activity timestamp in UTC
	 */
	public function getTimestampUTC( SMTContext $context )
	{
		return SMTUtil::getTimestampUTC( $this->date, $context );
	}
	
    /**
     * Serialize as Json data the role object.<br> 
     * 
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars( $this );
        $this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }    
    
    /**
     * Returns the serializable properties of the object.
     *
     * @see \app\http\SMTISerializable::getSerializableProperties()
     * 
     * @return array with the object properties to serialize.
     */
    public function getSerializableProperties()
    {
    	$objectProperties = get_object_vars($this);
    	return array_keys( $objectProperties );
    }    
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\util\SMTPaginationDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    } 
    
    /**
     * Current port dto Identifier: return always 0 because we have only one OTU current port in cache
     *
     * @return integer
     */
    public function getId()
    {
    	return self::CURRENT_PORT_DTO_ID;
    }
    
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }
    
    /**
     * Returns the dto class name.
     *
     * @return the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }
    
    /**
     * current port is never obsolete
     *
     * @return FALSE
     */
    public function isObsolete()
    {
    	return FALSE;
    }
    
    public function getLastAccess()
    {
    	return $this->lastAccess;
    }
    
    public function touchAccess()
    {
    	$this->lastAccess = microtime(true);
    }    
}


?>