<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2014. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\util\SMTUtil;

use app\util\SMTInvalidValueException;
use app\util\SMTRwDto;
use app\serviceshelper\media\SMTSnmp;

/**
 * SNMP configuration DTO. There is only 1 snmp config on an OTU.
 * @author Sylvain Desplat
 */
class SMTConfSnmpDto extends SMTRwDto
{	
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\setup\SMTSnmpDto',islist='true')
     * @var array app\services\setup\SMTSnmpDto
     */
    protected $snmpArray = array();
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\setup\SMTSnmpV3Dto',islist='true')
     * @var array app\services\setup\SMTSnmpV3Dto
     */
    protected $snmpV3Array = array();
    
	/**
	 * @var boolean
	 */
	protected $active = FALSE;
	
	/**
	 * @var boolean
	 */
	protected $snmpHeartbeatActive = FALSE;

	/**
	 * @var boolean
	 */
	protected $snmpV3 = FALSE;
	
	/**
	 * Snmp heartbeat period in mn
	 * 
	 * @var integer
	 */
	protected $snmpHeartbeatPeriod = SMTSnmp::DEFAULT_SNMP_HEART_BEAT_PERIOD;
	
	/**
	 * @param boolean
	 */
	public function setActive( $active )
	{
		$this->active = $active;
	}
	
	/**
	 * @return boolean
	 */
	public function isActive()
	{
		return $this->active;
	}	
	
	/**
	 * @param boolean
	 */
	public function setSNMPV3 ( $active )
	{
	    $this->snmpV3 = $active;
	}
	
	/**
	 * @return boolean
	 */
	public function isSNMPV3()
	{
	    return $this->snmpV3;
	}	
	
	/**
	 * @param boolean
	 */
	public function setSnmpHearbeatActive( $active )
	{
		$this->snmpHeartbeatActive = $active;
	}
	
	/**
	 * @return boolean
	 */
	public function isSnmpHearbeatActive()
	{
		return $this->snmpHeartbeatActive;
	}
	
	/**
	 * @param integer
	 */
	public function setSnmpHearbeatPeriod( $period )
	{
		if ( !isset($period) || $period === NULL || !SMTUtil::isInt( $period ) || $period <= 0 )
		{
			throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "period", $period );
		}		
		$this->snmpHeartbeatPeriod = $period;
	}
	
	/**
	 * @return integer
	 */
	public function getSnmpHearbeatPeriod()
	{
		return $this->snmpHeartbeatPeriod;
	}	
	
	/**
	 * @param array
	 */
	public function setSnmpArray( $snmpArray)
	{
		$this->snmpArray = $snmpArray;
	}
	
	/**
	 * @return array
	 */
	public function getSnmpArray()
	{
		return $this->snmpArray;
	}
	
	/**
	 * @param app\services\setup\SMTSnmpDto
	 */
	public function addSnmp( SMTSnmpDto $snmp )
	{
		array_push( $this->snmpArray, $snmp );
	}	
	
	/**
	 * 
	 * @return SMTSnmpDto
	 */
	public function getSnmp($index)
	{
		return count( $this->snmpArray ) > 0 ? $this->snmpArray[$index] : NULL ;
	}
	/**
	 * @param array
	 */
	public function setSnmpV3Array( $snmpArray)
	{
	    $this->snmpV3Array = $snmpArray;
	}
	/**
	 * @return array
	 */
	public function getSnmpV3Array()
	{
	    return $this->snmpV3Array;
	}
	
	/**
	 * @param app\services\setup\SMTSnmpDto
	 */
	public function addSnmpV3( SMTSnmpV3Dto $snmp )
	{
	    array_push( $this->snmpV3Array, $snmp );
	}
	
	/**
	 *
	 * @return SMTSnmpDto
	 */
	public function getSnmpV3($index)
	{
	    return count( $this->snmpV3Array ) > 0 ? $this->snmpV3Array[$index] : NULL ;
	}
	/**
	 * Build new DTO
	 * 
	 * @param boolean $isActive
	 * @param boolean $isHeartbeat
	 * @param integer $heartbeatPeriod
	 * @return \app\services\setup\SMTConfSnmpDto
	 */
	public static function buildSMTConfSnmpDto( $isActive, $isHeartbeat, $heartbeatPeriod, $snmpV3 )
	{
	    $snmpConfDto =  new SMTConfSnmpDto();
	    $snmpConfDto->setActive( $isActive );
	    $snmpConfDto->setSnmpHearbeatActive( $isHeartbeat );
	    $snmpConfDto->setSnmpHearbeatPeriod( $heartbeatPeriod );
	    $snmpConfDto->setSNMPV3( $snmpV3 );
	    
	    return $snmpConfDto;
	}
	
	/**
	 * Test whether sms media is active on OTU
	 *
	 * @return boolean
	 */
	public function isSnmpActive()
	{
	    return $this->isActive() && ((count( $this->snmpArray ) > 0) || (count( $this->snmpV3Array ) > 0));
	}
	
	
	public function stripHtmlTags()
	{
		foreach ( $this->snmpArray as $snmpDto )
		{
			$snmpDto->stripHtmlTags();
		}
		foreach ( $this->snmpV3Array as $snmpDto )
		{
		    $snmpDto->stripHtmlTags();
		}
	}
	
	/**
	 * Serialize as Json data the user object.<br>
	 *
	 * @return $objectProperties the object properties serialized as a Json string.
	 */
	function getJsonData()
	{
		$objectProperties = get_object_vars($this);
		$this->serializeObjectAsJsonData( $objectProperties );
		return $objectProperties;
	}

	/**
	 * Returns Dto class name.
	 *
	 * @return string the dtop class name
	 */
	function getDtoClassName()
	{
		return get_class();
	}

	/**
	 * Create the Dto from a Json Dto in an array
	 *
	 * @param array $json_array json data in an array
	 *
	 * @return \app\services\setup\SMTConfSnmpDto
	 */
	static function getInstance( array &$json_array )
	{
		$instance = self::forge( get_class(), $json_array );
		$instance->stripHtmlTags();
		return $instance;
	}
}

?>