<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\otdr;

use app\serviceshelper\otdr\SMTCommonMeasurement;

use app\util\SMTInvalidValueException;

use app\util\SMTUtil;

use app\util\SMTRwDto;

/**
 * OTDR acquisition parameters of an OTDR measurement.
 * 
 * @author Sylvain Desplat
 */
class SMTAcquisitionParametersDto extends SMTRwDto
{
    /**
     * Default acquisition time: AUTO
     * @var int
     */
    const DEFAULT_ACQUISITION_TIME = -2; //AUTO
    /**
     * Default pulse: 10ns
     * @var int
     */
    const DEFAULT_PULSE_NS = 10;
    
    /**
     * @var int
     */    
    protected $acquisitionTimeSec = NULL;
    /**
     * @var int
     */    
    protected $wavelengthNm = NULL;
    /**
     * @var int
     */
    protected $dwdmFrequencyGhz = NULL;
    /**
     * @var int
     */    
    protected $pulseNs = NULL;
    /**
     * @var float
     */    
    protected $rangeKm = NULL;
    /**
     * @var int
     */    
    protected $resolutionCm = NULL;    
    /**
     * @var float
     */
    protected $fiberIndex = NULL;
        
    /**
     * @param integer $acquisitionTimeSec
     * @throws SMTInvalidValueException
     */
    public function setAcquisitionTimeSec( $acquisitionTimeSec )
    {
        if ( !isset($acquisitionTimeSec) || $acquisitionTimeSec === NULL || !SMTUtil::isInt( $acquisitionTimeSec ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "duration", $acquisitionTimeSec );
        }        
    	$this->acquisitionTimeSec = intval($acquisitionTimeSec);
    }
    
    /**
     * @return integer
     */
    public function getAcquisitionTimeSec()
    {
    	return $this->checkAcquisitionTimeSec();
    }    
    /**
     * @param integer setDwdmFrequencyGhz
     * @throws SMTInvalidValueException
     */
    public function setDwdmFrequencyGhz( $dwdmFrequencyGhz )
    {
    	if ( !isset($dwdmFrequencyGhz) || $dwdmFrequencyGhz=== NULL || !SMTUtil::isInt( $dwdmFrequencyGhz) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "dwdmFrequencyGhz", $dwdmFrequencyGhz);
        }        
        $this->dwdmFrequencyGhz= intval($dwdmFrequencyGhz);
    }
    
    /**
     * @return integer or NULL
     */
    public function getDwdmFrequencyGhz()
    {
    	return $this->dwdmFrequencyGhz!== NULL ? intval($this->dwdmFrequencyGhz) : NULL;
    }
    
    /**
     * @param integer $wavelengthNm
     * @throws SMTInvalidValueException
     */
    public function setWavelengthNm( $wavelengthNm )
    {
    	if ( !isset($wavelengthNm) || $wavelengthNm === NULL || !SMTUtil::isInt( $wavelengthNm ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "wavelength", $wavelengthNm );
    	}
    	$this->wavelengthNm = intval($wavelengthNm);
    }
    
    /**
     * @return integer or NULL
     */
    public function getWavelengthNm()
    {
    	return $this->wavelengthNm !== NULL ? intval($this->wavelengthNm) : NULL;
    }
    
    /**
     * @param integer $pulseNs
     * @throws SMTInvalidValueException
     */
    public function setPulseNs( $pulseNs )
    {
        if ( !isset($pulseNs) || $pulseNs === NULL || !SMTUtil::isInt( $pulseNs ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "pulse", $pulseNs );
        }
        
    	$this->pulseNs = intval($pulseNs);
    }
    
    /**
     * @return integer
     */
    public function getPulseNs()
    {
    	return $this->checkPulseNs();
    }

    /**
     * @param float $rangeKm
     * @throws SMTInvalidValueException
     */
    public function setRangeKm( $rangeKm )
    {
        if ( !isset($rangeKm) || $rangeKm === NULL || !is_numeric( $rangeKm ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "range", $rangeKm );
        }        
    	$this->rangeKm = floatval($rangeKm);
    }
    
    /**
     * @return float or NULL
     */
    public function getRangeKm()
    {
        return $this->rangeKm !== NULL ? floatval($this->rangeKm) : NULL;
    }
    /**
     * @param integer $resolutionCm
     * @throws SMTInvalidValueException
     */
    public function setResolutionCm( $resolutionCm )
    {
        if ( !isset($resolutionCm) || $resolutionCm === NULL || !SMTUtil::isInt( $resolutionCm ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "resolution", $resolutionCm );
        }        
    	$this->resolutionCm = intval($resolutionCm);
    }
    
    /**
     * @return Integer or NULL
     */
    public function getResolutionCm()
    {
        return $this->resolutionCm !== NULL ? intval($this->resolutionCm) : NULL;
    }    
    
    /**
     * @param float $fiberIndex
     * @throws SMTInvalidValueException
     */
    public function setFiberIndex( $fiberIndex )
    {
    	if ( !isset($fiberIndex) || $fiberIndex === NULL || !is_numeric( $fiberIndex ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "fiberIndex", $fiberIndex );
    	}
    	$this->fiberIndex = floatval($fiberIndex);
    }
    
    /**
     * @return float
     */
    public function getFiberIndex()
    {
    	return $this->checkFiberIndex();
    }    
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }    
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }    
    
    /**
     * Fix acquisition time if its value is in valid. (0 in the case of automatic acquisition)
     *  
     * @return int The acquisition time or the default value if its value is invalid.
     */
    private function checkAcquisitionTimeSec()
    {
        return ($this->acquisitionTimeSec <= 0 )? self::DEFAULT_ACQUISITION_TIME : $this->acquisitionTimeSec;
    }
    
    /**
     * Fix pulse if its value is invalid. (0 in the case of automatic acquisition)
     *
     * @return integer The pulse value or the default value if its value is invalid.
     */
    private function checkPulseNs()
    {
        return ($this->pulseNs <= 0 )? self::DEFAULT_PULSE_NS : $this->pulseNs;
    }  

    /**
     * Fix fiber index value if its value is invalid.
     *
     * @return integer The fiber index value or the default value if its value is invalid (<=0).
     */
    private function checkFiberIndex( )
    {
    	return (!isset($this->fiberIndex) || $this->fiberIndex == NULL || $this->fiberIndex <= 0 )? SMTCommonMeasurement::DEFAULT_FIBER_INDEX : floatval( $this->fiberIndex );
    }    
}
?>