<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of Viavi Solution
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from Viavi Solution
// Copyright Viavi Solution2019. All rights reserved.
// *********************************************************
namespace app\services\api;

use app\parser\SMTOtuApi;
use app\services\security\SMTAccessControl;
use app\services\SMTIService;
use app\services\SMTService;

use app\error\SMTRestException;
use app\events\operations\SMTOperationEventDto;
use app\http\SMTHttpResponse;
use app\serviceshelper\monitoring\SMTLinkTest;
use app\serviceshelper\monitoring\SMTMeasureOnDemandException;
use app\serviceshelper\monitoring\SMTTestOnDemand;
use app\serviceshelper\monitoring\SMTTestOnDemandException;
use app\util\SMTInvalidValueException;
use app\util\SMTUtil;
use app\serviceshelper\otdr\SMTMeasurementInstrument;
use app\serviceshelper\otdr\SMTOtdrTraceType;
use app\util\SMTLogger;
use app\view\SMTMeasureResultFileDownloadPage;


/**
 * Public API service to execute OTU port operations.
 *
 * @author Sylvain Desplat
 */
class SMTApiPortService extends SMTService implements SMTIService
{
	
	/**
	 *
	 * @url GET /count
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getOtuPortCount()
	{
		$this->checkApi();
		$portDto = new SMTApiPortCountDto();
		try
		{	
			$linkHelper = new SMTLinkTest();
			$linkHelper->setContext( $this->getContext() );
			
			//retrieve the number of ports available:
			$portCount = $linkHelper->getPortsCount();
			$portDto->setPortCount($portCount);
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve ports count');
		}
		$this->getContext()->closeSession('api');
		
		return $portDto->getJsonData();
	}
	
	/**
	 * 
	 * @url POST /{portNumber}/testOnDemand
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function startTestOnDemandOnPort( $portNumber )
	{
		$this->checkApi();
		$testStatusDto = new SMTApiTestOnDemandStatusDto();
		$testId = NULL;
		try
		{		
			//test if a link exists:
			$linkId = $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand( intval($portNumber), intval($portNumber) ) );
			
			SMTLinkTest::checkLinkTestId($linkId);
		
			//retrieve test identifier:
			$linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId) );
			$testId = SMTLinkTest::parseLinkTestId( $linkTestIds );
		}
		catch ( \Exception $e )
		{
			$testId = NULL;
			$this->getContext()->getLogger()->traceException($e);
		}
		if ( ($testId == NULL) || ($testId < 0) )
		{
			$testStatusDto->setStatus(SMTApiTestOnDemandStatusDto::NO_TEST);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, $testStatusDto->getStatus(), 'No test available on port '.$portNumber);
		}
		else
		{
			try
			{
				$testOnDemandHelper = new SMTTestOnDemand();
				$testOnDemandHelper->setContext( $this->getContext() );
				$status = $testOnDemandHelper->startTest( $testId, $linkId, TRUE );
			}
			catch( SMTTestOnDemandException $ex )
			{
				$this->getContext()->getLogger()->traceException($ex);
				if ( strcasecmp($ex->getErrorCode(), SMTOperationEventDto::ERROR_ONE_TEST_ON_DEMAND) == 0 )
				{
					$testStatusDto->setStatus(SMTApiTestOnDemandStatusDto::TEST_ALREADY_RUNNING);
					$message = 'One test on demand already runnning';
				}
				else
				{
					$testStatusDto->setStatus(SMTApiTestOnDemandStatusDto::FAILURE);
					$message = 'Failed to start test on demand on port '.$portNumber;
				}
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $testStatusDto->getStatus(), $message);
			}
			catch ( \Exception $e )
			{
				$this->getContext()->getLogger()->traceException($e);
				$testStatusDto->setStatus(SMTApiTestOnDemandStatusDto::FAILURE);
				$message = 'Failed to start test on demand on port '.$portNumber;
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $testStatusDto->getStatus(), $message);
			}
			
			if ( $status != NULL && !$status->isOperationRunning() )
			{
				$testStatusDto->setStatus(SMTApiTestOnDemandStatusDto::FAILURE);
				$message = 'Failed to start test on demand on port '.$portNumber;
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $testStatusDto->getStatus(), $message);
			}
		}
		$this->getContext()->closeSession('api');
		
		return $testStatusDto->getJsonData();
	}
	
	/**
	 * 
	 * @url GET /{portNumber}
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getPort($portNumber)
	{
		$this->checkApi();
		$linkDto = new SMTApiLinkDto();
		$linkInfo = NULL;
		try
		{
			$linkInfo= $this->sendReceive( SMTOtuApi::getPortConfig($portNumber) );
			if ( ($linkInfo == NULL) || (strlen($linkInfo) <= 0) )
			{
				SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'No monitoring on port '.$portNumber);
			}
			
			list( $portNumber, $name, $monitored ) = explode(",", $linkInfo);
			
			$linkDto->setPortNumber( trim($portNumber) );
			$linkDto->setName( trim($name));
			$linkDto->setMonitored(trim($monitored));

// 			$linkDto->setPortNumber($portNumber);
// 			$linkHelper = new SMTLinkTest();
// 			$linkHelper->setContext( $this->getContext() );
			
// 			$linkId = $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand( intval($portNumber), intval($portNumber) ) );
			
// 			if ( ($linkId != NULL) && (strlen($linkId) > 0) )
// 			{
// 				$linkInfo = $this->sendReceive( SMTOtuApi::getlinkUtf8Command( $linkId ), FALSE );
				
// 				if ( ($linkInfo != NULL) && (strlen($linkInfo) > 0))
// 				{
// 					//look for  ","  sequence to explode the string into an array $conf
// 					//remove spaces and double quotes in array values
// 					$linkInfoArray = SMTUtil::splitStringByComma( $linkInfo );
// 					$name = trim( $linkInfoArray[SMTLinkTest::LINK_UTF8_NAME_INDEX] );
// 					if ( strlen($name) <= 0 )
// 					{
// 						$name = trim( $linkInfoArray[SMTLinkTest::LINK_NAME_INDEX] );
// 					}
// 					$linkDto->setName( $name );
					
					
// 					//retrieve test identifier:
// 					$linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) );
// 					$testId = SMTLinkTest::parseLinkTestId( $linkTestIds );
// 					if ( $testId !== NULL && $testId >= 0 )
// 					{
// 						$isMonitoringAllowed = $linkHelper->isMonitoringAllowed( $testId);
// 						$linkDto->setMonitored( $isMonitoringAllowed );
// 					}
// 				}
// 			}
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'No monitoring on port '.$portNumber);
		}
		$this->getContext()->closeSession('api');
		
		return $linkDto->getJsonData();
	}
	
	/**
	 *
	 * @url GET /
	 * @param int $start
	 * @param int $count
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getPorts($start = 1, $count = 1)
	{
		$this->checkApi();
		
		try 
		{
			$linkListDto = new SMTApiLinkListDto();
			$linkHelper = new SMTLinkTest();
			$linkHelper->setContext( $this->getContext() );
		
			
			$start = intval($start);
			$end = $start + intval($count) - 1;
			$portCount = $linkHelper->getPortsCount();
			
			if ( $start > $portCount)
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "start > port count", "start", $start);
				
			}
			if ( $start > $end)
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "start > end", "end", $end);
				
			}
			if ( $end> $portCount)
			{
				$end = $portCount;
			}
			
			$ports = explode(",", $this->sendReceive( SMTOtuApi::getLinkPortListPerPortCommand( $start, $end ) ) );
			if ( ($ports!= false) && (count($ports) > 0) )
			{
				foreach ( $ports as $portNumber)
				{
					try
					{
						if ( $portNumber > 0 )
						{
							$linkInfo= $this->sendReceive( SMTOtuApi::getPortConfig($portNumber) );
							if ( ($linkInfo != NULL) && (strlen($linkInfo) > 0) )
							{	
								$linkApiDto = new SMTApiLinkDto();
								list( $portNumber, $name, $monitored ) = explode(",", $linkInfo);
								
								$linkApiDto->setPortNumber( trim($portNumber) );
								$linkApiDto->setName( trim($name));
								$linkApiDto->setMonitored(trim($monitored));
								$linkListDto->addLink($linkApiDto);
							}
						}
					}
					catch ( \Exception $exp )
					{
						$this->getContext()->getLogger()->traceException($exp);
					}
				}
			}
		}
		catch( SMTInvalidValueException $ex)
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid ports $start -> $end : '.$ex->getMessage() );
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve ports $start -> $end');
		}
			
// 			$linkIds = explode(",", $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand( $start, $end ) ) );
			
// 			if ( ($linkIds != false) && (count($linkIds) > 0) )
// 			{
// 				foreach ( $linkIds as $linkId)
// 				{
// 					if ( SMTLinkTest::checkLinkTestId( $linkId, FALSE) )
// 					{
// 						$linkApiDto = new SMTApiLinkDto();
						
// 						//retrieve link info
// 						try
// 						{
// 							$linkDto = $linkHelper->parseLinkInfo( $linkId);
// 						}
// 						catch ( \Exception $e )
// 						{
// 							throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_NOT_FOUND, sprintf("Link not found %s!", $linkId) );
// 						}
						
// 						if ( $linkDto!= NULL) //&& (strlen($linkInfo) > 0))
// 						{
// 							$linkApiDto->setName($linkDto->getName() );
// 							$linkApiDto->setPortNumber($linkDto->getPortNumber() );
							
// 							//retrieve test identifier:
// 							$linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) );
// 							$testId = SMTLinkTest::parseLinkTestId( $linkTestIds );
// 							if ( $testId !== NULL && $testId >= 0 )
// 							{
// 								$isMonitoringAllowed = $linkHelper->isMonitoringAllowed( $testId);
// 								$linkApiDto->setMonitored( $isMonitoringAllowed );
// 							}
// 							$linkListDto->addLink($linkApiDto);
// 						}
// 					}
// 				}
// 			}
			
// 		}
// 		catch( SMTInvalidValueException $ex)
// 		{
// 			$this->getContext()->getLogger()->traceException($e);
// 			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid ports $start -> $end : '.$ex->getMessage() );
// 		}
// 		catch ( \Exception $e )
// 		{
// 			$this->getContext()->getLogger()->traceException($e);
// 			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve ports $start -> $end');
// 		}
        $this->getContext()->closeSession('api');
		return $linkListDto->getJsonData();
	}	
	
	
	/**
	 *
	 * @url POST /{portNumber}
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function updatePort($portNumber, $request_data)
	{
		$this->checkApi();
		
		$linkDto = SMTApiLinkDto::getInstance( $request_data );
		try
		{
			$linkName = SMTUtil::escapeShellArgUtf8WithoutQuotes($linkDto->getName());
			$this->send( SMTOtuApi::updatePortConfig($portNumber,$linkName, $linkDto->isMonitored() ) );
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to update link name and monitoring status');
		}
		return $this->getPort($linkDto->getPortNumber());
	}
	
	/**
	 *
	 * @url GET /{portNumber}/alarms
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getPortAlarmStatus($portNumber)
	{
		$this->checkApi();
		
		$portAlarmStatusDto = new SMTApiPortAlarmStatusDto();
		$portAlarmStatus = NULL;
		
		try
		{
			$portAlarmStatus= $this->sendReceive( SMTOtuApi::getPortAlarmStatus(intval($portNumber) ) );
			
			list( $severity, $timestamp, $alarmType, $distance, $probableCause ) = explode(",", $portAlarmStatus);
			
			$portAlarmStatusDto->setPortNumber( trim($portNumber) );
			$portAlarmStatusDto->setSeverity( trim($severity) );
			$portAlarmStatusDto->setTimestamp( trim($timestamp) );
			$portAlarmStatusDto->setType( trim($alarmType) );
			$portAlarmStatusDto->setDistance(trim($distance));
			$portAlarmStatusDto->setProbableCause(trim($probableCause));
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'No monitoring on port '.$portNumber);
		}
		$this->getContext()->closeSession('api');
		
		return $portAlarmStatusDto->getJsonData();
	}
	
	/**
	 *
	 * @url GET /alarms
	 * @param int $start
	 * @param int $count
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getPortAlarmStatusList($start = 1, $count = 1)
	{
		$this->checkApi();
		
		try
		{
			$portAlarmStatusListDto = new SMTApiPortAlarmStatusListDto();
			$linkHelper = new SMTLinkTest();
			$linkHelper->setContext( $this->getContext() );
			
			$start = intval($start);
			$end = $start + intval($count) - 1;
			$portCount = $linkHelper->getPortsCount();
			
			if ( $start > $portCount)
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "start > port count", "start", $start);
				
			}
			if ( $start > $end)
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "start > end", "end", $end);
				
			}
			if ( $end> $portCount)
			{
				$end = $portCount;
			}
			
			$ports = explode(",", $this->sendReceive( SMTOtuApi::getLinkPortListPerPortCommand( $start, $end ) ) );
			if ( ($ports!= false) && (count($ports) > 0) )
			{
				foreach ( $ports as $portNumber)
				{
					try
					{
						if ( $portNumber > 0 )
						{
							$portAlarmStatus= $this->sendReceive( SMTOtuApi::getPortAlarmStatus(intval($portNumber) ) );
							
							list( $severity, $timestamp, $alarmType, $distance, $probableCause) = explode(",", $portAlarmStatus);
							
							$portAlarmStatusDto = new SMTApiPortAlarmStatusDto();
							$portAlarmStatusDto->setPortNumber( trim($portNumber) );
							$portAlarmStatusDto->setSeverity( trim($severity) );
							$portAlarmStatusDto->setTimestamp( trim($timestamp) );
							$portAlarmStatusDto->setType( trim($alarmType) );
							$portAlarmStatusDto->setDistance(trim($distance));
							$portAlarmStatusDto->setProbableCause(trim($probableCause));
							
							$portAlarmStatusListDto->addAlarm($portAlarmStatusDto);
						}
					}
					catch ( \Exception $exp )
					{
						$this->getContext()->getLogger()->traceException($exp);
					}
				}
			}
		}
		catch( SMTInvalidValueException $ex)
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid ports $start -> $end : '.$ex->getMessage() );
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve alarms status for ports $start -> $end');
		}
		
		$this->getContext()->closeSession('api');
		
		return $portAlarmStatusListDto->getJsonData();
	
	}
	
	/**
	 *
	 * @url POST /{portNumber}/alarm/replay
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function replayAlarm($portNumber)
	{
		$this->checkApi();
		
		$statusDto = new SMTApiStatusDto();
		
		try
		{
			$this->send( SMTOtuApi::replayAlarm(intval($portNumber) ) );
			$statusDto->setStatus(SMTApiStatusDto::SUCCESS);
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'No optical alarm on port '.$portNumber);
		}
		$this->getContext()->closeSession('api');
		
		return $statusDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /{portNumber}/alarm/clear
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function clearAlarm($portNumber)
	{
		$this->checkApi();
		
		$statusDto = new SMTApiStatusDto();
		try
		{
			$this->send( SMTOtuApi::clearAlarm(intval($portNumber) ) );
			$statusDto->setStatus(SMTApiStatusDto::SUCCESS);
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'failed to clear alarm on port '.$portNumber);
		}
		$this->getContext()->closeSession('api');
		
		return $statusDto->getJsonData();
	}
	
	
	/**
	 *
	 * @url POST /{portNumber}/measureOnDemand/status
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getMeasureOnDemandOnPortStatus( $portNumber )
	{
	    $measureStatusDto = new SMTApiMeasureInstrumentStatusDto();
	    $measureOnDemandHelper = new SMTMeasurementInstrument(  );
	    $measureOnDemandHelper->setContext( $this->getContext() );
	    try 
	    {
	        $measureStatusDto = $measureOnDemandHelper->getMeasurementStatusFromPort( $portNumber );
    	}
    	catch ( \Exception $e )
    	{
    	    $this->getContext()->getLogger()->traceException($e);
    	    $measureStatusDto->setStatus(SMTApiMeasureInstrumentStatusDto::FAILURE);
    	    $message = 'Failed to start measure on demand on port '.$portNumber;
    	    SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $measureStatusDto->getStatus(), $message);
    	}
	    
	    $this->getContext()->closeSession('api');
	    
	    return $measureStatusDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /{portNumber}/measureOnDemand
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function startMeasureOnDemandOnPort( $portNumber, $request_data )
	{
	    $this->checkApi();
	    $port = intval($portNumber);
	    $measureStatusDto = new SMTApiMeasureInstrumentStatusDto();
	    $measurementDto = SMTApiMeasurementDto::getInstance( $request_data );
	    
	    $linkHelper = new SMTLinkTest();
	    $linkHelper->setContext( $this->getContext() );
	    $portCount = $linkHelper->getPortsCount();
	    
	    if ( ($port > $portCount) || ($port < 1) )
	    {
	        SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, "Invalid port number ".$port);	        
	    }

        try
        {            
            $measureOnDemandHelper = new SMTMeasurementInstrument(  );
            $measureOnDemandHelper->setContext( $this->getContext() );
            $status = $measureOnDemandHelper->startMeasure( $measurementDto, $port );
        }
        catch( SMTMeasureOnDemandException $ex )
        {
            $this->getContext()->getLogger()->traceException($ex);
            if ( strcasecmp($ex->getErrorCode(), SMTOperationEventDto::ERROR_ONE_MEASURE_ON_DEMAND) == 0 )
            {
                $measureStatusDto->setStatus(SMTApiMeasureInstrumentStatusDto::MEASURE_ALREADY_RUNNING);
                $message = 'One measure already runnning';
            }
            else
            {
                $measureStatusDto->setStatus(SMTApiMeasureInstrumentStatusDto::FAILURE);
                $message = 'Failed to start measure on demand on port '.$portNumber;
            }
            SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $measureStatusDto->getStatus(), $message);
        }
        catch ( \Exception $e )
        {
            $this->getContext()->getLogger()->traceException($e);
            $measureStatusDto->setStatus(SMTApiMeasureInstrumentStatusDto::FAILURE);
            $message = 'Failed to start measure on demand on port '.$portNumber;
            SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $measureStatusDto->getStatus(), $message);
        }
        
        if ( $status != NULL && !$status->isOperationRunning() )
        {
            $measureStatusDto->setStatus(SMTApiMeasureInstrumentStatusDto::FAILURE);
            $message = 'Failed to start measure on demand on port '.$portNumber;
            SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $measureStatusDto->getStatus(), $message);
        }
	    $this->getContext()->closeSession('api');
	    
	    return $measureStatusDto->getJsonData();
	}
	
	
	
	/**
	 * Retrieve OTDR acquisition, CDM & json file
	 * 
	 * @url POST /{portNumber}/measureOnDemand/result
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function downloadResultFile($portNumber, $request_data)
	{
	    SMTLogger::getInstance()->trace( sprintf("Retrieve measurement file for type %s", json_encode($request_data)), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);

	    $port = intval($portNumber);
	    $resultType = trim($request_data['resultType'],"\"");
	    
	    $linkHelper = new SMTLinkTest();
	    $linkHelper->setContext( $this->getContext() );
	    $portCount = $linkHelper->getPortsCount();
	    
	    if ( ($port > $portCount) || ($port < 1) )
	    {
	        SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, "Invalid port number ".$port);
	    }
	    
	    try
	    {
	        if ( SMTOtdrTraceType::isMeasurementResultFileType( $resultType) )
	            {
	                switch ($resultType)
	                {
	                    case SMTOtdrTraceType::ACQUISITION_FILE:
	                        $traceFilePath = SMTMeasurementInstrument::getTraceFileName($port);
	                        break;
	                    case SMTOtdrTraceType::CDM_FILE:
	                        $traceFilePath = SMTMeasurementInstrument::getCDMFileName($port);
	                        break;
	                    case SMTOtdrTraceType::PDF_FILE:
	                        $traceFilePath = SMTMeasurementInstrument::getPdfFileName($port);
	                        break;
	                    default:
	                        $measureStatusDto = new SMTApiMeasureInstrumentStatusDto( SMTApiMeasureInstrumentStatusDto::FAILURE, "invalid result type ".$resultType );
	                        SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $measureStatusDto->getStatus(), "invalid result type ".$resultType);
	                        break;
	                }
	                
	                SMTLogger::getInstance()->trace( sprintf("Retrieve measurement file %s, for type %s ",  $traceFilePath, $resultType), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
	                $page = new SMTMeasureResultFileDownloadPage( $traceFilePath );
	                $this->getContext()->getHttpResponse()->send( $page, FALSE );

	                //delete file downloaded before exit
	                SMTMeasurementInstrument::deleteFile($traceFilePath);
	                
	                $this->getContext()->closeSession('api');

	                exit(0);
	            }
	            else
	            {
	                $measureStatusDto = new SMTApiMeasureInstrumentStatusDto( SMTApiMeasureInstrumentStatusDto::FAILURE, "invalid result type ".$resultType );
	                SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, $measureStatusDto->getStatus(), "invalid result type ".$resultType);
	            }
	    }
	    catch ( SMTException $e )
	    {
	        $ex = new SMTRestException( SMTRestException::SERVICE_NOT_FOUND , $e->getMessage()." ".$e->getDetailedMessage(), $e->getCode(), $e->getTrace() );
	        $this->handleException($ex);
	    }
	    catch ( SMTException $catchAll )
	    {
	        $ex = new SMTRestException( SMTRestException::SERVICE_NOT_FOUND , $catchAll->getMessage(), $catchAll->getCode(), $catchAll->getTrace() );
	        $this->handleException($ex);
	    }
	}
	
}

