<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\database\SMTIExecuteInTransaction;

use app\serviceshelper\alarm\SMTAlarmSeverity;

use app\database\SMTSmartOtuDB;

use app\util\SMTLogger;

use app\database\SMTDatabaseException;

use app\util\SMTResultSetDto;

use app\util\SMTInvalidValueException;

use app\util\SMTUtil;

/**
 * Base alarm dto class.
 * An alarm is defined by its specific problem and its originating resource;
 * That is encoded in the alarm discriminator.
 * 
 * @author Sylvain Desplat
 */
abstract class SMTAlarmDto extends SMTResultSetDto
{   
    /**
     * Alarm discriminator = specific problem + originating resource
     * 
     * @var string
     */
    protected $discriminator;
    
    /**
     * Alarm severity
     * 
     * @var integer
     */
    protected $severity = SMTAlarmSeverity::SEVERITY_INFO;

    /**
     * Alarm clear status
     * WARNING: boolean values cannot be saved in SqLite3
     * @var integer
     */
    protected $clearStatus = SMTAlarmSeverity::UNCLEARED_STATUS;
    
    /**
     * UTC date time in seconds
     * 
     * @var integer/float
     */
    protected $lastUpdate = 0;

    /**
     * The specific problem code returned by OTU. That specific problem is shared by all events referencing to that alarms.
     * 
     * @var string
     */
    protected $specificProblemCode;  
    
    /**
     * The specific problem text built from specific problem code. It is localised in client language.
     * WARNING NOT bound in database
     * 
     * @var string
     */
    protected $specificProblemText;
    
    /**
     * Alarm type (//tOtu_Alarm_Event_Type)
     * 
     * @var string
     */
    protected  $alarmType;
    
    const FETCH_ALARM_SEQUENCE = "SELECT sequence from alarm_sequence";
    
    const INCREMENT_ALARM_SEQUENCE = "UPDATE alarm_sequence set sequence = sequence + 1";
    
    /**
     * Generate new alarm Id (moke database sequence)
     * WARNING Should be called from a critical section!
     *
     * @param \SQLite3 $dbConnection
     *
     * @return integer
     */
    public function generateId( SMTSmartOtuDB $dbConnection )
    {            	    
        SMTLogger::getInstance()->trace( "Generate new id for alarm: ".$this->getDiscriminator(), SMTLogger::DEBUG );
        $query = self::INCREMENT_ALARM_SEQUENCE;
        $success = $dbConnection->execWithTrace( $query );
    	
    	if ( $success === FALSE )
    	{
    		SMTLogger::getInstance()->trace( "Query failed:: ".$query , SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	
    	$query = self::FETCH_ALARM_SEQUENCE;
    	$sequence = $dbConnection->querySingleWithTrace( $query );
    	
    	if ( $sequence == NULL || $sequence === FALSE )
    	{
    		SMTLogger::getInstance()->trace( "Query failed:: ".$query , SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	
    	return $sequence;
    }
    
    /**
     *
     * @return SMTAlarmEventDto
     */
    public abstract function getCurrentAlarmEvent();
    
    /**
     * Set whether the alarm is cleared
     *
     * @param boolean $isCleared
     */
    public function setClearStatus( $isCleared )
    {
    	$this->clearStatus = SMTAlarmSeverity::fromOtuClearStatusToSmartOtuClearStatus( $isCleared );
    }
    
    /**
     * Whether the alarm is cleared
     *
     * @return boolean
     */
    public function isCleared()
    {
    	return SMTAlarmSeverity::isAlarmCleared( $this->clearStatus );
    }
    
    
    /**
     * Set the alarm type
     *
     * @param string $alarmType
     */
    public function setAlarmType( $alarmType )
    {
    	$this->alarmType = $alarmType;
    }
    
    /**
     * Alarm type
     *
     * @return string
     */
    public function getAlarmType()
    {
    	return $this->alarmType;
    }
    
    /**
     * Set the alarm discriminator
     *
     * @param string $discriminator
     */
    public function setDiscriminator( $discriminator )
    {
    	$this->discriminator = $discriminator;
    }
    /**
     * Alarm discriminator
     *
     * @return string
     */
    public function getDiscriminator()
    {
    	return $this->discriminator;
    }    
    
    /**
     *
     * @param integer  $severity
     */
    public function setSeverity( $severity )
    {
    	$this->severity = $severity;
    }
    /**
     *
     * @return integer
     */
    public function getSeverity()
    {
    	return $this->severity;
    }

    /**
     * Set the alarm UTC date time in seconds
     * 
     * @param integer $lastUpdate alarm UTC date time in seconds
     */
    public function setLastUpdate( $lastUpdate )
    {
        if ( !isset($lastUpdate) || $lastUpdate === NULL || !SMTUtil::isInt( $lastUpdate ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "lastUpdate", $lastUpdate );
        }
        
    	$this->lastUpdate = SMTUtil::convertToInt( $lastUpdate );
    }
    /**
     * Get the alarm UTC date time in seconds
     * 
     * @return integer/float alarm UTC date time in seconds
     */
    public function getLastUpdate()
    {
    	return $this->lastUpdate;
    }
    
    /**
     * Set the alarm specific problem code. 
     *
     * @param string $specificProblemCode
     */
    public function setSpecificProblemCode( $specificProblemCode )
    {
        $this->specificProblemCode = $specificProblemCode;
    }
    
    /**
     * Returns the alarm specific problem code. 
     * 
     * @return string
     */
    public function getSpecificProblemCode()
    {
    	return $this->specificProblemCode;
    }
    
    /**
     * Set the alarm specific problem text localised in client language.
     *
     * @param string $specificProblemText
     */
    public function setSpecificProblemText( $specificProblemText )
    {
    	$this->specificProblemText = $specificProblemText;
    }
    
    /**
     * Returns the alarm specific problem text localised in client language.
     *
     * @return string
     */
    public function getSpecificProblemText()
    {
    	return $this->specificProblemText;
    }    
    
    /**
     * Delete all alarms
     * 
     * @param SMTSmartOtuDB $dbConnection
     */
    public static function deleteAll( SMTSmartOtuDB $dbConnection )
    {
    	$dbConnection->runInTransaction( new SMTDeleteAllAlarms() );
    }
            
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
}

/**
 * Delete all alarms from database in a transaction
 *
 * @author Sylvain Desplat
 */
class SMTDeleteAllAlarms implements SMTIExecuteInTransaction
{
	/**
	 * @param SMTSmartOtuDB $dbConnection
	 */
	public function run( SMTSmartOtuDB $dbConnection )
	{
		SMTAttenuationOpticalAlarmDto::performDeleteAll( $dbConnection );
		
		SMTSystemAlarmDto::performDeleteAll( $dbConnection );
	}
	
	/**
	 * Returns always NULL since that executor is applied to multiple dtos.
	 * 
	 * @return SMTResultSetDto NULL
	 */
	public function getResultSetDto()
	{
	    return NULL;
	}
}
?>