<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\http;

use app\admin\SMTUserUtil;

use \app\admin\SMTLanguage;
use app\util\SMTLogger;

/**
 * Http Request
 * 
 * @author sdesplat
 */
class SMTHttpRequest
{
    const LOCAL_LOOP_IPV4="127.0.0.1";
    const LOCAL_LOOP_IPV6="::1";
    const TOKEN = "tokenId";
    
	public function cookieData($key)
	{
		return isset($_COOKIE[$key]) ? $_COOKIE[$key] : null;
	}

    public function cookieExists($key)
    {
        return isset($_COOKIE[$key]);
    }
    
    public function getData($key)
    {
        return isset($_GET[$key]) ? $_GET[$key] : null;
    }
    
    /**
     * Returns safe Get request parameter to avoid html injection
     * 
     * @param String $key
     * @return String the GET parameter value without malicious script tags
     */
    public function getSafeData($key)
    {
    	return isset($_GET[$key]) ? strip_tags($_GET[$key]) : null;
    }
    
    public function getExists($key)
    {
        return isset($_GET[$key]);
    }
    
    public function method()
    {
        return $_SERVER['REQUEST_METHOD'];
    }
    
    public function postData($key)
    {
        return isset($_POST[$key]) ? $_POST[$key] : null;
    }
    
    /**
     * Returns safe Post request parameter to avoid html injection
     *
     * @param String $key
     * @return String the POST parameter value without malicious script tags
     */    
    public function postSafeData($key)
    {
    	return isset($_POST[$key]) ? strip_tags($_POST[$key]) : null;
    }
    
    public function postExists($key)
    {
        return isset($_POST[$key]);
    }
    
    public function requestURI()
    {
        return $_SERVER['REQUEST_URI'];
    }
    
    /**
     * Retrieve client ip address
     *
     * @return string Ip address
     */
    public function getClientIpAddress()
    {
        return $_SERVER['REMOTE_ADDR'];
    }
    
    /**
     * Whether the request is coming from the server host and not a remote HTTP client
     */
    public function isLocalHTTPQuery()
    {
        $ipClientAddress = $this->getClientIpAddress();
        return ( strcasecmp($ipClientAddress, self::LOCAL_LOOP_IPV4) == 0 || strcasecmp($ipClientAddress, self::LOCAL_LOOP_IPV6) == 0 || strcasecmp($ipClientAddress, $_SERVER['SERVER_ADDR']) == 0 );
    }
    
    /**
     * Retrieve browser default language or return null if not found
     * 
     * @return string
     */
    public function getBrowserDefaultLanguage()
    {
        if ( isset( $_SERVER['HTTP_ACCEPT_LANGUAGE'] ) )
        {
            $lang = substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2);
            switch ( $lang )
            {
                case SMTLanguage::FR :
                case SMTLanguage::EN :
                    return $lang;
                    break;
                default:
                    return NULL;
                    break;
            }
        }
        return NULL;
    }
    
    /**
     * Decode username and password from header or null if none is found.
     * 
     * @return array with username and password
     */
    public function decodeUserAuthorizationFromHeader()
    {
        $userIdentifiers = null;
        
        //Authorization String in header: <Authorization: Basic user:password> with user:password encoded in Base64  
        if ( isset ( $_SERVER["HTTP_AUTHORIZATION"] ) )
        {
            $identifiers = $_SERVER["HTTP_AUTHORIZATION"];
            if ( isset($identifiers) )
            {
                $identifiers= base64_decode( substr($identifiers, strlen("Basic ") - 1) );
                if ( $identifiers != null )
                {
                    $identifiersArray = preg_split("/[:]/", $identifiers);
                    if ( count( $identifiersArray ) > 1 )
                    {
                        $userIdentifiers[0] = array_shift( $identifiersArray );
                        $userIdentifiers[1] = array_shift( $identifiersArray );
                    }
                }
            }            
        }    
        return $userIdentifiers;
    }
    
    /**
     * Check Token against Cross-Site Request Forgery
     *
     * @return whether Cross-Site Request Forgery token is valid
     */
    public function checkToken($requestTextData)
    {
    	$checked = $_SERVER['REQUEST_METHOD'] === 'GET';
    	
    	if ( !$checked )
    	{
    		if ( isset( $_SERVER['HTTP_KEY'] ) && isset( $_SERVER['HTTP_TOKEN'] ) )
    		{
	    		$key = $_SERVER['HTTP_KEY'];
	    		$token = $_SERVER['HTTP_TOKEN'];
	    		$phpSession = $_SERVER['HTTP_PHPSESSID'];
    		}
    		else if ( isset( $_POST['KEY'] ) && isset( $_POST['TOKEN'] ) )
    		{
    			$key = $_POST['KEY'];
    			$token = $_POST['TOKEN'];
    			$phpSession = $_COOKIE["PHPSESSID"];
    		}
//     		SMTLogger::getInstance()->trace("key $key token $token cookie $phpSession msg $requestTextData");
    		$hashValue = hash('sha256', $phpSession."?".$key."&".$requestTextData);
    		
    		$checked = (strcmp($hashValue, $token) == 0) && !SMTSession::hasToken($token);
    		if ($checked )
    		{
    			SMTSession::addToken($token);
    		}
//     		SMTLogger::getInstance()->trace( ($checked) ? 'checked: true' : 'checked: false'." token client:".$hashValue);
    	}
    	return $checked;
    }
    
    /**
     * Check HTTP request referer: shouldn't change between request
     */
    public function checkReferer()
    {
    	$checked = TRUE;
    	if ( isset ($_SERVER['HTTP_REFERER']) )
    	{
    		$referer = ltrim($_SERVER['HTTP_REFERER'],'https://');
    		$referer = ltrim($referer,'http://');
    		$referer = substr($referer, 0, strpos($referer, '/'));
    		
//     		SMTLogger::getInstance()->trace( "referer:".$referer.''.$_SESSION['org_referer']);
    		
	    	if(!isset($_SESSION['org_referer']) )
	    	{
	    		$_SESSION['org_referer'] = $referer;
	    	}
	    	else if ( strcasecmp($referer, $_SESSION['org_referer']) != 0 )
	    	{
	    		$checked = FALSE;
	    	}
    	}
    	else if ( isset($_SESSION['org_referer']) )
    	{
//     		SMTLogger::getInstance()->trace( "referer:".$referer.''.$_SESSION['org_referer']);
    		$checked = FALSE;
    	}
    	return $checked;
    }
    
    /**
     * Whether the token given in the HTTP GET request is found in the login token file
     * @return boolean
     */
    public function hasValidLoginToken( SMTContext $context)
    {
        //SMTLogger::getInstance()->trace(sprintf("hasValidLoginToken >> Found token ".$this->getSafeData(self::TOKEN)), SMTLogger::INFO );
        if ( $this->getExists(self::TOKEN) )
        {
            $tokenId = $this->getSafeData(self::TOKEN);
            if ( !empty($tokenId) )
            {
                //search entry in login_tokens
                return SMTUserUtil::checkLoginToken($tokenId, $context);
            }
        }
        return FALSE;
    }
}
?>