<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\events\linktest;



use app\sharedmemory\SMTMemoryManager;

use app\sharedmemory\SMTIMemorySupport;

use app\services\monitoring\SMTLinkTestDto;

use app\events\SMTEventDto;
use app\services\monitoring\SMTCurrentPortDto;

/**
 * Link-test update event: wrap a link test dto with new values to notify to registered consumers
 * 
 * @author Sylvain Desplat
 *
 */
class SMTLinkTestUpdateEventDto extends SMTEventDto implements SMTIMemorySupport
{           
    /**
     * LINKTEST_UPDATE event key
     * 
     * @var string
     */
    const LINKTEST_UPDATE="LINKTEST_UPDATE";
    
    /**
     * Event is obsolete after 5 mn
     * @var integer
     */
    const EVENT_TIMEOUT = 300;
    
    /**
     * $linkTest can be null for an update of all the link-tests
     * 
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTLinkTestDto')
     * @var SMTLinkTestDto
     * @see refreshAll
     */
    protected $linkTest = NULL;

    /**
     * Whether the given link-test is being deleted ( the link + the test)
     * 
     * @var boolean
     */
    protected $isDeleted = FALSE;
    
    /**
     * Whether the link-tests must be reloaded
     *
     * @var boolean
     */
    protected $refreshAll = FALSE;
    
    /**
     * Last access to query the event
     *
     * @var float
     */
    private $lastAccess;
    
    /**
     *
     * @var integer OTU current port when used with external switch managed by external application
     */
    protected $currentPort = SMTCurrentPortDto::INVALID_CURRENT_PORT;
    
    /**
     * @param string $releaseVersion
     */
    public function setCurrentPort( $currentPort )
    {
    	$this->currentPort = $currentPort;
    }
    
    /**
     * @return int currentPort
     */
    public function getCurrentPort()
    {
    	return $this->currentPort;
    }
    
    /**
     * 
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    function getLinkTest()
    {
    	return $this->linkTest;
    }    
    
    /**
     * Whether the current link-test is being deleted ( the link + the test)
     * 
     * @param boolean $deleted
     */
    function setIsDeleted( $deleted )
    {
        $this->isDeleted = $deleted;
        $this->touchAccess();
    }
    
    /**
     * Whether all the link-tests must be refreshed
     *
     * @param boolean $refreshAll
     */
    function setRefreshAll( $refreshAll )
    {
        if ( $refreshAll )
        {
            //no link-test set, force identifier to -1
            $this->setId( -1 );
        }
    	$this->refreshAll = $refreshAll;
    	$this->touchAccess();
    }
    
    /** 
     * 
     * @param SMTLinkTestDto $linkTest
     */
    function setLinkTest( SMTLinkTestDto $linkTest = NULL )
    {
    	$this->linkTest = $linkTest;
    	if ( $linkTest != NULL )
    	{
    	    $this->setId( $linkTest->getId() );
    	}
    	else
    	{
    	    $this->setId( -1 );
    	}
    	$this->touchAccess();
    }         
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }        
    
    /**
     * Returns the dto class name.
     *
     * @return the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }
    
    public function getLastAccess()
    {
    	return $this->lastAccess;
    }
    
    public function touchAccess()
    {
    	$this->lastAccess = microtime(true);
    }
    
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
        $this->touchAccess();
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }
    
    /**
     * Check whether the event was not requested since EVENT_TIMEOUT
     *
     * @return boolean Whether the event consumer was not requested since EVENT_TIMEOUT
     */
    public function isObsolete()
    {
    	if ( !isset( $this->lastAccess ) )
    	{
    		$this->lastAccess = 0;
    	}    	
    	return ( microtime(TRUE) - $this->getLastAccess() ) > self::EVENT_TIMEOUT;
    }    
}

?>