<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI
// Copyright VIAVI Solutions 2017. All rights reserved.
// *********************************************************
namespace app\admin;    

use app\settings\SMTSmartOtuSettings;
//use app\util\SMTLogger;

/**
 * Handle the OTU type: 
 *  - retrieve it from settings
 *  - store it in session variable
 * 
 * @author Sylvain Desplat
 *
 */
class SMTOtuType
{        
    const OTU_TYPE = 'OTU_TYPE';
    const COTU     = 'Otu5000';
    const OTU8KV2  = 'Otu8kv2';
    const OTU8KV2e = 'Otu8kv2e';
    const EOTDRV2  = 'Eotdrv2';
    const OLD_OTH  = 'A-FTH';
    const OTH      = 'FTH-9000';
    const FTH7K    = 'FTH-7000';
    
    const DEFAULT_OTU_TYPE = self::COTU; // OTU8kv2/OTU8kv2e no longer supported
    
    const COTU_DISPLAY    = 'OTU 5000';
    const OTU8KV2_DISPLAY = 'OTU 8000v2';
    const EOTDRV2_DISPLAY = 'EOTDR';
    const OTH_DISPLAY     = 'FTH-9000';
    const FTH7K_DISPLAY   = 'FTH-7000';
    
    
    /**
     * The supported OTU types keys.
     *
     *
     * @return string The OTU types keys.
     */
    public static function getSupportedTypes()
    {
        return array(self::OTU8KV2, self::OTU8KV2e, self::COTU, self::EOTDRV2, self::OTH, self::FTH7K );
    }    
    
    /**
     * Whether the OTU type is OTU8KV2
     * @return boolean Whether the OTU type is OTU8KV2
     */
    public static function isOtu8KV2()
    {
        return ( (strcasecmp( self::getOtuType() , self::OTU8KV2 ) == 0) || (strcasecmp( self::getOtuType() , self::OTU8KV2e ) == 0) );
    }

    /**
     * Whether the OTU type is COTU
     * @return boolean Whether the OTU type is COTU
     */
    public static function isCOtu()
    {
    	return ( strcasecmp( self::getOtuType() , self::COTU ) == 0 );
    }
    
    /**
     * Whether the OTU type is EOTDRV2
     * @return boolean Whether the OTU type is EOTDRV2
     */
    public static function isEOTDRV2()
    {
    	return ( strcasecmp( self::getOtuType() , self::EOTDRV2 ) == 0 );
    }
    
    /**
     * Whether the OTU type is COTU or EOTDRV2
     * @return boolean Whether the OTU type is COTU or EOTDRV2
     */
    public static function isCOtu_EOTDRV2()
    {
    	return self::isEOTDRV2() || self::isCOtu();
    }

    /**
     * Whether the OTU type is isOTH
     * @return boolean Whether the OTU type is isOTH
     */
    public static function isOTH()
    {
        return ( (strcasecmp( self::getOtuType() , self::OTH ) == 0) );
    }
    
    /**
     * Whether the OTU type is FTH-7000
     * @return boolean Whether the OTU type is FTH-7000
     */
    public static function isFTH7K()
    {
        return ( strcasecmp( self::getOtuType() , self::FTH7K ) == 0);
    }
    
    /**
     * Whether the string value is an existing OTU type
     *
     * @param string $type
     * @return boolean
     */
    public static function isValid( $type )
    {
    	if ( (strcasecmp( $type , self::COTU )    == 0)  || 
  	         (strcasecmp( $type , self::OTU8KV2 ) == 0)  ||
    	     (strcasecmp( $type , self::OTU8KV2e ) == 0) ||
	         (strcasecmp( $type , self::EOTDRV2 ) == 0)  ||
    	     (strcasecmp( $type , self::OTH ) == 0)      ||
    	     (strcasecmp( $type , self::FTH7K ) == 0)
    	   )
    	{
    		return true;
    	}
    	return false;
    }
    
    /**
     * The OTU type display name
     *
     * @return string  The OTU type display name.
     */
    public static function getOtuTypeDisplay()
    {
    	if ( self::isOtu8KV2() )
    	{
    		return self::OTU8KV2_DISPLAY;
    	}
    	else if ( self::isCOtu() )
    	{
    		return self::COTU_DISPLAY;
    	}
    	else if ( self::isEOTDRV2() )
    	{
    		return self::EOTDRV2_DISPLAY;
    	}
    	else if ( self::isOTH() )
    	{
    	    return self::OTH_DISPLAY;
    	}
    	else if ( self::isFTH7K() )
    	{
    	    return self::FTH7K_DISPLAY;
    	}
    	else
    	{
    	    return "";
    	}
   	
    }
    
    /**
     * Set OTU type in session
     *
     * @param string $type the OTU type
     */
    public static function setOtuType( $type )
    {
    	if ( self::isValid( $type ) )
    	{
    		$_SESSION[self::OTU_TYPE] = $type;
    	}
    }    
    
    /**
     * Return the OTU type
     * 
     * @return string
     */
    public static function getOtuType()
    {
        //update session current language
        if ( !isset($_SESSION[self::OTU_TYPE] ) || $_SESSION[self::OTU_TYPE] == NULL )
        { 
            $type = self::retrieveOtuType();
            if ( self::isValid( $type ) )
                {
                self::setOtuType( $type );
                }
        }        
        
        $type = isset($_SESSION[self::OTU_TYPE])?$_SESSION[self::OTU_TYPE]: self::DEFAULT_OTU_TYPE;
        
        if ( self::isValid( $type ) )
        {
            return $type;
        }
        
        return self::DEFAULT_OTU_TYPE;    
    }
    
    /**
     * Retrieve the OTU type from settings
     *
     */
    private static function retrieveOtuType()
    {	    	
    	return SMTSmartOtuSettings::getOtuType();
    }    
    
    /**
     * Retrieve the OTU type
     * 
     * @return string
     */
    public static function queryOtuType()
    {
        // get OTU Type
        $otuInfo = exec('sudo /sbin/getinfoversion Otu');
        $values = explode(";", $otuInfo);                
        $type = trim( $values[0] );        
        
//         $templogFileHandle = fopen( SMTLogger::getLogFilePath(),"a");
//         if ( $templogFileHandle !== FALSE )
//         {
//         	fwrite( $templogFileHandle, SMTLogger::formatMessage( '$otuInfo:'.$otuInfo.' $type'.$type.' self::isValid($type):'.self::isValid($type)) );
//         	fclose( $templogFileHandle );
//         }
        
        return (self::isValid($type)? $type : self::DEFAULT_OTU_TYPE);
    }
    
}
?>