
from json import loads
from jsonpath import jsonpath
from copy import deepcopy


class CdmProxy:
    def __init__(self, data: dict):
        """
        CdmProxy constructor.
        :param data: string containing the data data in json format
        """
        self.__data = deepcopy(data)

        # <TEST TYPE>
        test_type_path = "$.tests.0.type"   # all tests have the same type, so we extract it from the first one
        self.__test_type = self.get(test_type_path, "")

        # <TESTS>
        tests_path = "$.tests"
        self.__tests = self.get(tests_path, [])

        self.__tests_number = len(self.__tests)

        # <RESULTS>
        self.__results_number = [0] * self.tests_number
        for test_idx in range(self.tests_number):
            self.__results_number[test_idx] = len(self.get_measured_results(test_idx))

        # <EVENTS>
        self.__ids_slm = set()
        for test_idx in range(self.tests_number):
            for result_idx in range(self.results_number[test_idx]):
                events = self.get_events(test_idx, result_idx)

                for event in events:
                    self.__ids_slm.add(event.get("idSlm", -1))

    @classmethod
    def from_str(cls, data: str):
        return cls(loads(data))

    @classmethod
    def from_file(cls, filename: str):
        with open(filename, 'rb') as file:
            data = file.read().decode('utf8')

        return cls.from_str(data)

    def get(self, path: str, default=None):
        """

        :param path: json path to specific cdm entry (expected format: $.key_1.key_2.key_3...key_n)
        :param default: default output.
        :return: entry corresponding to specified cdm path, if exists
                 or default otherwise
        """
        if type(path) is str and path.startswith("$."):
            entry = jsonpath(self.__data, path)
        else:
            entry = self.__data.get(path, default)

        if (type(entry) is list) and (len(entry) == 1):
            return entry[0]

        return default

    def get_measured_results(self, test_index: int) -> list:
        """
        :param test_index:
        :return:
        """
        results_path = "$.tests.{}.results.data.measuredResults"

        if self.test_type == "OTDR":
            results_path = "$.tests.{}.results.data.otdrResults.measuredResults"

        return self.get(results_path.format(test_index), [])

    def get_events(self, test_index: int, result_index: int) -> list:
        events_path = ""

        if self.test_type == "OTDR":
            events_path = "$.tests.{}.results.data.otdrResults.measuredResults.{}.events"

        return self.get(events_path.format(test_index, result_index), [])

    @property
    def data(self):
        return deepcopy(self.__data)

    @property
    def test_type(self) -> str:
        return self.__test_type

    @property
    def tests_number(self) -> int:
        return self.__tests_number

    @property
    def results_number(self) -> list:
        return self.__results_number

    @property
    def max_id_slm(self) -> int:
        return max(self.__ids_slm)

    def get_cable_id(self, test_index: int) -> str:
        cable_id_path = "$.tests.{}.configuration.fiber.cableId"

        if self.test_type == "OTDR":
            cable_id_path = "$.tests.{}.configuration.otdrSettings.fiber.cableId"

        return self.get(cable_id_path.format(test_index), "")

    def get_fiber_id(self, test_index: int) -> str:
        fiber_id_path = "$.tests.{}.configuration.fiber.fiberId"

        if self.test_type == "OTDR":
            fiber_id_path = "$.tests.{}.configuration.otdrSettings.fiber.fiberId"

        return self.get(fiber_id_path.format(test_index), "")

    def get_fiber_number(self, test_index: int) -> int:
        fiber_number_path = "$.tests.{}.configuration.fiber.number"

        if self.test_type == "OTDR":
            fiber_number_path = "$.tests.{}.configuration.otdrSettings.fiber.number"

        return self.get(fiber_number_path.format(test_index))




