<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI
// Copyright JDSU 2016. All rights reserved.
// *********************************************************
namespace app\serviceshelper\maintenance;

use app\admin\SMTOtuType;
use app\http\SMTContext;
use app\parser\SMTOtuApi;
use app\parser\SMTOtuSocket;
use app\parser\SMTOtuSocketManager;
use app\services\maintenance\SMTLicenseDto;
use app\services\maintenance\SMTLicenseListDto;
use app\serviceshelper\maintenance\SMTLicenseList;
use app\serviceshelper\SMTServiceHelper;
use app\util\SMTIOException;
use app\util\SMTLogger;
use app\util\SMTUtil;
use app\settings\SMTSmartOtuSettings;

/**
 *
 * @author Sylvain Desplat
 */
class SMTLicence extends SMTServiceHelper
{
    const ENHANCED_SECURITY_ON = "ON";
    const ENHANCED_SECURITY_OFF = "OFF";
    
    const SSH_KEY_ROOT_ACCESS_ON = "ON";
    const SSH_KEY_ROOT_ACCESS_OFF = "OFF";
    
    const OPTION_IMPOSSIBLE= "IMPOSSIBLE";
    const OPTION_POSSIBLE  = "POSSIBLE";
    const OPTION_AVAILABLE = "AVAILABLE";
    const OPTION_EXPIRED   = "EXPIRED";
    
    const COPY_LICENSE_FILE_CMD = "sudo /usr/lib/jdsu/scripts/smartotu_copy_files.sh %s license";
    const DEFAULT_LICENSE_NAME = "Licence.lic";    
    
    /**
     * License full file path on disk
     *
     *
     */
    static function getLicenceFullPath()
    {
        return '/user/harddisk/'.self::DEFAULT_LICENSE_NAME; 
    }
    
    /**
     * Remove software options from session (license is installed in ISU during upload)
     *
     * @param $filename string The name of the license file
     *
     */
    function processLicense($fileName)
    {
    	//license is installed in ISU during the upload to avoid to store file with invalid content on disk
//     	$fullPath = SMTUtil::SMARTOTU_TEMP_DIR.$fileName;
//     	$this->installLicense($fileName, $fullPath);
    	
    	//reset licence in session
    	$this->resetSessionOptions();  
    }
    
    /**
     * Load license file in ISU.
     *
     * @param $filename string The name of the license file
     * @param $fullPath string The full path name of the temporary file
     * @throws \Exception if process fails
     */
    function installLicense($fileName, $fullPath)
    {
        $success = -1;
        
    	try
    	{    		
    	    if ( file_exists( $fullPath) )
    	    {
    	        $command = sprintf(self::COPY_LICENSE_FILE_CMD, SMTUtil::escapeShellArgUtf8($fullPath) );
    	        SMTLogger::getInstance()->trace( sprintf("command: %s.", $command ), SMTLogger::DEBUG);
    	        
    	        // update license
    	        $value = exec ($command, $output, $success);
    	        
    	        SMTLogger::getInstance()->trace(sprintf("copy uploaded license file output: %s success %s", $value, $success) );
    	        
    	        // 0 -> update license OK, otherwise KO
    	        if ($success !== 0)
    	        {
    	        	throw new SMTIOException(SMTIOException::INVALID_FILE, 'failed to copy uploaded license file');
    	        }    	            	        
    	        
        	    $socketManager = new SMTOtuSocketManager( $this->getContext() );
        	    //retrieve connection from the socket manager
        	    $socket = $socketManager->openSocket( SMTOtuSocket::ISU_PORT );
        	    try 
        	    {
        	        $socket->send( SMTOtuApi::setSsh_License_Option( self::DEFAULT_LICENSE_NAME ) );
        	    }
        	    catch(\Exception $ex)
        	    {        	    	
        	        SMTLogger::getInstance()->traceException( $ex );
        	        throw new SMTIOException(SMTIOException::INVALID_FILE, sprintf("invalid license file %s",$fileName) );
        	    }
        	    $socket->close();        	          	    
    	    }
    	    else
    	    {
    	        throw new SMTIOException(SMTIOException::FILE_NOT_FOUND, sprintf("uploaded license file %s not found",$fileName) );
    	    }
    	}
    	catch ( \Exception $e )
    	{
    		//remove invalid file
    	    unlink(self::getLicenceFullPath());
    		
    		SMTLogger::getInstance()->traceException( $e );
    		throw $e;
    	}
    	
    	//always remove file after installation in ISU, because it can contain malicious stuff
    	self::getLicenceFullPath();
    }
    
    function resetSessionOptions()
    {
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_SSH_KEY_ACCESS]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_HIGH_SEC]);    	
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_DUAL_OS_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_TEST_PON_MONITORING]);    	
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_PEAK_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_SAA_L2_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_TAPPING_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_LIGHT_SOURCE]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_REST_API]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_ULTRAFAST_MON]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_VPN]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_DUAL_ETH]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_ONMSI_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_ONMSI_BUILD]);
    }
    
    /**
     * Test whether OTU has ssh key root access option.
     *
     * @param SMTContext $context
     *
     * @return $sshKeyRootAccess whether the OTU has ssh key root access option.
     */
    static function isSshKeyRootAccessOption(SMTContext $context)
    {
    	$sshKeyRootAccess = FALSE;
    	
    	if ( isset($_SESSION[SMTLicenseList::OTU_OPTION_SSH_KEY_ACCESS]) )
    	{
    		$sshKeyRootAccess= $_SESSION[SMTLicenseList::OTU_OPTION_SSH_KEY_ACCESS];
    	}
    	else
    	{
	    	try
	    	{
	    		$sshKeyRootAccess = ( $context->getOtuSocket()->sendReceive( SMTOtuApi::isOtu_Ssh_Key_Root_Access_Option() ) === self::OPTION_AVAILABLE);
	    	    
    	    	$_SESSION[SMTLicenseList::OTU_OPTION_SSH_KEY_ACCESS] = $sshKeyRootAccess;
	    	}
	    	catch ( \Exception $e )
	    	{
	    		SMTLogger::getInstance()->traceException( $e );
	    	}
    	}
    	return $sshKeyRootAccess;
    }
    
    /**
     * Test whether OTU is in High Security mode.
     *
     * @param SMTContext $context
     *
     * @return $highSecurity whether the OTU is in High Security mode.
     */
    static function isHighSecurityMode(SMTContext $context)
    {
    	$highSecurity = FALSE;

    	if ( isset($_SESSION[SMTLicenseList::OTU_OPTION_HIGH_SEC]) )
    	{
    		$highSecurity = $_SESSION[SMTLicenseList::OTU_OPTION_HIGH_SEC];    		
    	}   
    	else
    	{
	    	try
	    	{
	    		$highSecurity = ( $context->getOtuSocket()->sendReceive( SMTOtuApi::isOtu_High_Security_Option() ) === self::OPTION_AVAILABLE);
	    		
	    		$_SESSION[SMTLicenseList::OTU_OPTION_HIGH_SEC] = $highSecurity;
	    	}
	    	catch ( \Exception $e )
	    	{
	    		SMTLogger::getInstance()->traceException( $e );
	    	}
    	}
    	return $highSecurity;
    }
    
    /**
     * Test whether OTU has peak monitoring option.
     *
     * @param SMTContext $context
     *
     * @return $peakMonitoring whether the OTU has peak monitoring option.
     */
    static function isPeakMonitoringOption(SMTContext $context)
    {
    	$peakMonitoring = FALSE;
    	
    	if ( isset($_SESSION[SMTLicenseList::OTU_OPTION_PEAK_MONITORING]) )
    	{
    		$peakMonitoring = $_SESSION[SMTLicenseList::OTU_OPTION_PEAK_MONITORING];
    	}
    	else
    	{
	    	try
	    	{
	    		$peakMonitoring = ( $context->getOtuSocket()->sendReceive( SMTOtuApi::isOtu_Peak_Monitoring_Option() ) === self::OPTION_AVAILABLE);

	    		$_SESSION[SMTLicenseList::OTU_OPTION_PEAK_MONITORING] = $peakMonitoring;
	    	}
	    	catch ( \Exception $e )
	    	{
	    		SMTLogger::getInstance()->traceException( $e );
	    	}
    	}
    	return $peakMonitoring;
    }
    
    /**
     * Test whether OTU has Tapping monitoring option.
     *
     * @param SMTContext $context
     *
     * @return $tappingMonitoring whether the OTU has Tapping monitoring option.
     */
    static function isTappingMonitoringOption(SMTContext $context)
    {
    	$tappingMonitoring = FALSE;
    	
    	if ( isset($_SESSION[SMTLicenseList::OTU_OPTION_TAPPING_MONITORING]) )
    	{
    		$tappingMonitoring= $_SESSION[SMTLicenseList::OTU_OPTION_TAPPING_MONITORING];
    	}
    	else
    	{
    		try
    		{
    			$tappingMonitoring = ( strcasecmp( $context->getOtuSocket()->sendReceive( SMTOtuApi::isOtu_Tapping_Monitoring_Option() ), self::OPTION_AVAILABLE) == 0);
    			
    			$_SESSION[SMTLicenseList::OTU_OPTION_TAPPING_MONITORING] = $tappingMonitoring;
    		}
    		catch ( \Exception $e )
    		{
    			SMTLogger::getInstance()->traceException( $e );
    		}
    	}
    	
    	return $tappingMonitoring;
    }
    
    /**
     * Test whether OTU has SAA option.
     *
     * @param SMTContext $context
     *
     * @return $saa whether the OTU has SAA monitoring option.
     */
    static function isSaaOption(SMTContext $context)
    {
    	$saa= FALSE;
    	
    	if ( isset($_SESSION[SMTLicenseList::OTU_OPTION_SAA_L2_MONITORING]) )
    	{
    		$saa = $_SESSION[SMTLicenseList::OTU_OPTION_SAA_L2_MONITORING];
    	}
    	else
    	{
    		try
    		{
    			$saa= ( $context->getOtuSocket()->sendReceive( SMTOtuApi::isOtu_Saa_Option() ) === self::OPTION_AVAILABLE);
    			
    			$_SESSION[SMTLicenseList::OTU_OPTION_SAA_L2_MONITORING] = $saa;
    		}
    		catch ( \Exception $e )
    		{
    			SMTLogger::getInstance()->traceException( $e );
    		}
    	}
    	return $saa;
    }
    
    /**
     * Test whether OTU has REST API option.
     *
     * @param SMTContext $context
     *
     * @return $RestApi whether the OTU has REST API option.
     */
    static function isRestApiOption(SMTContext $context)
    {
    	$RestApi = FALSE;
    	
    	if ( SMTSmartOtuSettings::isRestApi() !== NULL )
    	{
    		$RestApi= SMTSmartOtuSettings::isRestApi();
    	}
    	else
    	{
    		try
    		{
    			$RestApi = ( $context->getOtuSocket()->sendReceive( SMTOtuApi::isOtu_Rest_Api_Option() ) === self::OPTION_AVAILABLE);
    			
    			SMTSmartOtuSettings::setRestApi($RestApi);
    		}
    		catch ( \Exception $e )
    		{
    			SMTLogger::getInstance()->traceException( $e );
    		}
    	}
    	return $RestApi;
    }
    
    /**
     * Test whether OTU has VPN option.
     *
     * @param SMTContext $context
     *
     * @return $vpn whether the OTU has VPN option.
     */
    static function isVpnOption(SMTContext $context)
    {
        $vpn = FALSE;
        
        if ( SMTSmartOtuSettings::isVpn() !== NULL )
        {
            $vpn= SMTSmartOtuSettings::isVpn();
        }
        else
        {
            try
            {
                $vpn = ( $context->getOtuSocket()->sendReceive( SMTOtuApi::isOtu_Vpn_Option() ) === self::OPTION_AVAILABLE);
                
                SMTSmartOtuSettings::setVpn($vpn);
            }
            catch ( \Exception $e )
            {
                SMTLogger::getInstance()->traceException( $e );
            }
        }
        return $vpn;
    }
    
    /**
     * Returns the software options
     *
     * @param SMTContext $context
     *
     * @return $softwareOptions the list of software options as a string
     */
    static function retrieveSoftwareOptions(SMTContext $context)
    {
        $softwareOptions = "";
        try
        {
            $socketManager = new SMTOtuSocketManager( $context );
            //retrieve connection from the socket manager
            $socket = $socketManager->openSocket( SMTOtuSocket::ISU_PORT );
            $softwareOptions = $socket->sendReceive(SMTOtuApi::getOtuLicensesFromIsu(), 10);
            if (SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2())
            {
            $softwareOptions = $softwareOptions.",".$socket->sendReceive(SMTOtuApi::getOtauLicensesFromIsu());
            $softwareOptions = $softwareOptions.",".$socket->sendReceive(SMTOtuApi::getOtdrLicensesFromIsu());
            $softwareOptions = $softwareOptions.",".$socket->sendReceive(SMTOtuApi::getSensingLicensesFromIsu());
            if (SMTOtuType::isOTH() )
               {
               $softwareOptions = $softwareOptions.",".$socket->sendReceive(SMTOtuApi::getHdosLicensesFromIsu());
               }
            }
            else if (SMTOtuType::isFTH7K() )
            {
                $softwareOptions = $softwareOptions.",".$socket->sendReceive(SMTOtuApi::getHdosLicensesFromIsu());
                $softwareOptions = $softwareOptions.",".$socket->sendReceive(SMTOtuApi::getOtdrLicensesFromIsu());
            }
            else if (SMTOtuType::isCOtu_EOTDRV2())
            {
            	$saaOption = $socket->sendReceive(SMTOtuApi::getSAALicensesFromIsu());
            	if (strlen($saaOption) > 0)
            	{
            		//rename SAA_L2 ISU option to SSA display name
            		if ( strcasecmp(SMTLicenseList::ISU_OPTION_SAA_L2_MONITORING, $saaOption) == 0 )
            		{
            			$saaOption = SMTLicenseList::SAA;
            		}
            		
	            	if ( (strlen($softwareOptions) > 0) )
	            	{
	            		$softwareOptions = $softwareOptions.",".$saaOption;
	            	}
	            	else
	            	{
	            		$softwareOptions = $saaOption;
	            	}
            	}
            	
            	$softwareOptions = $softwareOptions.",".$socket->sendReceive(SMTOtuApi::getHdosLicensesFromIsu());
            }
            $softwareOptions = trim( str_replace(',,',',', $softwareOptions) );
            $softwareOptions = trim( str_replace(',',' ', $softwareOptions) );
            
            SMTLogger::getInstance()->trace(sprintf("software options: %s", $softwareOptions) );
            $socket->close();
        }
        catch ( \Exception $e )
        {
            SMTLogger::getInstance()->traceException( $e );
        }
        
    return $softwareOptions;
    }
    
    /**
     * Returns the software options with their status
     *
     * @param SMTContext $context
     *
     * @return $softwareOptions the list of software options with their status
     */
    static function retrieveSoftwareOptionList(SMTContext $context, SMTLicenseListDto $licenseList)
    {
    	$softwareOptions = "";
    	$license = NULL;
    	try
    	{
    		//retrieve connection from the socket manager
//     		$softwareOptions = $context->getOtuSocket()->sendReceive( SMTOtuApi::getOtu_Option_List() );
//     		$softwareOptionList = explode(",", $softwareOptions);
    		
    		$softwareOptionList = SMTLicenseList::getOtuOptionList();
    		
    		foreach ($softwareOptionList as $option=>$name)
    		{
    			$option = trim($option);
    			if ( $option !== '')
    			{
	    			$status = $context->getOtuSocket()->sendReceive( SMTOtuApi::getOtu_Option_Status($option) );
	    			
	    			$payg_license_count = self::getPayg_Option_Value( $context, $option, $status );
	    			$license = new SMTLicenseDto();
	    			$license->setSoftwareOption($option);
	    			$license->setSoftwareOptionStatus($status);
	    			if ( $payg_license_count > 0 )
	    			{
	    			    $license->setSoftwareOptionName($name.' x'.$payg_license_count.'');
	    			}
	    			else
	    			{
	    			    $license->setSoftwareOptionName($name);
	    			}
	    			$licenseList->addLicense( $license );
    			}
    		}
    		
    		SMTLogger::getInstance()->trace(sprintf("software options: %s", $softwareOptions) );
    	}
    	catch ( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException( $e );
    	}
    	
    	return $softwareOptions;
    }
    
    /**
     * Returns the software options PAYG license count
     *
     * @param SMTContext $context
     *
     * @return $optionValue the software options PAYG license count
     */
    static function getPayg_Option_Value( SMTContext $context, $option, $status )
    {
        $optionValue = 0;
        if ( strcmp($status, 'IMPOSSIBLE') !=0 )
        {
            if (!SMTOtuType::isCOtu_EOTDRV2() && !SMTOtuType::isFTH7K())
            {
                if ( strcasecmp( $option, SMTLicenseList::OTU_OPTION_PAYG_PORT12) == 0)
                {
                    try
                    {
                        $optionValue = intval($context->getOtuSocket()->sendReceive( SMTOtuApi::getOtu_PAYG_Internal_Switch_Lic_Count()) );                    
                    }
                    catch ( \Exception $e )
                    {
                        SMTLogger::getInstance()->traceException( $e );
                    }
                }
            }
            if ( strcasecmp( $option, SMTLicenseList::OTU_OPTION_PAYG_HDOS12) == 0)
            {
                try
                {
                    $optionValue = intval($context->getOtuSocket()->sendReceive( SMTOtuApi::getOtu_PAYG_HDOS_Switch_Lic_Count()) );
                }
                catch ( \Exception $e )
                {
                    SMTLogger::getInstance()->traceException( $e );
                }
            }
        }
        return $optionValue;
    }
}
?>
