<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\alarm;

/**
 * Alarm_Severity
 * 
 * @author Sylvain Desplat
 */
class SMTAlarmSeverity
{
    const SEVERITY_UNKNOWN = -1;
    const SEVERITY_NONE = 0;
    const SEVERITY_INFO = 1;
    const SEVERITY_WARNING = 2;
    const SEVERITY_MINOR = 3;
    const SEVERITY_MAJOR = 4;
    const SEVERITY_CRITICAL = 5;

    const SEVERITY_CRITICAL_IMAGE = 'severityCritical.png';
    const SEVERITY_CRITICAL_CLEARED_IMAGE = 'severityCriticalCleared.png';
    const SEVERITY_INDETERMINATE_IMAGE = 'severityIndeterminate.png';
    const SEVERITY_INDETERMINATE_CLEARED_IMAGE = 'severityIndeterminateCleared.png';
    const SEVERITY_MAJOR_IMAGE = 'severityMajor.png';
    const SEVERITY_MAJOR_CLEARED_IMAGE = 'severityMajorCleared.png';
    const SEVERITY_MINOR_IMAGE = 'severityMinor.png';
    const SEVERITY_MINOR_CLEARED_IMAGE = 'severityMinorCleared.png';
    const SEVERITY_WARNING_IMAGE = 'severityWarning.png';
    const SEVERITY_WARNING_CLEARED_IMAGE = 'severityWarningCleared.png';
    
    /**
     * Whether the given severity is an alarm severity
     * 
     * @param integer $severity
     * @return boolean
     */
    static function isAlarmSeverity( $severity )
    {
        return $severity > SMTAlarmSeverity::SEVERITY_INFO;
    }
    
    
    /**
     * Convert OTU severity into SmartOtu severity
     * 
     * @param string $otuSeverity
     * @return integer SmartOtu severity
     */
    static function fromOtuSeverityToSmartOtuSeverity( $otuSeverity )
    {
        $severity = self::SEVERITY_NONE;
        
        switch( $otuSeverity)
        {
            case SMTOtuAlarmSeverity::OTU_SEVERITY_CRITICAL:
                $severity = self::SEVERITY_CRITICAL;
                break;
            case SMTOtuAlarmSeverity::OTU_SEVERITY_MAJOR:
            	$severity = self::SEVERITY_MAJOR;
            	break;
        	case SMTOtuAlarmSeverity::OTU_SEVERITY_MINOR:
        		$severity = self::SEVERITY_MINOR;
        		break;
    		case SMTOtuAlarmSeverity::OTU_SEVERITY_WARNING:
    			$severity = self::SEVERITY_WARNING;
    			break;
			case SMTOtuAlarmSeverity::OTU_SEVERITY_INFO:
				$severity = self::SEVERITY_INFO;
				break;
			case SMTOtuAlarmSeverity::OTU_SEVERITY_CLEAR:
				$severity = self::SEVERITY_NONE;
				break;
			case SMTOtuAlarmSeverity::OTU_SEVERITY_UNKNOWN:
				$severity = self::SEVERITY_UNKNOWN;
				break;								
        }
        return $severity;
    }
    
    
    /**
     * Convert SmartOtu severity into OTU severity
     *
     * @param string $otuSeverity
     * @return integer SmartOtu severity
     */
    static function fromSmartOtuSeverityToOtuSeverity( $severity )
    {
    	$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_CLEAR;
    
    	switch( $severity)
    	{
    		case self::SEVERITY_CRITICAL:
    			$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_CRITICAL;
    			break;
    		case self::SEVERITY_MAJOR:
    			$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_MAJOR;
    			break;
    		case self::SEVERITY_MINOR:
    			$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_MINOR;
    			break;
    		case self::SEVERITY_WARNING:
    			$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_WARNING;
    			break;
    		case self::SEVERITY_INFO:
    			$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_INFO;
    			break;    	    
    		case self::SEVERITY_UNKNOWN:
    			$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_UNKNOWN;
    			break;
    		default: //case self::SEVERITY_NONE:
				$otuSeverity = SMTOtuAlarmSeverity::OTU_SEVERITY_CLEAR;
				break;    			
    	}
    	return $otuSeverity;
    }
    
    /**
     * Severity string
     * 
     * @param integer $severity
     * 
     * @return string Severity string
     */
    static function getSeverityTranslation( $severity, $cleared )
    {
        $translation = OTU_ALARM_INDETERMINATE;
        if ( $cleared )
        {
            $translation = OTU_ALARM_CLEAR;
        }
        else
        {
            switch( $severity)
            {
            	case self::SEVERITY_CRITICAL:
            	    $translation = OTU_ALARM_CRITICAL;
            		break;
            	case self::SEVERITY_MAJOR:
            	    $translation = OTU_ALARM_MAJOR;
            		break;
            	case self::SEVERITY_MINOR:
            	    $translation = OTU_ALARM_MINOR;
            		break;
            	case self::SEVERITY_WARNING:
            	    $translation = OTU_ALARM_WARNING;
            		break;
            	case self::OTU_SEVERITY_UNKNOWN:
            	    $translation = OTU_ALARM_INDETERMINATE;
            		break;
            	default:
            		$translation = OTU_ALARM_INDETERMINATE;
            		break;
            }
        }
        return $translation;
    }
    
    /**
     * Severity image url
     * @param integer $severity
     * @param boolean $cleared
     * @param string $serverName
     * @return string Severity image url
     */
    static function getSeverityImageUrl( $severity, $cleared, $serverName, $isHighSecurity = FALSE )
    {
        $imageUrl = $isHighSecurity? sprintf("https://%s/app/view/images/", $serverName ) : sprintf("http://%s/app/view/images/", $serverName );
        switch( $severity)
        {
        	case self::SEVERITY_CRITICAL:
        	    if (!$cleared )
        	    {
        	        $imageUrl.= self::SEVERITY_CRITICAL_IMAGE;
        	    }
        	    else
        	    {
        	        $imageUrl.= self::SEVERITY_CRITICAL_CLEARED_IMAGE;
        	    }
        		break;
        	case self::SEVERITY_MAJOR:
                if (!$cleared )
        	    {
        	        $imageUrl.= self::SEVERITY_MAJOR_IMAGE;
        	    }
        	    else
        	    {
        	        $imageUrl.= self::SEVERITY_MAJOR_CLEARED_IMAGE;
        	    }        	    
        		break;
        	case self::SEVERITY_MINOR:
                if (!$cleared )
        	    {
        	        $imageUrl.= self::SEVERITY_MINOR_IMAGE;
        	    }
        	    else
        	    {
        	        $imageUrl.= self::SEVERITY_MINOR_CLEARED_IMAGE;
        	    }        	    
        		break;
        	case self::SEVERITY_WARNING:
                if (!$cleared )
        	    {
        	        $imageUrl.= self::SEVERITY_WARNING_IMAGE;
        	    }
        	    else
        	    {
        	        $imageUrl.= self::SEVERITY_WARNING_CLEARED_IMAGE;
        	    }        	    
        		break;
        	case self::OTU_SEVERITY_UNKNOWN:
                if (!$cleared )
        	    {
        	        $imageUrl.= self::SEVERITY_INDETERMINATE_IMAGE;
        	    }
        	    else
        	    {
        	        $imageUrl.= self::SEVERITY_INDETERMINATE_CLEARED_IMAGE;
        	    }
        		break;
        	default:
        	    $imageUrl = NULL;
        	    break;
        }
        return $imageUrl;
    }
    
    const UNCLEARED_STATUS = 0;
    const CLEARED_STATUS = 1;
    
    /**
     * Whether the given alarm clear status is the cleared status
     *
     * @param integer $clearStatus
     * @return boolean
     */
    static function isAlarmCleared( $clearStatus )
    {
    	return $clearStatus == self::CLEARED_STATUS;
    }
    
    /**
     * Convert OTU clear status into SmartOtu clear status
     *
     * @param boolean $isCleared
     * @return integer SmartOtu clear Status
     */
    static function fromOtuClearStatusToSmartOtuClearStatus( $isCleared )
    {
    	return ( $isCleared )? self::CLEARED_STATUS : self::UNCLEARED_STATUS;
    }
    
}
?>