<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\events;

use app\parser\SMTOtuSocketManager;

use app\util\SMTLogger;

use app\sharedmemory\SMTIMemorySupport;

use app\sharedmemory\SMTMemoryManager;

use app\util\SMTIClonable;

use app\events\operations\SMTIReplaceableEventDto;

use app\events\SMTEventDto;

use app\util\SMTDto;

abstract class SMTEventConsumerDto extends SMTDto implements SMTIClonable, SMTIMemorySupport
{
    /**
     * No more than 9 events can be stored for an event consumer. 
     * When that event overflow is reached, pop the older event
     * 
     * @var integer
     */
    const EVENTS_OVERFLOW = 10;
    
    /**
     * @var string
     */
    protected $consumerId;

    /**
     * Last access to query the event consumer
     *
     * @var float
     */
    private $lastAccess;
        
    /**
     * @SMTAttributeInfosAnnotation(classname='app\events\SMTEventDto',islist='true')
     * @var array app\events\SMTEventDto
     */
    protected $events = array();
    
    public function getLastAccess()
    {
    	return $this->lastAccess;
    }    

    public function touchAccess()
    {
    	$this->lastAccess = microtime(true);
    }    
    
    public function setConsumerId( $consumerId )
    {
    	$this->consumerId = $consumerId;
    }
    
    public function getConsumerId()
    {
    	return $this->consumerId;
    }    
    
    public function getId()
    {
    	return $this->getConsumerId();
    }    

    /**
     * Whether the maximum of events (9) stored for that consumer has been overwhelmed
     * 
     * @return boolean
     */
    public function isOverflow()
    {
        return ( count( $this->events ) >= self::EVENTS_OVERFLOW );
    }
    
    /**
     * Add the given event. 
     * WARNING: if the count of events is already = overflow, remove the oldest event and add the given one.
     * WARNING: if event consumer is obsolete, don't add event and, unregister and clean up the event consumer
     * 
     * @param SMTEventDto $event
     */
    public function addEvent( SMTEventDto $event )
    {       
        //add not null event if the consumer is not obsolete
        if ( $event != NULL )
        {
            if ( !$this->isObsolete() )
            {
                $eventOverflow = $this->isOverflow();
                
                if ( $eventOverflow )
                {
                    SMTLogger::getInstance()->trace( sprintf( "Event overflow for consumer %s of type %s: ", $this->consumerId, $this->getDtoClassName()), SMTLogger::INFO );
                }
                
                if ( ($event instanceof SMTIReplaceableEventDto && count( $this->events ) > 0) || $eventOverflow ) 
                {
                    $firstEvent = reset( $this->events );
                    if ( $firstEvent !== NULL && $firstEvent->canReplace( $event ) || $eventOverflow )
                    {
                        //if last event can be replaced or if we are in overflow, remove the first (oldest) event from the array
                        array_shift( $this->events );
                    }
                }
                
                //Add new event at the end of the array
        	    array_push( $this->events, $event);
            }
        }        
    }
    
    /**
     * Returns the events of that consumer and update translations of the event localized strings according to the consumer language.
     * 
     * @return array of SMTEventDto
     */    
    public function getEvents()
    {
        foreach( $this->events as $event )
        {
            $event->updateTranslations();
        }
    	
        return $this->events;
    }    
    
    /**
     * @return boolean
     */
    public function hasEvents()
    {
    	return count( $this->events ) > 0;
    }        

    /**
     * Remove all events from Dto
     */
    public function clearEvents()
    {
    	$this->events = array();
    	
    	//always update last access property when removing events
    	$this->touchAccess();
    }    
        
    public function __clone() 
    {    
        foreach ( $this->events as $key => $event )
        {
            $this->events[$key] = clone $this->events[$key]; 
        }
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    public function getDtoClassName()
    {
    	return get_class();
    }
   
    
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
        SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
    */
    public function delete()
    {
        SMTMemoryManager::deleteDto( $this );
    }        
    
    /**
     * Check whether the event consumer was not requested since a time >  10 times Max ( the period between 2 message retrievals, the ping timout on OTU parser ) .
     *
     * @return boolean Whether the event consumer was not requested since a time >  10 times Max ( the period between 2 message retrievals, the ping timout on OTU parser).
     */
    public function isObsolete()
    {
    	if ( !isset( $this->lastAccess ) )
	    {
	        $this->touchAccess();
	    }
    	//if 10 times MAX ( the period between 2 message retrieval, the ping timout on OTU parser) has been spent, the message is obsolete
    	return ( microtime(TRUE) - $this->getLastAccess() ) > max( 10 * SMTEventMessageManager::SLEEP_TIME_SEC, 10 * SMTOtuSocketManager::getPingTimout() );
    }    
} 

?>