"""
Module that contains schemas for the CDM configuration section for OTDR tests to
launch on the ONA product family
"""

from marshmallow import Schema, fields, post_load
from marshmallow.validate import OneOf

from rest_api.products.config_schema_common import ConfigBase, test_type_factory, config_factory

#pylint: disable=no-self-use

class FOTestLaunch():
    """
    Fiber Optics app's Test launcher details
    """
    launch = {
       'launch_type' : "exe",
       'can_launch'  : "/usr/share/job-manager/launchers/fiber/fiber_optics_ijm_tests_can_launch",
       'launch'      : "/usr/bin/fiber_optics_ijm_test_launcher"
    }

class MicTestLaunch():
    """
    Microscope app's Test launcher details
    """
    launch = {
       'launch_type' : "exe",
       'can_launch'  : "/usr/share/job-manager/launchers/fiber/fiber_inspection_ijm_tests_can_launch",
       'launch'      : "/usr/bin/fiber_inspection_ijm_test_launcher"
    }

class FiberRefInfoFormatter():
    """
    Formatters must implement a static method with signature ui_config_parameters(cdm_test, test_index=0, location_index=0)
    location_index is the index of the test location of interest within the "testLocations" array
    location_index of -1 indicates there are no locations associated with the test
    Expected output is a dictionary of key/value pairs to display in UI e.g. '{"fiberNumber":"1","fiberId":"fid1"}'
    """
    @classmethod
    def ui_config_parameters(cls, cdm_test, location_index=0):
        #Formatter for sub type info and reference info
        ui_params = {}
        test_attrs = None
        try:
            # Deal with refInfo fields (or other attributes if they should exist at some point in the future)
            if location_index == -1:
                #No locations associated with this test
                test_attrs = cdm_test["attributes"]
            else:
                #Use the specified location
                test_attrs = cdm_test["testLocations"][location_index]["attributes"]
            try:
                for k, v in test_attrs["referenceInfo"].items():
                    #ui_params[k] = v
                    if k == "fiberId":
                        ui_params["Fiber ID"] = v #TODO Make this work with translations provided by solution software
                    if k == "fiberNumber":
                        ui_params["Fiber Number"] = v #TODO Make this work with translations provided by solution software
                    if k == "connectorId":
                        ui_params["Connector ID"] = v #TODO Make this work with translations provided by solution software
            except:
                pass
            
            # TODO Also consider adding the config filename if present and not empty
            # (Need to improve formatting of column - no room for this at present: 2 lines? display order?)
            #config = cdm_test["configuration"]
            #try:
            #    for k, v in config.items():
            #        if k == "configFile" and v:
            #            ui_params["Config File"] = v #TODO Make this work with translations provided by solution software
            #        elif k == "otdrSettings":
            #            try:
            #                for k2, v2 in config["otdrSettings"].items():
            #                    if k2 == "configFile" and v2:
            #                        ui_params["Config File"] = v2 #TODO Make this work with translations provided by solution software
            #            except:
            #                pass
            #
            #except:
            #    pass
            
        except:
            pass
        return ui_params

class ManualStepRefInfoFormatter():
    """
    Formatters must implement a static method with signature ui_config_parameters(cdm_test, test_index=0, location_index=0)
    location_index is the index of the test location of interest within the "testLocations" array
    location_index of -1 indicates there are no locations associated with the test
    Expected output is a dictionary of key/value pairs to display in UI e.g. '{"fiberNumber":"1","fiberId":"fid1"}'
    """
    @classmethod
    def ui_config_parameters(cls, cdm_test, location_index=0):
        #Formatter for sub type info and reference info
        ui_params = {}
        test_attrs = None
        try:
            if location_index == -1:
                #No locations associated with this test
                test_attrs = cdm_test["attributes"]
            else:
                #Use the specified location
                test_attrs = cdm_test["testLocations"][location_index]["attributes"]

            try:
                for k, v in test_attrs["referenceInfo"].items():
                    ui_params[k] = v
            except:
                pass
            
        except:
            pass
        return ui_params

class CdmOTDRSchema(Schema, FOTestLaunch, FiberRefInfoFormatter):
    """
    Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
        'bidirOtdr',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        required=False,
        allow_none=True,
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='alarmThresholds',
        dump_to='alarmThresholds',
        description='The alarm thresholds'
    )
    config_file = config_factory('/user/disk/')
    setups = fields.Nested(
        Schema,
        missing=[],
        required=False,
        allow_none=True,
        load_from='setup',
        dump_to='setup',
        description='The setup(s)'
    )
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )
    fiber_type = fields.Str(
        required=False,
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )

class OTDRSettingsSchema(Schema, FOTestLaunch, FiberRefInfoFormatter):
    """
    Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    test_type = 'OTDR'

    otdr_settings = fields.Nested(
        CdmOTDRSchema,
        required=True,
        missing={},
        load_from='otdrSettings',
        dump_to='otdrSettings',
        description='The CDM configuration structure for OTDRs'
    )

class FcompProIlOrlSchema(Schema, FOTestLaunch, FiberRefInfoFormatter):
    """
    Cdm Schema for FCompPRO bidir Il+Orl measurements
    """
    meas_seq_choices = (
        'IL+ORL',
        'IL',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    test_type = test_type_factory('bidirIlOrl')
    meas_sequence = fields.Str(
        validate=OneOf(meas_seq_choices),
        load_from='measurementSequence',
        dump_to='measurementSequence',
        description='The IL/ORL measurement sequence to be used'
    ) 
    fiber_type = fields.Str(
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )
    config_file = config_factory('/user/disk/')


class FcompProMainSchema(Schema, FOTestLaunch, FiberRefInfoFormatter):
    """
    Cdm Schema for full FCompPRO sequence measurements
    """
    tpro_meas_seq_choices = (
        'IL+ORL',
        'IL',
        'bidirOtdr',
        'loopbackOtdr',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    test_type = test_type_factory('fcompPro') # Don't want this here but need it for native Job Mgr format

    # This is what SHOULD work, but doesn't because lib-jobmanager can't handle anything other than basic strings!!!
    #meas_sequence = fields.List(
    #        fields.Str(
    #        validate=OneOf(tpro_meas_seq_choices)
    #        ),
    #        load_from='measurementSequence',
    #        dump_to='measurementSequence',
    #        description='The FCompPRO measurement sequence to be used'
    #
    meas_sequence = fields.Str(
       # Impossible to validate content at this level with current lib jobmanager - do it in solution sw
       load_from='measurementSequence',
       dump_to='measurementSequence',
       description='The FCompPRO measurement sequence to be used'
    )
    fiber_type = fields.Str(
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )
    config_file = config_factory('/user/disk/')

class TrueFiberSchema(Schema, FOTestLaunch, FiberRefInfoFormatter):
    """
    Cdm Schema for ATT TRUE FIBER measurements
    """
    test_type = test_type_factory('trueFiber')


class FiberInspectionCdmSchema(Schema, MicTestLaunch, FiberRefInfoFormatter):
    """
    Schema for the configuration information of the Fiber Inspection Test.
    """
    profiles = (
        "SM UPC (IEC 61300-3-35 2.0)",
        "SM APC (IEC 61300-3-35 2.0)",
        "SM PC&FiberStub(IEC 61300-3-35 2.0)",
        "MM (IEC 61300-3-35 2.0)",
        "Ribbon, SM APC (IEC 61300-3-35 2.0)",
        ""
    )

    test_type = test_type_factory('fiberInspection')
    profileName = fields.Str(
        load_from='profileName',
        dump_to='profileName',
        description='The Inspection Probe profile to use for pass fail criteria',
        validate=OneOf(profiles)
    )

class OptitrakSchema(Schema, FOTestLaunch, FiberRefInfoFormatter):
    """
    Schema for OptiTrak tests in a Test Plan.
    """
    test_type = test_type_factory('optiTrak')

class ManualSchema(Schema, ManualStepRefInfoFormatter):
    """
    Schema for Generic Manual Steps in a Test Plan.
    """
    test_type = test_type_factory('manualStep')

class FiberContinuitySchema(Schema, FiberRefInfoFormatter):
    """
    Schema for Fiber Continuity tests in a Test Plan.
    """
    test_type = test_type_factory('fiberContinuity')


CONFIG_SCHEMAS = {
                 'OTDR': OTDRSettingsSchema,
                 'optiTrak': OptitrakSchema,
                 'trueFiber': TrueFiberSchema,   # Specific test type for AT&T combining OTDR & PON Powermeter
                 'fcompPro': FcompProMainSchema, # full FCompPRO sequence (can combine subtests)
                 'bidirIlOrl': FcompProIlOrlSchema,
                 'fiberInspection': FiberInspectionCdmSchema,
                 'fiberContinuity': FiberContinuitySchema, # Added for GUI Formatter (TODO: Manage the test)
                 'manualStep': ManualSchema # Added for GUI formatter
                 }

