import QtQuick 2.9
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import com.viavi.style 1.0
import com.viavi.style.components 1.0
import com.viavi.jobmanager_ui 1.0

Item {
    id: root

    property Job job
    property Test test

    //TODO: a way for TableHeader to derive/reuse this

    // Indicates if the buttons are enabled or not
    property bool buttonsEnabled: true

    // A list of test properties to be displayed if enough space
    property var preferredTestProperties: []

    // Out of the preferred properties mentioned above, how many of them should always be visible(even if not enough space)
    property int preferredTestPropertiesAlwaysVisible: 1

    // Provides widths for the columns. If no calculator is provided, implicit widths will be used
    property ColumnWidthCalculator widthCalculator
    
    //Internal
    readonly property int maxTestTypeWidth: widthCalculator ? widthCalculator.maxTestTypeWidth : 0
    readonly property var maxTestPropertyWidths: widthCalculator ? widthCalculator.maxTestPropertyWidths : ({})
    readonly property int maxStatusWidth: widthCalculator ? widthCalculator.maxStatusWidth : 0
    readonly property bool isTestLaunchable: {
        if(!test)
        {
            return true;
        }

        if(job != JobManager.currentJob)
        {
            return false;
        }

        return test.testRequired === TestRequirement.TEST_REQUIRED || test.testRequired === TestRequirement.TEST_OPTIONAL;
    }


    //TODO: dynamic props to display
    //TODO: max widths for test type, status string and dynamic props

    width: parent.width
    height: rowLayout.height

    Rectangle
    {
        anchors.left: parent.left
        anchors.right: parent.right
        height: parent.height
        color: ViaviColor.red
        opacity: 0.1
        // The row has a red background when an error is fatal
        // A fatal error has a severity greater or equal to ten
        // As per ErrorMessageHelper.h 
        visible: test ? test.errorMessage !== "" && test.errorSeverity >= 10 : false
    }

    RowLayout {
        id: rowLayout

        anchors.left: parent.left
        anchors.right: parent.right

        height: implicitHeight * 1.5

        spacing: ViaviStyle.layout.mediumMargin

        Image
        {
            id: launchIcon
            Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter

            Layout.preferredWidth: JobManagerStyle.launchIconSize.width
            Layout.preferredHeight: JobManagerStyle.launchIconSize.height

            fillMode: Image.PreserveAspectFit
            
            opacity: root.buttonsEnabled && isTestLaunchable && (!test || test.errorSeverity == 0) ? 1.0 : 0.0

            // This should update after the test is run
            source: test && test.testStatus !== "" ? "image://165x165/Replay-165x165.png" : "image://165x165/Play-165x165.png"

            MouseArea {
                anchors.fill: parent
                enabled: launchIcon.opacity === 1.0

                onClicked: {
                    JobManager.launchTest(test);
                }
            }
        }


        ColumnLayout
        {
            spacing: ViaviStyle.layout.mediumMargin

            Label {
                id: testType

                Layout.preferredWidth: maxTestTypeWidth !== 0 ? maxTestTypeWidth : implicitWidth
                Layout.preferredHeight: implicitHeight

                font: JobManagerStyle.testTypeFont
                text: test ? test.testTypeDisplayName : ""
                color: test && test.errorSeverity > 0 ? ViaviColor.lightGray3 : ViaviColor.black
            }

            Label {
                id: testErrorMessageLabel

                Layout.preferredWidth: maxTestTypeWidth !== 0 ? maxTestTypeWidth : implicitWidth
                Layout.preferredHeight: test ? test.errorMessage === "" ? 0 : implicitHeight : 0

                font: JobManagerStyle.testTypeFont
                color: ViaviColor.red
                text: test ? test.errorMessage : ""
                // We want to use this error message for recoverable errors?
                visible: test && test.errorMessage !== "" && test.errorSeverity == 0
                elide: Label.ElideRight
                wrapMode: Text.Wrap
            }
        }

        Label
        {
            id: testRowErrorMessageLabel

            Layout.preferredWidth: implicitWidth
            Layout.preferredHeight: test ? test.errorMessage === "" ? 0 : implicitHeight : 0

            font: JobManagerStyle.testTypeFont
            color: test && test.errorSeverity < 10 ? ViaviColor.lightGray3 : ViaviColor.red
            text: test ? test.errorMessage : ""
            visible: test && test.errorMessage !== "" && test.errorSeverity > 0
            elide: Label.ElideRight
            wrapMode: Text.Wrap
        }
        Item
        {
            // This is just to take up space when displaying the error message
            id: errorSpacer
            Layout.fillWidth: true
            Layout.preferredHeight: testType.height
            visible: test && test.errorMessage !== "" && test.errorSeverity > 0
        }

        Item {
            id: optionalProperties

            Layout.fillWidth: true
            Layout.preferredHeight: testType.height

            visible: test && (test.errorMessage == "" || test.errorSeverity == 0)

            RowLayout {
                id: testOptionalPropertiesRow
                anchors.fill: parent

                spacing: 0

                Repeater {
                    model: root.preferredTestProperties

                    Label {
                        Layout.alignment: Qt.AlignHCenter
                        Layout.preferredWidth: {
                            var w = implicitWidth;
                            if(modelData in maxTestPropertyWidths)
                            {
                                w = maxTestPropertyWidths[modelData];
                            }

                            if(index < preferredTestPropertiesAlwaysVisible)
                            {
                                var availableWidth = testOptionalPropertiesRow.width;
                                for(var i = 0; i < index; ++i)
                                {
                                    availableWidth -= maxTestPropertyWidths[root.preferredTestProperties[i]];
                                }

                                return Math.min(w, Math.max(availableWidth, 0));
                            }

                            return Math.min(w, testOptionalPropertiesRow.width / preferredTestPropertiesAlwaysVisible);
                        }

                        opacity: (index <= preferredTestPropertiesAlwaysVisible - 1 || x + width <= testOptionalPropertiesRow.width) ? 1.0 : 0.0

                        elide: Label.ElideRight
                        font: JobManagerStyle.testPropertyFont
                        text: {
                            if(!test)
                            {
                                return "";
                            }

                            if(modelData in test)
                            {
                                return test[modelData];
                            }
                            else if(modelData in test.optionalProperties && test.optionalProperties[modelData] !== undefined)
                            {
                                return test.optionalProperties[modelData];
                            }
                            return "";
                        }
                    }
                }
            }
        }

        Label {
            id: testStatus

            Layout.preferredWidth: maxStatusWidth !== 0 ? maxStatusWidth : implicitWidth
            Layout.preferredHeight: implicitHeight
            Layout.margins: JobManagerStyle.testRowMargins

            font: JobManagerStyle.testStatusFont

            color: {
                //TODO: better way of doing this
                if(!test)
                {
                    return ViaviStyle.theme.foreground;
                }
                
                if(test.testStatus === "pass")
                {
                    return ViaviStyle.theme.pass;
                }
                else if(test.testStatus === "fail")
                {
                    return ViaviStyle.theme.fail;
                }
                else if(test.testStatus === "none")
                {
                    return ViaviStyle.theme.foreground;
                }
                else if(test.testStatus === "skipped")
                {
                    return ViaviStyle.theme.foreground;
                }
                else if(test.testStatus === "marginal")
                {
                    return ViaviStyle.theme.warning;
                }
                else
                {
                    return ViaviStyle.theme.foreground;
                }
            }

            text: {
                //TODO: better way of doing this
                if(!test)
                {
                    return "";
                }

                if(test.testStatus === "pass")
                {
                    return qsTr("Pass");
                }
                else if(test.testStatus === "fail")
                {
                    return qsTr("Fail");
                }
                else if(test.testStatus === "none")
                {
                    return qsTr("Completed");
                }
                else if(test.testStatus === "skipped")
                {
                    return qsTr("Skipped");
                }
                else if(test.testStatus === "marginal")
                {
                    return qsTr("Marginal")
                }
                else if(test.testStatus === "")
                {
                    return qsTr("To Do")
                }
                else
                {
                    return test.testStatus;
                }
            }
        }


        Image
        {
            id: syncIcon
            Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter

            Layout.preferredHeight: JobManagerStyle.syncStatusIconSize.height
            Layout.preferredWidth: JobManagerStyle.syncStatusIconSize.width
            
            visible: JobManagerStyle.config.showTestSyncIcon
            fillMode: Image.PreserveAspectFit

            source: "image://230x165/Cloud-black-230x165.png"
        }

        Image
        {
            id: moreOptions
            Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter

            Layout.rightMargin: ViaviStyle.layout.mediumMargin

            Layout.preferredHeight: JobManagerStyle.moreTestOptionsIconSize.height
            Layout.preferredWidth: JobManagerStyle.moreTestOptionsIconSize.width
            visible: JobManagerStyle.config.showMoreOptions

            fillMode: Image.PreserveAspectFit
            // This button is not relevant for mobile tech only tests
            // Disable it in that case
            opacity: root.buttonsEnabled && (!test || test.errorSeverity !== 5)  ? 1.0 : 0.0

            source: "image://165x165/More-165x165.png"

            MouseArea {
                anchors.fill: parent
                enabled: moreOptions.opacity === 1.0

                onClicked: {
                    JobManagerStyle.config.openMoreOptionsMenu(root.job, root.test, moreOptions);
                }
            }
        }
    }
}
