import QtQuick 2.9
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import com.viavi.style 1.0
import com.viavi.style.components 1.0
import com.viavi.jobmanager_ui 1.0

Item {
    id: root
    property Job job
    property Test test
    property var testConfigurations: test.configuration
    readonly property var vmtTestTypes: ["Picture", "Signature"]
    width: parent.width
    height: testRow.height
    
    // Indicates if the buttons are enabled or not
    property bool buttonsEnabled: true

    readonly property bool isTestLaunchable: {
        if(!test)
        {
            return true;
        }

        if(job != JobManager.currentJob)
        {
            return false;
        }
        return test.testRequired === TestRequirement.TEST_REQUIRED || test.testRequired === TestRequirement.TEST_OPTIONAL;
    }

    RowLayout {
        id: testRow
        anchors.left: parent.left
        anchors.right: parent.right
        spacing: ViaviStyle.layout.largeMargin * 1.5

        Image
        {
            id: launchIcon
            Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter
            Layout.leftMargin: JobManagerStyle.launchIconSize.width 
            Layout.preferredWidth: JobManagerStyle.launchIconSize.width * 1.5
            Layout.preferredHeight: JobManagerStyle.launchIconSize.height * 1.5

            fillMode: Image.PreserveAspectFit
            opacity: root.buttonsEnabled && isTestLaunchable && (!test || test.errorSeverity == 0) && !(test.testType === "manual") ? 1.0 : 0.0

            // This should update after the test is run
            source: test && test.testStatus !== "" ? "image://165x165/Replay-165x165.png" : "image://165x165/Play-165x165.png"

            MouseArea {
                anchors.fill: parent
                enabled: launchIcon.opacity === 1.0

                onClicked: {
                    if(root.vmtTestTypes.includes(test.testType))
                    {
                        //by default, error severity is NON_FATAL
                        test.errorMessage = qsTr("Use Viavi Mobile Tech to complete test");
                    }
                    else
                    {
                        JobManager.launchTest(test);
                    }
                }
            }
        }

        Label {
            id: requiredTestMarker
            Layout.alignment: Qt.AlignHCenter
            Layout.leftMargin: -(1.5 * ViaviStyle.layout.largeMargin)
            Layout.bottomMargin: testInfoLayout.height - ViaviStyle.layout.largeMargin
            text: "*"
            font: ViaviStyle.layout.mediumBoldFont
            color: ViaviColor.red
            visible: {
                if(!test || test.testRequired !== TestRequirement.TEST_REQUIRED)
                {
                    return false;
                }
                
                if(JobManager.groupType === TestGroupType.DEFAULT_VIEW && test.numberOfTestLocations > 0)
                {
                    return true;
                }
                else if(JobManager.groupType === TestGroupType.GROUPBY_TEST_NAME)
                {
                    if(test.numberOfTestLocations > 0 && JobManager.currentJob.countTestsByDisplayName(test.testTypeDisplayName) >= 1)
                    {
                        return true;
                    }
                    else if(test.numberOfTestLocations === 0 && JobManager.currentJob.countTestsByDisplayName(test.testTypeDisplayName) > 1)
                    {
                        return true;
                    }
                }
                else if(JobManager.groupType === TestGroupType.GROUPBY_LOCATION)
                {
                    return true;
                }
                return false;
            }
        }

        Column {
            id: testInfoLayout          
            Layout.alignment: Qt.AlignLeft | Qt.AlignHCenter | Qt.AlignVCenter
            Layout.leftMargin: requiredTestMarker.visible ? -(1.5 * ViaviStyle.layout.largeMargin + requiredTestMarker.width) : -requiredTestMarker.width
            Layout.topMargin: Layout.preferredHeight * 0.2
            Layout.bottomMargin: Layout.preferredHeight * 0.2
            Layout.preferredWidth: parent.width / 2
            Layout.fillWidth: true

            Label {
                id: testOrLocation
                font: ViaviStyle.layout.mediumBoldFont
                text: {
                    if(!test)
                    {
                        return "";
                    }
                    if(JobManager.groupType === TestGroupType.GROUPBY_LOCATION)
                    {
                        return test.testTypeDisplayName;
                    }
                    else if((JobManager.groupType === TestGroupType.GROUPBY_TEST_NAME || 
                             JobManager.groupType === TestGroupType.DEFAULT_VIEW) &&
                             test.optionalProperties.location !== undefined)
                    {
                        return test.optionalProperties.location;
                    }
                    return "";
                }
                visible: text !== ""
            }

            Label {
                id: testDescription
                width: parent.width
                font: ViaviStyle.layout.mediumFont
                text: test ? test.description : ""
                visible: {
                    if(!test || test.description === "")
                    {
                        return false;
                    }
                    else if(test.testType !== "manual" && (JobManager.groupType === TestGroupType.GROUPBY_LOCATION || JobManager.groupType === TestGroupType.GROUPBY_TEST_NAME))
                    {
                        return true;
                    }
                    else if(test.testType === "manual")
                    {
                        return true;
                    }
                    else return false;
                }
                elide: Label.ElideRight
            }

            Label {
                id: testErrorMessageLabel
                font.pixelSize: ViaviStyle.layout.mediumFont.pixelSize
                width: parent.width
                color: ViaviColor.red
                text: test && test.testType !== "manual" ? test.errorMessage : ""
                visible: test && test.errorMessage !== "" && test.errorSeverity < 10
                elide: Label.ElideRight
                wrapMode: Text.Wrap
            }

            Repeater {
                model: root.testConfigurations

                Label {
                    visible: modelData !== ""
                    text: test ? modelData : ""
                    font: ViaviStyle.layout.mediumFont
                    verticalAlignment: Label.AlignVCenter
                    width: parent.width
                    elide: Label.ElideRight
                }
            }

            Label {
                id: testTimeLabel
                text: test && test.testType !== "manual" ? test.testTime : ""
                visible: test && text !== ""
                font: ViaviStyle.layout.mediumFont
                width: parent.width
                elide: Label.ElideRight
            }

            Loader {
                width: parent.width
                active: root.test.testType === "manual"
                sourceComponent: ManualTestEntry {
                    id: manualTest
                    test: root.test ? root.test : null
                } 
            }           
        }

        Image
        {
            id: statusIcon
            Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter | Qt.AlignRight
            Layout.rightMargin: moreOptions.visible ? 0 : root.width * 0.02
            Layout.preferredWidth: JobManagerStyle.launchIconSize.width
            Layout.preferredHeight: JobManagerStyle.launchIconSize.height

            fillMode: Image.PreserveAspectFit
            visible: test.testType !== "manual"
            opacity: root.buttonsEnabled && isTestLaunchable && (!test || test.errorSeverity == 0) ? 1.0 : 0.0

            // This should update after the test is run
            source: {
                if(test && test.testStatus !== "fail" && test.testStatus !== "")
                    return "image://64x64/checkmark-circle64_green.png";
                else if(test && test.testStatus === "fail") 
                    return "image://64x64/close-circle-outline-64.png";
                else if(test && test.testStatus === "")
                    return "image://64x64/time64-lightgray.png"
                else return "";
            }
        }

        Image
        {
            id: moreOptions
            Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter
            Layout.rightMargin: root.width * 0.02 // space between separator(96% of parent width) and right margin
            Layout.preferredHeight: JobManagerStyle.moreTestOptionsIconSize.height
            Layout.preferredWidth: JobManagerStyle.moreTestOptionsIconSize.width
            visible: JobManagerStyle.config.showMoreOptions

            fillMode: Image.PreserveAspectFit
            // This button is not relevant for mobile tech only tests
            // Disable it in that case
            // opacity: root.buttonsEnabled && (!test || test.errorSeverity !== 5)  ? 1.0 : 0.0

            source: "image://165x165/More-165x165.png"

            MouseArea {
                anchors.fill: parent
                enabled: moreOptions.opacity === 1.0

                onClicked: {
                    JobManagerStyle.config.openMoreOptionsMenu(root.job, root.test, moreOptions);
                }
            }
        }
    }
}