import QtQuick 2.9
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import com.viavi.style 1.0
import com.viavi.style.components 1.0
import com.viavi.jobmanager_ui 1.0

Item {
    id: root

    property Job job
    property string propName: ""
    property bool allowEditing: true
    property bool editable: displayInfo ? displayInfo.editable : true

    implicitHeight: loader.height

    property var displayInfo: {
        if(!job)
        {
            return null;
        }

        return propName in job.jobPropertyDisplayInfos ? job.jobPropertyDisplayInfos[propName] : null
    }

    Loader {
        id: loader
        width: parent.width
        sourceComponent: {
            if(root.displayInfo && root.displayInfo.validValues.length > 0) 
            {
                return dropdownComponent;
            }
            return textFieldComponent;
        }
    }

    Component {
        id: dropdownComponent
        ComboBox {
            id: dropdown
            currentIndex: dropdown.find(root.getPropertyValue())
            model: root.displayInfo ? root.displayInfo.validValues : null
            enabled: root.editable && root.allowEditing

            onActivated: {
                root.setPropertyValue(dropdown.currentText);
                try {
                    root.setPropertyValue(dropdown.currentText);
                } catch (error) { 
                    console.log("Could not set property " + propName + ": " + error);
                }
                dropdown.currentIndex = Qt.binding(function() { return dropdown.find(root.getPropertyValue()); });
            }

            font: ViaviStyle.layout.mediumFont
            delegate: ItemDelegate {
                width: dropdown.width
                contentItem: Label {
                    text: modelData
                    color: ViaviStyle.theme.clickableForeground
                    font: dropdown.font
                    elide: Text.ElideRight
                    verticalAlignment: Text.AlignVCenter
                }
                highlighted: dropdown.highlightedIndex === index
            }

            background: Rectangle {
                color: JobManagerStyle.textEditBackgroundColor
                border.width: 1
                border.color: JobManagerStyle.textEditBorderColor
            }

            JobManagerTextField {
                id: dummyTextField
                visible: false
            }
            height: dummyTextField.height

            Component.onCompleted: {
                dropdown.currentIndex = Qt.binding(function() { return dropdown.find(root.getPropertyValue()); });
            }
        }
    }

    Component {
        id: textFieldComponent
        JobManagerTextField {
            id: textField
            text: "" + getPropertyValue()
            enabled: root.editable && root.allowEditing

            backgroundColor: acceptableInput ? JobManagerStyle.textEditBackgroundColor : ViaviStyle.theme.fail
            backgroundColorDisabled: JobManagerStyle.textEditBackgroundColorDisabled
            borderColor: JobManagerStyle.textEditBorderColor
            borderColorFocused: JobManagerStyle.textEditBorderColorFocused
            borderWidth: 1
            textColorDisabled: JobManagerStyle.textEditDisabledColor
            font: {
                var f = ViaviStyle.layout.getFont(ViaviStyle.layout.mediumFontSize);
                if (!textField.enabled){
                    f.italic = true;
                }
                return f;
            }

            isNumeric: {
                if(!root.displayInfo)
                {
                    return false;
                }

                return root.displayInfo.type === PropertyType.DOUBLE;
            }

            inputMethodHints: isNumeric ? Qt.ImhFormattedNumbersOnly : Qt.ImhNone

            
            validator: {
                if(!root.displayInfo)
                {
                    console.log("No display info for " + propName);
                    return null;
                }

                if(root.displayInfo.type === PropertyType.DOUBLE)
                {
                    return doubleValidator;
                }
                else if(root.displayInfo.type === PropertyType.STRING)
                {
                    if(root.displayInfo.regex)
                    {
                        return regexValidator;
                    }
                    return null;
                }
                console.log("Property type not supported");
                return null;
            }

            // Validators
            DoubleValidator {
                id: doubleValidator
            }

            RegExpValidator {
                id: regexValidator
                regExp: root.displayInfo && root.displayInfo.regex ? root.displayInfo.regex : /.*/
            }
            

            onEditingFinished: {
                focus = false;
            }

            onFocusChanged: {
                if(!focus)
                {
                    console.log("Updating property...")
                    try {
                        root.setPropertyValue(textField.text);
                    } catch (error) { 
                        console.log("Could not set property " + propName + ": " + error);
                    }

                    textField.text = Qt.binding(function() { return "" + root.getPropertyValue(); });
                }
            }
        }
    }

    function getPropertyValue()
    {
        if(!job)
        {
            return "N/A";
        }

        if(propName in job)
        {
            return job[propName];
        }
        else if(propName in job.optionalProperties)
        {
            return job.optionalProperties[propName];
        }
        return "N/A";
    }

    function convertToType(txtValue)
    {
        if(!displayInfo)
        {
            return txtValue;
        }

        if(displayInfo.type === PropertyType.DOUBLE)
        {
            return parseFloat(txtValue);
        }
        
        return txtValue;
    }

    function setPropertyValue(txtValue)
    {
        if(!job)
        {
            return;
        }
        
        var val = convertToType(txtValue);
        if(propName in job)
        {
            job[propName] = val;
        }
        else if(propName in job.optionalProperties)
        {
            job.optionalProperties[propName] = val;
        }
        else
        {
            console.log("Could not set property value")
        }
    }
}