import QtQuick 2.9
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import Qt.labs.folderlistmodel 2.1
import com.viavi.style 1.0
import com.viavi.style.components 1.0

Dialog {
    id: root

    // API

    // Title of the dialog
    title: qsTr("Choose a file")

    // A list folder to be displayed by the dialog. The user fill be able
    //to select one of them and browse the files inside. 
    /*
        An array of objects with the following fields:
            - name: Name displayed on the UI
            - path: The path to the actual directory
            - icon: An icon to be displayed for the path
    */
    property var folders: ({})

    // Filter for files to be displayed (e.g. nameFilters: [ "*.png", "*.jpg" ])
    property alias nameFilters: folderModel.nameFilters

    // URL to the selected file
    readonly property url fileUrl: internal.fileUrl

    // Reset currently selected file
    function resetSelectedFile()
    {
        internal.fileUrl = "";
    }

    // Center in parent
    x: Math.round((parent.width - width) / 2)
    y: Math.round((parent.height - height) / 2)
    width: 800
    height: 500

    // Don't close unless user presses cancel
    closePolicy: Popup.NoAutoClose

    // Refresh directory after use
    onVisibleChanged: {
        if(!visible)
        {
            foldersListView.currentIndex = 0;
            refreshCurrentFolder();
        }
        else
        {
            resetSelectedFile();
        }
    }

    // Internal
    QtObject {
        id: internal
        property url fileUrl: ""
        property var currentRootPath: {
            if(foldersListView.currentIndex < 0)
            {
                return "";
            }
            return folders[foldersListView.currentIndex]["path"];
        }
        
        property var currentRootDisplayName: {
            if(foldersListView.currentIndex < 0)
            {
                return "";
            }
            return folders[foldersListView.currentIndex]["name"];
        }
        property url currentlySelectedFileUrl: {
            if(fileListView.currentIndex < 0 || fileListView.currentIndex >= fileListView.count)
            {
                return "";
            }

            return folderModel.get(fileListView.currentIndex, "fileURL");
            internal.isCurrentlySelectedFileADirectory = folderModel.get(fileListView.currentIndex, "fileIsDir");
        }
        property bool isCurrentlySelectedFileADirectory: {
            if(fileListView.currentIndex < 0 || fileListView.currentIndex >= fileListView.count)
            {
                return "";
            }

            return folderModel.get(fileListView.currentIndex, "fileIsDir");
        }
    }

    function openCurrentFile()
    {
        if(internal.currentlySelectedFileUrl.toString() === "")
        {
            return;
        }

        if(internal.isCurrentlySelectedFileADirectory)
        {
            folderModel.folder = internal.currentlySelectedFileUrl;
        }
        else
        {
            internal.fileUrl = internal.currentlySelectedFileUrl;
            root.accept();
        }
    }

    function navigateBack()
    {
        var currentFolder = folderModel.folder.toString();
        currentFolder = currentFolder.replace("file://", "");

        var relativePath = currentFolder.replace(internal.currentRootPath, "");
        if(relativePath.startsWith("/"))
        {
            relativePath = relativePath.slice(1);
        }

        if(relativePath === "")
        {
            return;
        }

        relativePath = relativePath.split("/").filter(function(i){return i})
        relativePath = relativePath.slice(0, -1);
        relativePath = relativePath.join("/");

        folderModel.folder = "file://" + internal.currentRootPath + "/" + relativePath
    }

    function refreshCurrentFolder(args)
    {
        folderModel.folder = "";
        folderModel.folder = "file://" + internal.currentRootPath + "/"
    }

    header: Rectangle {
        width: parent.width
        implicitHeight: titleLabel.implicitHeight
        color: ViaviColor.viaviPurple
        Label {
            id: titleLabel
            
            anchors.centerIn: parent

            text: root.title

            font: ViaviStyle.layout.veryLargeBoldFont
            color: ViaviColor.white
            padding: ViaviStyle.layout.mediumMargin
        }
    }

    footer: Item {
        width: parent.width
        height: openButton.implicitHeight + ViaviStyle.layout.mediumMargin

        Button {
            id: openButton

            anchors.right: cancelButton.left
            anchors.rightMargin: ViaviStyle.layout.mediumMargin

            text: qsTr("Open")
            font: ViaviStyle.layout.mediumFont

            onClicked: {
                root.openCurrentFile();
            }
        }

        Button {
            id: cancelButton

            anchors.right: parent.right
            anchors.rightMargin: ViaviStyle.layout.mediumMargin

            text: qsTr("Cancel")
            font: ViaviStyle.layout.mediumFont

            onClicked: {
                root.reject();
            }
        }
    }

    Component {
        id: highlightComponent
        Rectangle {
            color: ViaviStyle.theme.highlightedBackground
            radius: 5
        }
    }
    

    contentItem: RowLayout {
        ListView {
            id: foldersListView

            Layout.fillHeight: true
            Layout.preferredWidth: parent.width / 4
            Layout.alignment: Qt.AlignTop

            model: root.folders
            ScrollBar.vertical: JobManagerScrollBar { }
            
            spacing: ViaviStyle.layout.mediumMargin
            highlight: highlightComponent
            highlightMoveDuration: 100
            currentIndex: 0
            clip: true

            delegate: Item {
                id: foldersListViewDelegate
                width: parent.width
                height: folderName.height * 1.5

                Image {
                    id: folderIcon
                    
                    anchors.left: parent.left
                    anchors.leftMargin: ViaviStyle.layout.smallMargin
                    anchors.verticalCenter: parent.verticalCenter

                    source: modelData["icon"] ? modelData["icon"] : "image://32x32/folder_open-simple.png"
                    width: height
                    height: folderName.implicitHeight
                }

                Label {
                    id: folderName

                    anchors.left: folderIcon.right
                    anchors.leftMargin: ViaviStyle.layout.smallMargin
                    anchors.right: parent.right
                    anchors.verticalCenter: parent.verticalCenter

                    text: modelData["name"]
                    elide: Label.ElideRight

                    color: foldersListViewDelegate.ListView.isCurrentItem ? ViaviColor.white : ViaviColor.black
                }

                MouseArea {
                    anchors.fill: parent

                    onClicked: {
                        foldersListView.currentIndex = index;
                        folderModel.folder = "file://" + modelData["path"] + "/";
                    }
                }
            }
        }

        ColumnLayout {
            Layout.fillWidth: true
            Layout.fillHeight: true

            visible: foldersListView.count > 0

            RowLayout {
                Layout.fillWidth: true

                Button {
                    Layout.preferredWidth: currentFolderLabel.implicitHeight * 3
                    Layout.preferredHeight: currentFolderLabel.implicitHeight * 2

                    enabled: {
                        if(folderModel.status === FolderListModel.Null)
                        {
                            return false;
                        }

                        var currentFolder = folderModel.folder.toString();
                        currentFolder = currentFolder.replace("file://", "");

                        var relativePath = currentFolder.replace(internal.currentRootPath, "");
                        if(relativePath.startsWith("/"))
                        {
                            relativePath = relativePath.slice(1);
                        }

                        return relativePath !== "";
                    }
                    
                    Image {
                        anchors.centerIn: parent

                        width: currentFolderLabel.implicitHeight
                        height: currentFolderLabel.implicitHeight

                        source: "image://32x32/page_left-simple.png"
                    }

                    onClicked: {
                        root.navigateBack();
                    }
                }

                Label {
                    id: currentFolderLabel
                    Layout.fillWidth: true
                    text: {
                        if(folderModel.status === FolderListModel.Null)
                        {
                            return internal.currentRootDisplayName + "/";
                        }
                        var currentFolder = folderModel.folder.toString();
                        currentFolder = currentFolder.replace("file://" + internal.currentRootPath + "/", internal.currentRootDisplayName + "/");
                        currentFolder = currentFolder.replace("file://" + internal.currentRootPath, internal.currentRootDisplayName + "/");
                        return currentFolder;
                    }
                    elide: Label.ElideMiddle
                }
            }

            Item {
                Layout.fillWidth: true
                Layout.fillHeight: true

                visible: folderModel.status === FolderListModel.Null

                Label {
                    id: noFilesFoundLabel
                    anchors.centerIn: parent
                    text: qsTr("Location not found. Click the button to refresh.")
                }

                Button {
                    anchors.top: noFilesFoundLabel.bottom
                    anchors.topMargin: ViaviStyle.layout.largeMargin
                    anchors.horizontalCenter: noFilesFoundLabel.horizontalCenter
                    width: refreshIcon.width * 1.3
                    height: refreshIcon.height

                    Image {
                        id: refreshIcon
                        anchors.centerIn: parent
                        source: "image://32x32/reload_refresh-simple.png"
                    }

                    onClicked: {
                        root.refreshCurrentFolder();
                    }
                }
            }

            ListView {
                id: fileListView
                Layout.fillWidth: true
                Layout.fillHeight: true

                model: FolderListModel {
                    id: folderModel

                    folder: "file://" + internal.currentRootPath + "/"
                    sortField: FolderListModel.Type

                    onFolderChanged: {
                        fileListView.currentIndex = 0;
                    }
                }

                ScrollBar.vertical: JobManagerScrollBar { }

                currentIndex: 0
                spacing: ViaviStyle.layout.mediumMargin
                highlight: highlightComponent
                highlightMoveDuration: 100
                clip: true


                delegate: Item {
                    id: fileListViewDelegate
                    width: fileListView.width
                    height: fileNameLabel.height * 3

                    Image {
                        id: fileIcon

                        anchors.left: parent.left
                        anchors.leftMargin: ViaviStyle.layout.smallMargin
                        anchors.verticalCenter: parent.verticalCenter

                        width: height
                        height: fileNameLabel.implicitHeight
                        
                        source: model.fileIsDir ? "image://32x32/folder_open-simple.png" : "image://32x32/file-document-simple.png"
                    }

                    Label {
                        id: fileNameLabel

                        anchors.left: fileIcon.right
                        anchors.leftMargin: ViaviStyle.layout.smallMargin
                        anchors.right: parent.right
                        anchors.verticalCenter: parent.verticalCenter

                        text: model.fileName
                        elide: Label.ElideRight

                        color: fileListViewDelegate.ListView.isCurrentItem ? ViaviColor.white : ViaviColor.black
                    }

                    MouseArea {
                        anchors.fill: parent

                        onClicked: {
                            fileListView.currentIndex = index;
                        }

                        onDoubleClicked: {
                            root.openCurrentFile();
                        }
                    }
                }
            }
        }
    }
}