import QtQuick 2.9
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import com.viavi.style 1.0
import com.viavi.style.components 1.0
import com.viavi.jobmanager_ui 1.0

Rectangle {
    id: root

    // Input properties
    property Job job
    property alias showProgressBarText: progressBar.showText
    property bool showDetailsButton: true
    property bool showCreateReportButton: false
    property bool showDeleteJobButton: false
    property bool showCurrentJobButton: job && !job.isArchived
    property alias headerGridHeight: headerGrid.height
    property bool showGroupByButton: false
    property bool showCompleteButton: false

    // Signals
    signal detailsClicked(Job job);

    width: parent.width
    height: childrenRect.height + ViaviStyle.layout.mediumMargin

    color: JobManagerStyle.jobHeaderBackground

    // Internal properties
    QtObject {
        id: internal

        property bool isCurrentJob: JobManager.currentJob === job
    }

    RowLayout {
        id: headerGrid
        width: parent.width
        anchors.top: parent.top
        anchors.topMargin: ViaviStyle.layout.smallMargin
        Item {
            id: jobInfo

            Layout.fillWidth: true
            Layout.preferredHeight: jobInfoColumn.height

            Column {
                id: jobInfoColumn

                anchors.left: parent.left
                anchors.leftMargin: ViaviStyle.layout.largeMargin
                anchors.right: parent.right
                spacing: ViaviStyle.layout.smallMargin / 2                

                    
                Label {
                    id: jobNumberLabel
                    elide: Label.ElideRight

                    font: ViaviStyle.layout.mediumBoldFont
                    text: job ? job.jobNumber : ""
                }
                

                JobProgressBar {
                    id: progressBar
                    job: root.job

                    width: parent.width / 2
                    Layout.minimumWidth: parent.width / 5
                }

                RowLayout {
                    width: parent.width
                    spacing: ViaviStyle.layout.smallMargin

                    visible: job && job.customerName !== ""

                    Label {
                        id: customerLabel

                        Layout.alignment: Qt.AlignVCenter
                        Layout.preferredWidth: implicitWidth

                        text: qsTr("Customer") + qsTr(":")
                    }

                    Label {
                        id: cutomerValueLabel

                        Layout.alignment: Qt.AlignVCenter
                        Layout.fillWidth: true

                        elide: Label.ElideRight

                        font: ViaviStyle.layout.mediumBoldFont
                        text: job ? job.customerName : ""
                    }
                }

                RowLayout {
                    id: conflictRow
                    width: parent.width
                    spacing: ViaviStyle.layout.smallMargin
                    visible: job  && job.state === "conflict"
                    Rectangle {
                        id: conflictWarning
                        color: ViaviColor.red
                        radius: 20
                        Layout.alignment: Qt.AlignVCenter
                        Layout.preferredWidth: conflictWarningMark.width * 3
                        Layout.preferredHeight: conflictWarningMark.height

                        Label {
                            id: conflictWarningMark
                            font: ViaviStyle.layout.mediumBoldFont                            
                            anchors.centerIn: parent
                            text: "!"
                            color: "white"
                        }
                    }

                    Label {
                        id: conflictWarningLabel

                        Layout.alignment: Qt.AlignVCenter
                        Layout.fillWidth: true

                        elide: Label.ElideRight
                        font: ViaviStyle.layout.mediumFont
                        color: ViaviColor.red
                        text: job ? qsTr("Conflict: Contact supervisor") : ""
                    }
                }
            }
        }

        Item {
            id: buttonContainer

            Layout.preferredWidth: parent.width / 3
            Layout.preferredHeight: childrenRect.height
            Layout.alignment: Qt.AlignRight

            RowLayout {
                id: buttonRow

                anchors.right: parent.right
                anchors.rightMargin: ViaviStyle.layout.largeMargin
                width: Math.min(buttonContainer.width - anchors.rightMargin, implicitWidth)
                layoutDirection: Qt.RightToLeft
                spacing: ViaviStyle.layout.mediumMargin

                Image
                {
                    id: detailsMenuButton
                    Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter
                    Layout.preferredHeight: JobManagerStyle.moreTestOptionsIconSize.height * 1.2
                    Layout.preferredWidth: JobManagerStyle.moreTestOptionsIconSize.width * 1.1

                    visible: root.showCompleteButton
                    fillMode: Image.PreserveAspectFit
                    source: "image://64x64/menu6-col-064.png"

                    MouseArea {
                        anchors { 
                            margins: -buttonRow.spacing / 2
                            fill: parent
                        }

                        onClicked: {
                            detailsMenu.popup(detailsMenuButton, -detailsMenu.width + detailsMenuButton.width, detailsMenuButton.height);
                        }
                    }
                }

                Image
                {
                    id: detailsButton
                    Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter
                    Layout.preferredHeight: JobManagerStyle.moreTestOptionsIconSize.height * 1.1
                    Layout.preferredWidth: JobManagerStyle.moreTestOptionsIconSize.width * 1.2
                    visible: showDetailsButton

                    fillMode: Image.PreserveAspectFit
                    source: "image://64x64/navigate_right-col-064.png"

                    MouseArea {
                        anchors { 
                            margins: -buttonRow.spacing / 2
                            fill: parent 
                        }
                        onClicked: {
                            detailsClicked(root.job);
                        }
                    }
                }

                Image
                {
                    id: groupByButton
                    Layout.alignment: Qt.AlignHCenter | Qt.AlignVCenter
                    Layout.preferredHeight: JobManagerStyle.moreTestOptionsIconSize.height * 1.1
                    Layout.preferredWidth: JobManagerStyle.moreTestOptionsIconSize.width * 1.1

                    visible: showGroupByButton
                    fillMode: Image.PreserveAspectFit
                    source: "image://165x165/menu-purple.png"

                    MouseArea {
                        anchors { 
                            margins: -buttonRow.spacing / 2
                            fill: parent 
                        }

                        onClicked: {
                            groupByMenu.popup(groupByButton, -groupByMenu.width + groupByButton.width, groupByButton.height);
                        }
                    }
                }

                Button {
                    id: currentJobButton

                    Layout.alignment: Qt.AlignRight
                    Layout.maximumWidth: currentJobFontMetrics.font, implicitWidth
                    Layout.fillWidth: true
                    Layout.preferredHeight: implicitHeight

                    property string currentJobText: qsTr("Current Job")
                    property string setAsCurrentJobText: qsTr("Set as Current")

                    FontMetrics {
                        id: currentJobFontMetrics

                        font: currentJobButton.font
                    }

                    leftPadding: 0
                    rightPadding: 0
                    labelFontSizeMode: Text.HorizontalFit
                    labelMinimumPixelSize: font.pixelSize / 1.5

                    visible: showCurrentJobButton
                    text: internal.isCurrentJob ? currentJobText : setAsCurrentJobText

                    backgroundColor: internal.isCurrentJob ? ViaviStyle.theme.highlightedBackground : ViaviStyle.theme.clickableBackground
                    textColor: internal.isCurrentJob ? ViaviStyle.theme.highlightedForeground : ViaviStyle.theme.clickableForeground

                    implicitWidth: {
                        var currentJobTextWidth = currentJobFontMetrics.boundingRect(currentJobText).width;
                        var setAsCurrentJobTextWidth = currentJobFontMetrics.boundingRect(setAsCurrentJobText).width;

                        return Math.max(currentJobTextWidth, setAsCurrentJobTextWidth) + 3 * currentJobButton.padding;
                    }

                    onClicked: {
                        if(!internal.isCurrentJob)
                        {
                            JobManager.setCurrentJob(job);
                        }
                        else
                        {
                            JobManager.unsetCurrentJob();
                        }
                    }
                }

                JobManagerMenu {
                    id: groupByMenu
                    width: {
                        var result = 0;
                        var padding = 0;
                        for (var i = 0; i < count; ++i) {
                            var item = itemAt(i);
                            result = Math.max(item.contentItem.implicitWidth, result);
                            padding = Math.max(item.padding, padding);
                        }
                        return result + padding * 2;
                    }

                    MenuItem {
                        font: ViaviStyle.layout.mediumFont
                        text: qsTr("By Job Order")
                        enabled: true

                        onClicked: {
                            JobManager.groupType = TestGroupType.DEFAULT_VIEW
                        }
                    }

                    MenuItem {
                        font: ViaviStyle.layout.mediumFont
                        text: qsTr("By Location")
                        enabled: true

                        onClicked: {
                            JobManager.groupType = TestGroupType.GROUPBY_LOCATION
                        }
                    }

                    MenuItem {
                        font: ViaviStyle.layout.mediumFont
                        text: qsTr("By Test")
                        enabled: true

                        onClicked: {
                            JobManager.groupType = TestGroupType.GROUPBY_TEST_NAME
                        }
                    }
                }


                JobManagerMenu {
                    id: detailsMenu
                    width: {
                        var result = 0;
                        var padding = 0;
                        for (var i = 0; i < count; ++i) {
                            var item = itemAt(i);
                            result = Math.max(item.contentItem.implicitWidth, result);
                            padding = Math.max(item.padding, padding);
                        }
                        return result + padding * 2;
                    }

                    MenuItem {
                        font: ViaviStyle.layout.mediumFont
                        text: qsTr("Create Report")
                        enabled: true
                        onClicked: {
                            JobManagerStyle.config.createReportClicked(root.job);
                        }
                    }

                    MenuItem {
                        font: ViaviStyle.layout.mediumFont
                        text: root.job && !root.job.isArchived ? qsTr("Archive Job") : qsTr("Move to Job List")
                        enabled: true
                        visible: root.showCompleteButton
                        height: visible ? implicitHeight : 0
                        onClicked: {
                            if(root.job)
                            {
                                root.job.isArchived = !root.job.isArchived;
                            }
                        }
                    }

                    MenuItem {
                        font: ViaviStyle.layout.mediumFont
                        text: qsTr("Delete")
                        enabled: true
                        visible: root.job && root.job.managedBy === "inst" ? root.job.isArchived : false
                        height: visible ? implicitHeight : 0
                        onClicked: {
                            console.log("Delete Archived Job")
                            JobManagerStyle.config.deleteJob(root.job);
                        }
                    }
                }
            }
        }
    }
}
