import QtQuick 2.9
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import com.viavi.style 1.0
import com.viavi.style.components 1.0
import com.viavi.jobmanager_ui 1.0

Page {
    id: root

    title: qsTr("Job Manager")

    background: Rectangle {
        anchors.fill: parent
        color: ViaviStyle.theme.background
    }

    Flickable {
        id: flickable

        anchors.fill: parent
        anchors.bottomMargin: ViaviStyle.layout.mediumMargin

        ScrollBar.vertical: JobManagerScrollBar { id: scrollBar }

        flickableDirection: Flickable.VerticalFlick
        contentHeight: column.implicitHeight
        contentWidth: width - Math.max(scrollBar.width - ViaviStyle.layout.mediumMargin, 0)

        ColumnLayout {
            id: column

            anchors.top: parent.top
            anchors.topMargin: ViaviStyle.layout.smallMargin
            anchors.left: parent.left
            anchors.leftMargin: ViaviStyle.layout.mediumMargin
            anchors.right: parent.right
            anchors.rightMargin: ViaviStyle.layout.mediumMargin

            spacing: ViaviStyle.layout.smallMargin


            Loader {
                id: currentJobLoader

                Behavior on Layout.preferredHeight {
                    PropertyAnimation {
                        duration: 250
                    }
                }

                onLoaded:
                {
                    // Handle the case where the current job isn't set and therefore isn't loaded in memory
                    // We do the animation and scrolling to the top after the item is drawn so the scrolling accounts for the loaded item's height
                    root.startCurrentJobAnimation()
                }
            }

            Rectangle {
                color: ViaviStyle.theme.background

                Layout.preferredWidth: parent.width
                Layout.preferredHeight: childrenRect.height

                ListView {
                    id: jobList

                    width: parent.width
                    height: childrenRect.height

                    interactive: false
                    spacing: column.spacing
                    visible: count > 0

                    model: JobManager.requiredJobs

                    delegate: JobHeader {
                        job: model.job
                        radius: JobManagerStyle.cardCornerRadius / 2
                        border.width: JobManagerStyle.cardBorderWidth / 2
                        onDetailsClicked: {
                            openJobDetailsPage(job);
                        }
                    }
                }
            }

            Label {
                Layout.preferredWidth: parent.width
                Layout.preferredHeight: flickable.height

                visible: !jobList.visible && !JobManager.currentJob

                verticalAlignment: Label.AlignVCenter
                horizontalAlignment: Label.AlignHCenter

                font: ViaviStyle.layout.largeFont
                text: qsTr("No required jobs")
            }

            states: [
                State {
                    name: "noCurrentJobSelected"
                    when: !JobManager.currentJob

                    PropertyChanges {
                        target: currentJobLoader
                        sourceComponent: undefined
                        Layout.preferredHeight: 0
                    }
                }, 

                State {
                    name: "currentJobSelected"
                    when: !!JobManager.currentJob

                    PropertyChanges {
                        target: currentJobLoader
                        sourceComponent: currentJobComponent
                        Layout.preferredWidth: column.width
                        Layout.preferredHeight: item.implicitHeight
                    }
                }
            ]
        }
    }

    Component {
        id: currentJobComponent

        Item {
            implicitHeight: currentJobColumn.height
            property alias currentJobHeader: currentJobHeader
            clip: true

            Column {
                id: currentJobColumn
                width: parent.width
                spacing: 5

                JobHeader {
                    id: currentJobHeader
                    job: JobManager.currentJob
                    showGroupByButton: allTestsCardView.visible
                    height: showGroupByButton ? childrenRect.height : childrenRect.height - allTestsCardView.height
                    radius: JobManagerStyle.cardCornerRadius / 2
                    border.width: JobManagerStyle.cardBorderWidth / 2

                    TestListCardView {
                        id: allTestsCardView
                        job: parent.job
                        testModel: JobManager.currentJob ? JobManager.currentJob.tests : null
                        width: parent.width * 0.98
                        groupType: JobManager.groupType
                        anchors.top: parent.top
                        anchors.topMargin: currentJobHeader.headerGridHeight + ViaviStyle.layout.mediumMargin
                        anchors.horizontalCenter: parent.horizontalCenter
                    }
                    
                    onDetailsClicked: {
                        openJobDetailsPage(job);
                    }
                }
            }
        }
    }

    Connections
    {
        target: JobManager
        onCurrentJobChanged:
        {
            // Perform the current job animation when the current job is set
            root.startCurrentJobAnimation()
        }
    }

    // Parallelize the animations
    // If we are scrolled at the top or close to the top
    // then there would be a delay for the color animation since the scroll animation would have to happen first in a sequential animation
    // The scroll animation is significantly shorter than the color animation so a parallel animation won't be a problem
    ParallelAnimation
    {
        id: currentJobChangedAnimation

        PropertyAnimation
        {
            target: flickable
            property: "contentY"
            to: 0
            duration: 250
        }

        PropertyAnimation
        {
            target: currentJobLoader.item.currentJobHeader
            property: "color"
            from: ViaviColor.viaviPurple
            to: JobManagerStyle.jobHeaderBackground
            duration: 2000
        }
    }

    function startCurrentJobAnimation()
    {
        if(JobManager.currentJob && currentJobLoader.item)
        {
            currentJobChangedAnimation.restart()
        }
    }

    function openJobDetailsPage(job)
    {
        stackView.push("DefaultLayoutDetailsPage.qml", { "job": job });
    }
}
