import QtQuick 2.9
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import com.viavi.style 1.0
import com.viavi.style.components 1.0
import com.viavi.jobmanager_ui 1.0

Page {
    id: root

    // API properties, signals, slots

    // Scale factor for the text and icons. Set a value bigger than 1.0 to scale up and a value 
    //smaller than 1.0 to scale down
    property alias scaleFactor: defaultLayoutConfig.scaleFactor


    // A list of properties to be displayed in the Job Information section of
    //the Details page. 
    /*
        By default, all available job properties are visible. The available job properties
    are:
        - jobNumber: cdm::CdmSouthboundJob::Workflow::workOrderId
        - customerName: cdm::CdmSouthboundJob::Workflow::CustomerInfo::company
        - techId: cdm::CdmSouthboundJob::Workflow::TechInfo::techId
        - attributes/*: cdm::CdmSouthboundJob::Workflow::attributes
            All attributes in the AttributeMap will appear under the attributes/attributeName
            property. 
    */
    property alias jobPropertyList: defaultLayoutConfig.jobPropertyList
    
    // A map from the name of a job property to the name that is to be 
    //displayed on the UI. 
    /*
        The library provides default names for the jobNumber, customerName and techId
    properties. 
        attributes/* will use the provided label. 

        You may use this map to override any of the displayed job property names.
    */
    property var jobPropertyDisplayNames: ({})


    // A list of preferred properties to be shown in the TestTable. If there
    //isn't enough space to show them all, items from the end of the list will
    //start getting hidden
    /*
        By default, the "location" and "configuration" properties are shown.

        All the available test properties are:
            - location: cdm::Test::testLocations (the first item in the list)
            - configuration: The test configuration info provided by the API
            - attributes/*: cdm::Test::attributes
                All attributes in the AttributeMap will appear under the attributes/attributeName
                property.
    */
    property alias preferredTestProperties: defaultLayoutConfig.preferredTestProperties

    // By default, regardless of how little space available, the first two test properties from the
    //preferredTestProperties list will always be visible(if not enough space, second property will elide).
    /*
        This property controls the number of preferredTestProperties that should always be visible(default is 2, meaning the
    location and configuration properties will always be visible).

        If not enough space, the first preferredTestPropertiesAlwaysVisible properties will split the available space
    and use elide if necessary(starting from the last property).
    */
    property alias preferredTestPropertiesAlwaysVisible: defaultLayoutConfig.preferredTestPropertiesAlwaysVisible

    // This bool controls if the header on the top with the app name is displayed or not. This can be configured from the solution side
    property bool showHeader: true

    // This bool controls if the sync button is displayed or not. This can be configured from the solution side
    property bool showSyncButton: true

    // This bool controls if the New button is displayed or not. This can be configured from the solution side
    property bool showNewButton: true

    property var numberOfButtons: showSyncButton ? (showNewButton ? 3 : 2 ) : ( showNewButton ? 2 : 1 )

    // This alias provides choice of whether moreOptions "three dots" column is displayed or not
    property alias showMoreOptions: defaultLayoutConfig.showMoreOptions

    // These properties control the radius and border for IJM cards. Can be configured from the solution side
    property int cardCornerRadius: 20
    property int cardBorderWidth: 2

    // Scale the keyboard height(default value is 1.0)
    property real keyboardHeightScale: 1.0

    // A map from the name of a test property to the name that is to be
    //displayed on the UI.
    /*
        The library provides default names for the location and configuration properties.
        attributes/* will use the provided label.

        You may use this map to override any of the displayed job property names.
    */
    property var testPropertyDisplayNames: ({})

    // A list of folders to be displayed in the file dialog
    /*
        An array of objects with the following fields:
            - "name": Name displayed on the UI
            - "path": The path to the actual directory
            - "icon": An icon to be displayed for the path
    */
   property var fileDialogFolders: ({})


    // Signal emitted when we want to open the Create Report Dialog
    /*
        You will want to extract whatever parameters you want 
    from the job object(e.g. job.jobNumber, which is the workorderID).
    */
    signal createReportClicked(Job job);

    // Signal emitted when we want to open the Guide Dialog
    /*
        You will need to extract whatever parameters you want
    from the test object(e.g. test.testIndex and test.locationIndex)
    */
    signal openGuideClicked(Job job, Test test);

    // Signal emmited when we want to sync with StrataSync
    /*
        For version 1.0, it is up to the solution to implement this however
    they want. 
    */
    signal syncRequested();

    // Signal called before a test is launched. This is an optional signal that you
    //don't have to implement if you don't need it
    signal testAboutToLaunch(Test test);


    // Internal properties
    readonly property var standaloneKeyboard: standaloneKeyboardLoader.item

    // Set the configuration object base on input properties
    JobManagerConfig {
        id: defaultLayoutConfig

        jobPropertyList: []
        jobPropertyDisplayNames: buildDisplayNames(defaultJobPropertyDisplayNames, root.jobPropertyDisplayNames)
        testPropertyDisplayNames: buildDisplayNames(defaultTestPropertyDisplayNames, root.testPropertyDisplayNames)
        fileDialogFolders: root.fileDialogFolders

        windowWidth: root.width
        windowHeight: root.height
        
        onCreateReportClicked: {
            root.createReportClicked(job);
        }

        onOpenGuideClicked: {
            root.openGuideClicked(job, test);
        }

        onSelectLogoClicked: {
            root.selectLogoClicked(job);
        }

        onTestAboutToLaunch: {
            root.testAboutToLaunch(test);
        }

        onDeleteJob: {
            console.log("Opening delete dialog for job: " + job.jobNumber);
            deleteJobDialog.jobToDelete = job;
            deleteJobDialog.open();
        }

        onOpenMoreOptionsMenu: {
            console.log("Opening more options menu for test: " + test.testIndex + ", " + test.locationIndex);

            moreOptionsMenu.menuJob = job;
            moreOptionsMenu.menuTest = test;

            moreOptionsMenu.popup(referenceItem, -moreOptionsMenu.width + referenceItem.width, referenceItem.height);
        }

        Component.onCompleted: {
            JobManagerStyle.config = defaultLayoutConfig;
            JobManagerStyle.cardCornerRadius = root.cardCornerRadius;
            JobManagerStyle.cardBorderWidth = root.cardBorderWidth;
            JobManagerStyle.keyboardHeightScale = root.keyboardHeightScale;
        }

        function buildDisplayNames(defaultValues, apiValues)
        {
            var displayNames = {};

            for(var k in defaultValues)
            {
                displayNames[k] = defaultValues[k];
            }

            for(var k in apiValues)
            {
                displayNames[k] = apiValues[k];
            }

            return displayNames;
        }
    }

    background: Rectangle {
        anchors.fill: parent
        color: ViaviStyle.theme.background
    }

    header: Rectangle {
        id: headerRect

        width: root.width
        height: childrenRect.height

        color: JobManagerStyle.headerBackgroundColor

        visible: showHeader

        Item {
            id: titleItem

            width: titleLabel.implicitWidth
            height: titleLabel.implicitHeight * 2

            Label {
                id: titleLabel

                anchors.centerIn: parent
                leftPadding: ViaviStyle.layout.largeMargin

                font: ViaviStyle.layout.largeFont
                color: JobManagerStyle.headerForegroundColor

                text: stackView.currentItem ? stackView.currentItem.title : ""
            }
        }
    }

    footer: Rectangle {
        id: footerRect

        width: root.width
        height: childrenRect.height

        color: JobManagerStyle.footerBackgroundColor

        RowLayout {
            width: parent.width

            Item {
                id: archivedJobsItem

                Layout.preferredWidth: parent.width / numberOfButtons
                Layout.preferredHeight: archivedJobsLabel.implicitHeight * 2

                visible: stackView.depth === 1

                Label {
                    id: archivedJobsLabel

                    anchors.centerIn: parent

                    font: ViaviStyle.layout.largeFont
                    color: JobManagerStyle.footerForegroundColor

                    text: qsTr("Archived Jobs")
                }

                MouseArea {
                    anchors.fill: parent
                    onClicked: {
                        stackView.push("DefaultLayoutArchivedJobsListPage.qml");
                    }
                }
            }

            Item {
                id: backItem

                Layout.preferredWidth: parent.width / numberOfButtons
                Layout.preferredHeight: backLabel.implicitHeight * 2
                visible: stackView.depth > 1

                Label {
                    id: backLabel

                    anchors.centerIn: parent

                    font: ViaviStyle.layout.largeFont
                    color: JobManagerStyle.footerForegroundColor

                    text: qsTr("Back to Job Manager")
                }

                MouseArea {
                    anchors.fill: parent
                    onClicked: {
                        stackView.pop();
                    }
                }
            }

            Item {
                id: newItem

                Layout.preferredWidth: parent.width / numberOfButtons
                Layout.preferredHeight: newLabel.implicitHeight * 2
                
                visible: showNewButton

                Label {
                    id: newLabel

                    anchors.centerIn: parent

                    font: ViaviStyle.layout.largeFont
                    color: JobManagerStyle.footerForegroundColor

                    visible: stackView.depth === 1
                    text: qsTr("New")
                }

                MouseArea {
                    anchors.fill: parent
                    enabled: newLabel.visible
                    onClicked: {
                        console.log("New clicked");
                        // Create window with text field
                        createJobDialog.open()
                    }
                }
            }

            Item {
                id: syncItem

                Layout.preferredWidth: parent.width / numberOfButtons
                Layout.preferredHeight: syncLabel.implicitHeight * 2

                visible: showSyncButton

                Label {
                    id: syncLabel

                    anchors.centerIn: parent

                    font: ViaviStyle.layout.largeFont
                    color: JobManagerStyle.footerForegroundColor

                    text: qsTr("Sync")
                }

                MouseArea {
                    anchors.fill: parent
                    onClicked: {
                        root.syncRequested();
                    }
                }
            }
        }
    }

    CreateJobDialog
    {
        id: createJobDialog
        parent: stackView
    }

    StackView {
        id: stackView

        readonly property alias displayPopup: toast

        anchors.left: parent.left
        anchors.right: parent.right

        //When necessary, we move the StackView up so that the edited item is always visible
        y: {
            // Get the y coordinates relative to the stackview of the edited item
            var itemCoords = null;
            var keyboardHeight = 0;
            if(standaloneKeyboard && standaloneKeyboard.editedItem)
            {
                itemCoords = standaloneKeyboard.editedItem.mapToItem(stackView, 0, 0);
                keyboardHeight = standaloneKeyboard.height;
            }
            else if(Qt.inputMethod.visible)
            {
                // We get the position of the text field in global coordinates
                var globalX = Qt.inputMethod.inputItemClipRectangle.x;
                var globalY = Qt.inputMethod.inputItemClipRectangle.y;

                // Map global coordinates to stackView relative coordinates
                itemCoords = stackView.mapFromGlobal(globalX, globalY);

                keyboardHeight = Qt.inputMethod.keyboardRectangle.height;
            }

            if(itemCoords)
            {
                // The height remaining on the screen after the keyboard is shown
                var remainingHeight = parent.height - keyboardHeight;
                var center = remainingHeight / 2;

                var itemY = itemCoords.y;
                if(itemY > center)
                {
                    return center - itemY;
                }
            }

            return 0;
        }
        height: parent.height

        Behavior on y {

            SequentialAnimation {
                PauseAnimation {
                    duration: 100
                }

                PropertyAnimation {
                    duration: 250
                }
            }
        }

        initialItem: initialPageComponent
    }

    Component {
        id: initialPageComponent

        DefaultLayoutJobListPage {

        }
    }
    
    
    Loader {
        id: standaloneKeyboardLoader
        y: root.height - height
        width: parent.width
        active: JobManager.useInternalKeyboard
        sourceComponent: JobManagerKeyboard {
        }
    }

    JobManagerBusyIndicator {
        id: busyIndicator

        indicatorMessage: qsTr("Launching test") + qsTr("...")
        visible: JobManager.isTestLaunching
    }

    JobManagerBusyIndicator {
        id: changeCountQuery

        indicatorMessage: qsTr("Updating jobs") + qsTr("...")
        visible: JobManager.isQueryChangeCountInProgress
    }

    JobManagerToastMessage {
        id: toast

        anchors.bottom: parent.bottom
        anchors.bottomMargin: ViaviStyle.layout.mediumMargin + ( standaloneKeyboard && standaloneKeyboard.height !== 0 ? standaloneKeyboard.height - footer.height : 0)
        anchors.horizontalCenter: parent.horizontalCenter
    }

    Dialog
    {
        id: deleteJobDialog

        property Job jobToDelete
        
        x: (parent.width - width) / 2
        y: (parent.height - height) / 2

        height: header.height + footer.height
        
        modal: true
        standardButtons: Dialog.Ok | Dialog.Cancel
        
        header: Rectangle {
            width: parent.width
            height: deactiveJobText.height
            color: ViaviColor.viaviPurple
            
            Text {
                id: deactiveJobText
                text: qsTr("Delete Job")
                anchors.centerIn: parent
                font: ViaviStyle.layout.veryLargeBoldFont
                color: ViaviColor.white
                padding: ViaviStyle.layout.mediumMargin
            }
        }

        footer: DialogButtonBox {
            visible: count > 0
            alignment: Qt.AlignRight
            buttonLayout: DialogButtonBox.AndroidLayout
            delegate: Button { }
        }
        
        onAccepted:
        {
            if(deleteJobDialog.jobToDelete)
            {
                JobManager.deleteJob(deleteJobDialog.jobToDelete);
                deleteJobDialog.jobToDelete = null;
            }
        }

        onRejected: {
            deleteJobDialog.jobToDelete = null;
        }

        // Animations
        enter: Transition {
            NumberAnimation { property: "opacity"; from: 0.0; to: 1.0 }
        }
        exit: Transition {
            NumberAnimation { property: "opacity"; from: 1.0; to: 0.0 }
        }
    }

    JobManagerMenu {
        id: moreOptionsMenu

        property Job menuJob: null
        property Test menuTest: null

        MenuItem {
            font: ViaviStyle.layout.mediumFont
            text: qsTr("Open Guide")

            enabled: moreOptionsMenu.menuTest && moreOptionsMenu.menuTest.hasGuide

            onClicked: {
                JobManagerStyle.config.openGuideClicked(moreOptionsMenu.menuJob, moreOptionsMenu.menuTest);
            }
        }
    }

    Connections {
        target: JobManager

        onJobManagerUpdated: {
            toast.show(qsTr("Jobs updated!"));
        }

        onTestAboutToLaunch: {
            JobManagerStyle.config.testAboutToLaunch(test);
        }

        onNewJobCreated: {
            stackView.push("DefaultLayoutDetailsPage.qml", { "job": job });
        }
    }
}
