import QtQuick 2.9                                                                                                                                                      
import QtQuick.Controls 2.2
import QtQuick.Layouts 1.2
import com.viavi.style 1.0
import com.viavi.style.components 1.0
import com.viavi.jobmanager_ui 1.0

Dialog
{
    id: createJobDialog
    title: qsTr("Create Job")

    modal: true
    standardButtons: Dialog.Save | Dialog.Cancel
    closePolicy: Popup.NoAutoClose

    x: Math.round((parent.width - width) / 2)
    y: Math.round((parent.height - height) / 2)
    width: Math.max(parent.width / 3, implicitWidth)
    height: implicitHeight < parent.height ? implicitHeight : parent.height

    property int workOrderIdMaxLength: 34
    property string mostRecentUsedTemplate: ""
    property string workflowTerminology: {
        var templateTerminology = JobManager.getWorkOrderLabelFromTemplate(templateComboBox.currentText);
        if(templateTerminology !== "")
        {
            return templateTerminology;
        }
        return qsTr("Work Order ID");
    }

    footer: DialogButtonBox {
        visible: count > 0
        alignment: Qt.AlignRight
        buttonLayout: DialogButtonBox.AndroidLayout
        delegate: Button { }
    }

    header: Rectangle {
        width: parent.width
        implicitHeight: deactiveJobText.implicitHeight
        color: ViaviColor.viaviPurple
        Text {
            id: deactiveJobText
            text: qsTr("Create Job")
            anchors.centerIn: parent
            font: ViaviStyle.layout.veryLargeBoldFont
            color: ViaviColor.white
            padding: ViaviStyle.layout.mediumMargin
        }
    }

    contentItem: ColumnLayout
    {
        RowLayout {
            visible: templateComboBox.model.length > 0
            Label {
                text: qsTr("Job Template")
                Layout.fillWidth: true
                wrapMode: Label.Wrap
                elide: Label.ElideRight
            }

            ComboBox {
                id: templateComboBox
                implicitWidth: techIDField.width
                font: ViaviStyle.layout.mediumFont

                delegate: ItemDelegate {
                    width: templateComboBox.width
                    contentItem: Label {
                        text: modelData
                        color: ViaviStyle.theme.clickableForeground
                        font: templateComboBox.font
                        elide: Text.ElideRight
                        verticalAlignment: Text.AlignVCenter
                    }
                    highlighted: templateComboBox.highlightedIndex === index
                }
            }
        }

        RowLayout
        {
            id: jobNumberRow

            Label
            {
                id: workOrderNumberLabel
                text: createJobDialog.workflowTerminology
                Layout.fillWidth: true
                wrapMode: Label.Wrap
                elide: Label.ElideRight
            }

            JobManagerTextField
            {
                id: workOrderNumberField
                validator: RegExpValidator {
                    id: regexValidator
                    regExp: /^[A-Za-z0-9_ .-]+$/
                }
                maximumLength: createJobDialog.workOrderIdMaxLength
            }
        }

        RowLayout
        {
            id: techIDRow

            Label
            {
                id: techIDLabel
                text: qsTr("Tech ID")
                Layout.fillWidth: true
                wrapMode: Label.Wrap
                elide: Label.ElideRight
            }

            JobManagerTextField
            {
                id: techIDField
            }
        }

        RowLayout {
            visible: templateComboBox.currentIndex > 0 && JobManager.getTemplateNumberOfTests(templateComboBox.currentText) > 0
            Label {
                text: qsTr("Import Test Locations")
                Layout.fillWidth: true
                wrapMode: Label.Wrap
                elide: Label.ElideRight
            }

            Button {
                implicitWidth: techIDField.width
                text: {
                    var fileUrl = importLocationsDialog.fileUrl.toString();
                    if(fileUrl != "")
                    {
                        return fileUrl.slice(fileUrl.lastIndexOf("/")+1);
                    }
                    return qsTr("Browse") + "..."
                }
                font: ViaviStyle.layout.mediumFont

                onClicked: {
                    importLocationsDialog.open();
                }
            }
        }
    }
    

    onOpened:
    {
        // get the techID from job manager if already set
        techIDField.text = JobManager.getTechID();
        // set the visibily and text of the techID if already populated
        techIDRow.enabled = techIDField.text === "";

        // Fetch the latest list of templates
        templateComboBox.model = getUpdatedTemplateList();

        // Reset currently selected file
        importLocationsDialog.resetSelectedFile();
    }

    onAccepted:
    {
        var locationsFilename = importLocationsDialog.fileUrl.toString().replace("file://", "");

        // date and tech ID should be pulled from known data if possible.
        if (!JobManager.createNewJob(workOrderNumberField.text, techIDField.text, templateComboBox.currentText, locationsFilename))
        {
            console.log("Failure creating new job");
            createJobDialog.parent.displayPopup.show(qsTr("Failed to create job with a blank name\n or a name that would match existing job."));
        }
        workOrderNumberField.text = "";
        techIDField.text = "";
        mostRecentUsedTemplate = templateComboBox.currentText;
    }

    JobManagerFileDialog {
        id: importLocationsDialog

        width: createJobDialog.parent.width * 0.8
        height: createJobDialog.parent.height * 0.9

        title: qsTr("Select a CSV Label List")
        folders: JobManagerStyle.config.fileDialogFolders
        nameFilters: ["*.csv"]
    }

    function getUpdatedTemplateList()
    {
        var templateList = JobManager.fetchJobTemplates();
        const foundIdx = templateList.findIndex(el => el == mostRecentUsedTemplate);
        if(foundIdx !== -1){            
            templateList.splice(foundIdx, 1);
            templateList.unshift(mostRecentUsedTemplate);
        }
        return templateList;
    }
}
