import QtQuick 2.9
import QtQuick.Controls 2.2
import com.viavi.jobmanager_ui 1.0

// Given a list of tests(as the testModel), computes the maximum size of each column:
//  - the testType column
//  - the testStatus column
//  - for all the test properties specified in preferredTestProperties list
Item {
    id: root

    //Need to recompute when: 
    //      - Model changes(tests added/removed) - handled by instantiator
    //      - test property change(e.g. after editing a property) - Handled by Connections in instantiator
    //      - font changes - handled by onXChanged slots

    // Input
    property alias testModel: instantiator.model

    // Specifies what test properties we want to compute the widths for
    property var preferredTestProperties: []

    // If provided, table headers are also considered during calculations
    property string testTypeHeaderString: ""
    property var preferredTestPropertiesHeaders: ({})
    property string testStatusHeaderString: ""

    // The fonts used for the columns
    property font testTypeFont: JobManagerStyle.testTypeFont
    property font testPropertyFont: JobManagerStyle.testPropertyFont
    property font testStatusFont: JobManagerStyle.testStatusFont
    property font testHeaderFont: JobManagerStyle.testHeaderFont
    
    // Max allowed width. 0 means no max width
    property int maxWidth: 0

    // Output
    property int maxTestTypeWidth: 0
    property var maxTestPropertyWidths: ({}) // <Key, Value> = <propertyName, columnWidth>
    property int maxStatusWidth: 0

    QtObject {
        id: internal

        property var allTestObjects: ({})
    }

    Instantiator {
        id: instantiator 

        model: testModel

        Item {
            property Test testObject: test
        }

        onObjectAdded: {
            internal.allTestObjects[index] = object.testObject;

            computeTimer.start();
        }

        onObjectRemoved: {
            delete internal.allTestObjects[index];

            computeTimer.start();
        }
    }

    // Recompute when these properties change
    onTestTypeFontChanged: {
        computeTimer.start();
    }

    onTestPropertyFontChanged: {
        computeTimer.start();
    }

    onTestStatusFontChanged: {
        computeTimer.start();
    }

    onTestTypeHeaderStringChanged: {
        computeTimer.start();
    }

    onPreferredTestPropertiesHeadersChanged: {
        computeTimer.start();
    }

    Timer {
        id: computeTimer

        running: false
        repeat: false
        interval: 1

        onTriggered: {
            computeWidths();
        }
    }

    function computeWidths()
    {
        console.time("Recomputing column widths");
        
        computeMaxTestTypeWidth();
        computeMaxTestPropertyWidths();
        computeMaxStatusWidth();

        console.timeEnd("Recomputing column widths");
    }

    FontMetrics {
        id: testTypeFontMetrics

        font: testTypeFont
    }

    FontMetrics {
        id: testPropertyFontMetrics

        font: testPropertyFont
    }

    FontMetrics {
        id: testStatusFontMetrics

        font: testStatusFont
    }

    FontMetrics {
        id: testHeaderFontMetrics

        font: testHeaderFont
    }

    function getOutputWidth(fontMetrics, longestString)
    {
        var outputWidth = fontMetrics.boundingRect(longestString).width + 2 * fontMetrics.averageCharacterWidth;
        if(root.maxWidth > 0)
        {
            return Math.min(root.maxWidth, outputWidth)
        }
        return outputWidth;
    }

    function computeMaxTestTypeWidth()
    {
        var longestWidth = getOutputWidth(testHeaderFontMetrics, testTypeHeaderString);
        for(var key in internal.allTestObjects)
        {
            var currentTest = internal.allTestObjects[key];
            var currentWidth = getOutputWidth(testTypeFontMetrics, currentTest.testTypeDisplayName);
            if(currentWidth > longestWidth)
            {
                longestWidth = currentWidth;
            }
        }
        
        root.maxTestTypeWidth = longestWidth;
    }

    function computeMaxTestPropertyWidths()
    {
        // Map that will hold the longest widths for each property   
        var longestWidths = {};

        // Initialize the map with the header widths
        for(var key in preferredTestProperties)
        {
            var prop = preferredTestProperties[key];

            var headerString = prop;
            if(prop in preferredTestPropertiesHeaders)
            {
                headerString = preferredTestPropertiesHeaders[prop];
            }
            
            longestWidths[prop] = getOutputWidth(testHeaderFontMetrics, headerString);
        }

        // Compute longest name for each property
        for(var key in internal.allTestObjects)
        {
            var currentTestObject = internal.allTestObjects[key];

            for(var i in preferredTestProperties)
            {
                var propName = preferredTestProperties[i];
                var propValue = propName in currentTestObject ? currentTestObject[propName] : currentTestObject.optionalProperties[propName];

                var propStringValue = "" + propValue;
                var propWidth = getOutputWidth(testPropertyFontMetrics, propStringValue);
                if(propWidth > longestWidths[propName])
                {
                    
                    longestWidths[propName] = propWidth;
                }
            }
        }

        root.maxTestPropertyWidths = longestWidths;
    }

    function computeMaxStatusWidth()
    {
        //TODO: don't have these hardcoded
        var statusStrings = [qsTr("Completed"), qsTr("Fail"), qsTr("Pass"), qsTr("To Do"), qsTr("Skipped"), qsTr("Marginal")];

        var longestWidth = 0;
        for(var key in statusStrings)
        {
            var statusString = statusStrings[key];
            var statusWidth = getOutputWidth(testStatusFontMetrics, statusString);
            if(statusWidth > longestWidth)
            {
                longestWidth = statusWidth;
            }
        }

        root.maxStatusWidth = Math.max(
            longestWidth, 
            getOutputWidth(testHeaderFontMetrics, testStatusHeaderString)
        );
    }
}