
import subprocess
import shutil
import json
import datetime
import time

class LaunchException(Exception):
    def __init__(self, message, custom_error_message=''):            
        # Call the base class constructor with the parameters it needs
        super().__init__(message)            
        # Now for your custom code...
        self.custom_error_message = custom_error_message
class ServerLaunchException(Exception):
    def __init__(self, message, custom_error_message=''):            
        # Call the base class constructor with the parameters it needs
        super().__init__(message)            
        # Now for your custom code...
        self.custom_error_message = custom_error_message

#The expected interface for any test launchers handled by TestLauncher
class TestLauncherInterface:
    #Throws LaunchException(UI error message on failure)
    def __init__(self, launch_params, test_type, cdm_test, test_index, location_index, launcher_index):
        pass
    #Throws LaunchException(UI error message on failure)
    def can_launch(self):
        pass
    #Throws LaunchException(UI error message on failure)
    def can_launch_status(self):
        pass
    #Throws LaunchException(UI error message on failure)
    def launch(self):
        pass
    #Throws LaunchException(UI error message on failure)
    def cancel(self):
        pass

class ExeTestLauncher(TestLauncherInterface):    

    #https://conf1.ds.jdsu.net/wiki/display/PSP/ONA+Job+Manager+Solution+Test+Launcher+Specification
    #launch_params["can_launch"] 
    #launch_params["launch"]
    #Throws Exception with error message in str(excinfo.value)    
    def __init__(self, launch_params, test_type, cdm_test, test_index, location_index, launcher_index=0):
        if launch_params and "can_launch" in launch_params and "launch" in launch_params:
            self.can_launch_exe = launch_params["can_launch"]
            self.launch_exe = launch_params["launch"]
            self.launcher_index = launcher_index
        else:
            print("ExeTestLauncher missing can_launch or launch parameter:", launch_params)
            raise LaunchException("ERROR_INVALID_LAUNCH_PARAMETERS")

        if shutil.which(self.can_launch_exe) is None:
            print("ExeTestLauncher no can_launch found:", self.can_launch_exe)
            raise LaunchException("ERROR_INVALID_LAUNCH_PARAMETERS")
        if shutil.which(self.launch_exe) is None:
            print("ExeTestLauncher no launch executable found:", self.launch_exe)
            raise LaunchException("ERROR_INVALID_LAUNCH_PARAMETERS")

        try:
            self.can_launch_process = None
            self.state = "idle"
            self.test_type = test_type
            self.test_index = str(test_index)
            self.location_index = str(location_index)
            self.cdm_test = cdm_test
            if type(self.cdm_test) != str:
                self.cdm_test = json.dumps(self.cdm_test)
        except:
            raise LaunchException("ERROR_INVALID_LAUNCH_PARAMETERS")

    #Throws Exception with error message if fails
    def can_launch(self):
        #Asynchronously execute can_launch
        self.state = "can_launch_in_progress"
        try:
            self.can_launch_process = subprocess.Popen([self.can_launch_exe, self.test_type, self.cdm_test, self.test_index, self.location_index], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        except:
            self.state = "complete"
            raise LaunchException("ERROR_CAN_LAUNCH_FAILED_TO_RUN")

    #Returns None if still in progress, true if done and successful
    #Throws Exception with error message if fails
    def can_launch_status(self):
        rv = None
        if self.state == "complete":
            return True
        if self.state != "can_launch_in_progress":
            raise LaunchException("ERROR_CAN_LAUNCH_FAILED_TO_RUN")
        poll = self.can_launch_process.poll() 
        if poll == None:
            #Still in progress
            rv = None
        elif self.can_launch_process.returncode == 0:
            self.state = "can_launch_complete"
            #print("can_launch_complete")
            rv = True
        else:
            (results, errors) = self.can_launch_process.communicate()
            self.state = "complete"
            outp = results.decode()
            #print("can launch oupt", outp)
            outp_lines = outp.split("\n")
            error_code = "ERROR_CAN_LAUNCH_FAILED_TO_RUN"
            custom_error_message = ""
            #Find the last line that isn't empty
            if outp_lines:
                for outp_line in reversed(outp_lines):
                    soutp_line = outp_line.strip()
                    if soutp_line:
                        soutp_line = soutp_line.split('^', 1) 
                        error_code = soutp_line[0]
                        if len(soutp_line) > 1:
                            custom_error_message = soutp_line[1]
                        break
            rv = False
            raise LaunchException(error_code, custom_error_message)
        return rv

    #Returns true if launch was successful or already complete
    #Returns false if can launch isn't complete 
    #Throws Exception with error message if fails
    def launch(self):
        rv = False
        if self.state == "complete":
            rv = True
        elif self.state == "can_launch_complete":
            self.state = "complete"
            try:
                #Use start_new_session which detaches process
                subprocess.Popen([self.launch_exe, self.test_type, self.cdm_test, self.test_index, self.location_index], start_new_session=True)
                rv = True
            except:
                raise LaunchException("ERROR_INVALID_LAUNCH_PARAMETERS")
        else:
            raise LaunchException("ERROR_CAN_LAUNCH_FAILED_TO_RUN")
        return rv

    def cancel(self):
        if self.state == "can_launch_in_progress":
            self.can_launch_process.kill()
        self.state = "complete"
        return True


#See ErrorMessageHelper.h for error handling
class TestLauncher:
    current_launcher = None
    next_launch_index = 0
    #Known launch type mapping
    launch_strategies = {"exe" : ExeTestLauncher}
    launcher_timeout = 15
    _instance = None

    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(TestLauncher, cls).__new__(cls)
            # Put any initialization here.
        return cls._instance    

    def is_valid_launch_index(self, launch_id):
        rv = False
        if self.current_launcher and launch_id == self.current_launcher.launcher_index:
            rv = True
        return rv

    #launch_params Input is a dictionary like this
    #{"launch_type" : "exe", "can_launch" : "/user/job-manager/launch/fiber-can-launch.sh",  "launch" : "/user/job-manager/launch/fiber-launch.sh"}
    #Returns the launcher index if successful
    #Throws Exception with error message in str(excinfo.value)
    def create_launcher(self, launch_params, test_type, cdm_test, test_index, location_index):
        rv = None
        #If our previous launch is complete, reset the current launcher
        if self.current_launcher and self.current_launcher.state == "complete":
            self.current_launcher = None

        if self.current_launcher != None and self.current_launcher.state != "complete":
            launch_elapsed_time = int(time.monotonic() - self.launch_start)
            if launch_elapsed_time > self.launcher_timeout:
                print("create_launcher last launcher timed out")
                #Last launcher has taken too long. Clean it up
                self.current_launcher.cancel()
            else:
                #Can't create a new launcher until the last one has finished or has been canceled
                raise LaunchException("ERROR_ANOTHER_LAUNCH_IN_PROGRESS")
        launch_type = None
        try:
            launch_type = launch_params["launch_type"]
        except:
            raise ServerLaunchException("ERROR_INVALID_LAUNCH_PARAMETERS")
        if launch_type in self.launch_strategies:
            self.current_launcher = self.launch_strategies[launch_type](launch_params, test_type, cdm_test, test_index, location_index, self.next_launch_index)
            rv = self.next_launch_index
            self.next_launch_index += 1
            self.launch_start = time.monotonic()
        return rv 

    #Throws Exception with error message if fails
    def can_launch(self, index): 
        #make sure the request maps to our current launcher
        if self.current_launcher != None and self.current_launcher.launcher_index == index:
            rv = self.current_launcher.can_launch()
        else:
            raise LaunchException("ERROR_ANOTHER_LAUNCH_IN_PROGRESS")

    #Returns None if still in progress, True if done and successful
    #Throws Exception with error message if fails
    def can_launch_status(self, index):
        #make sure the request maps to our current launcher
        if self.current_launcher != None and self.current_launcher.launcher_index == index:
            if self.current_launcher.state == "complete":
                return True
            launch_elapsed_time = int(time.monotonic() - self.launch_start)
            if launch_elapsed_time > self.launcher_timeout:
                #Last launcher has taken too long. Clean it up
                self.current_launcher.cancel()
                raise LaunchException("ERROR_CAN_LAUNCH_FAILED_TO_RUN")
            else:
                return self.current_launcher.can_launch_status()
        else:
            raise LaunchException("ERROR_ANOTHER_LAUNCH_IN_PROGRESS")

    #Returns true if launch was successful or already complete
    #Returns false if can launch isn't complete 
    #Throws Exception with error message if fails
    def launch(self, index):
        if self.current_launcher != None and self.current_launcher.launcher_index == index:
            return self.current_launcher.launch()
        else:
            raise LaunchException("ERROR_ANOTHER_LAUNCH_IN_PROGRESS")

    def cancel(self, index):
        rv = False
        if self.current_launcher != None and self.current_launcher.launcher_index == index:
            rv = self.current_launcher.cancel()
        return rv
