 
from marshmallow import Schema, fields

class DateTimeSchema(Schema):
    """
    Schema for date and time configuration for the instrument

    """
    time_value = fields.DateTime(required=True,
                                 load_from='timeValue',
                                 dump_to='timeValue',
                                 format="iso8601",
                                 description='ISO 8601 UTC date/time string, \
                                              e.g. 2016-07-21T15:56:00-07:00')
    location = fields.Str(required=True,
                          load_from='location',
                          dump_to='location',
                          description='location of current timezone')


class LocationSchema(Schema):
    """
    Schema for location configuration for the instrument
    """
    latitude = fields.Decimal(
        required=True,
        load_from='latitude',
        dump_to='latitude',
        description='Latitude'
    )

    longitude = fields.Decimal(
        required=True,
        load_from='longitude',
        dump_to='longitude',
        description='Longitude'
    )

    timestamp = fields.Integer(
        required=True,
        load_from='timestamp',
        dump_to='timestamp',
        description='Timestamp of when the location information was sent'
    )

    accuracy = fields.Decimal(
        required=True,
        load_from='accuracy',
        dump_to='accuracy',
        description='Location accuracy metric'
    )

    altitude = fields.Decimal(
        required=True,
        load_from='altitude',
        dump_to='altitude',
        description='Altitude'
    )

    altitude_accuracy = fields.Decimal(
        required=False,
        allow_none=True,
        load_from='altitudeAccuracy',
        dump_to='altitudeAccuracy',
        description='Altitude accuracy metric',
        missing=0
    )

class InstrumentLoginCredentialsSchema(Schema):
    """Schema for the StrataSync information used to sync and instrument
    """
    accountId = fields.Integer(
        load_from="accountId",
        dump_to="accountId",
        missing="",
    )
    techId = fields.Str(
        load_from="techId",
        dump_to="techId",
        required=True,
    )

class StrataSyncSchema(Schema):
    """Schema for the StrataSync information used to sync and instrument
    """
    tech_first_name = fields.Str(
        load_from="techFirstName",
        dump_to="techFirstName",
        required=True,
    )
    tech_last_name = fields.Str(
        load_from="techLastName",
        dump_to="techLastName",
        required=True,
    )
    tech_id = fields.Str(
        load_from="techId",
        dump_to="techId",
        required=True,
    )
    url = fields.Str(
        load_from="url",
        dump_to="url",
        missing="",
    )
    account_id = fields.Str(
        load_from="accountId",
        dump_to="accountId",
        missing="",
    )

class OptionsSchema(Schema):
    options = fields.Nested('OptionSchema', many=True)
    """Schema for Options information deployed to the instrument
    """

class OptionSchema(Schema):
    name = fields.Str(
        load_from="name",
        dump_to="name",
        required=True
    )
    serialNo = fields.Str(
        load_from="serialNo",
        dump_to="serialNo",
        required=True
    )
    assetType = fields.Str(
        load_from="assetType",
        dump_to="assetType",
        required=True
    )
    model = fields.Str(
        load_from="model",
        dump_to="model",
        required=True
    )
    uniqueId = fields.Str(
        load_from="uniqueId",
        dump_to="uniqueId",
        required=True
    )
    fwDeployTime = fields.Integer(
        load_from="fwDeployTime",
        dump_to="fwDeployTime",
        required=True
    )
    key = fields.Str(
        load_from="key",
        dump_to="key",
        required=True
    )
