"""Module containing base REST API configurations for Fiber products
"""
import logging
import threading
import traceback
import json

from bottle import Bottle

from rest_api.products import product_specific_base

from rest_api.products.config_schema_common import SubTypeRefInfoFormatter
from rest_api.products.usc import file_sync
from rest_api.products.usc import cdm_sync
from rest_api.products.usc import set_dbus_bus

logging.basicConfig(level=logging.DEBUG)

log = logging.getLogger(__name__)

JOB_FILE_EXT = '.job.json'
TEMPLATE_FILE_EXT = '.tpl.json'

class ProductSpecific(product_specific_base.ProductSpecific, SubTypeRefInfoFormatter):
    """Class for all Fiber product specific functions
    """
    JOB_MANAGER_DIR = '/user/job-manager/'
    JOB_FILE_EXT = '.job.json'
    TEMPLATE_FILE_EXT = '.tpl.json'

    def __init__(self):
        super().__init__()
        self.test_definitions = {}

    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        log.debug('## startup_side_effect (Test_Product): Jobs in %s', jobs_dir)
        ProductSpecific.JOB_MANAGER_DIR = jobs_dir
        set_dbus_bus("session")
        cdm_sync.register_cdm_filepath(jobs_dir)

    @staticmethod
    def make_instrument_info():
        info_version = {'infoVersion':1}

        __version__ = '1.11.0' # Should we align this with latest Yocto project tag?
        __files_api_version__ = 1
        __job_manager_api_version__ = 2
        __location_api_version__ = 1
        __datetime_api_version__ = 1
        __workflow_api_version__ = [1,2]

        api_support = {
            'apiSupport': [
                {
                    'name': 'datetime',
                    'versions': [__location_api_version__]
                },
                {
                    'name': 'location',
                    'versions': [__datetime_api_version__]
                },
                {
                    'name': 'techinfo',
                    'versions': [1]
                },
                {
                    'name': 'filemgr',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'syncfiles',
                    'versions': [1]
                },
                {
                    'name': 'signature',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'testdata',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'jobmanager',
                    'versions': [__job_manager_api_version__]
                },
                {
                   'name': 'workflow', 
                   'versions': __workflow_api_version__
                }
            ]
        }


        asset_info = {
            'assetInfo': {
                'serialNo': 'Test-123',
                'assetType': 'Test Product',
                'uniqueId': 'Test Product 123',
                'model': 'Test_Product',
                'hwVersion': '1.0',
                'swVersion': '1.0',
                'swOptions': [],
                'hwOptions': [],
                'customFields': {},
            }
        }


        instrument_info = super(ProductSpecific, ProductSpecific).make_instrument_info()

        instrument_info.update(info_version)
        instrument_info.update(api_support)
        instrument_info.update(asset_info)

        log.debug('## make_instrument_info: version                 = %s', __version__)
        log.debug('## make_instrument_info: files_api_version       = %s', __files_api_version__)
        log.debug('## make_instrument_info: location_api_version    = %s', __location_api_version__)
        log.debug('## make_instrument_info: datetime_api_version    = %s', __datetime_api_version__)
        log.debug('## make_instrument_info: api_support: %s', api_support)
        return instrument_info


    @staticmethod
    def get_job_extension():
        return ProductSpecific.JOB_FILE_EXT

    @staticmethod
    def get_job_dir():
        return ProductSpecific.JOB_MANAGER_DIR

    @staticmethod
    def get_template_extension():
        return ProductSpecific.TEMPLATE_FILE_EXT

    @staticmethod
    def get_instrument_login_config():
        loginConfig = {}
        loginConfig['techId'] = 'kh001'
        loginConfig['accountId'] = '99777777'
        return loginConfig
