"""Module that deals with decoupling the rest_api web application from
from functions that need to be injected into it to deal with product specific
actions
"""
import importlib
import sys
import logging
import glob


log = logging.getLogger(__name__)

#Products handled within this project. For new instruments, we should add them here but rather use the solution installed product python file as defined here
#https://conf1.ds.jdsu.net/wiki/display/PSP/ONA+Job+Manager+Solution+Test+Definition+Specification
PRODUCT_STR_TO_MODULE_STR = {
    'T-BERD 5800 V2': 'products.mts_5800',
    'MTS 5800 V2': 'products.mts_5800',
    'MTS 5882': 'products.mts_5800',
    'T-BERD 5882': 'products.mts_5800',
    'T-BERD 5800-100G': 'products.mts_5800',
    'MTS 5800-100G': 'products.mts_5800',
    'SC 4800': 'products.metro_base',
    'SC 4800P': 'products.metro_base',
    'MAP-2100': 'products.metro_base',
    'fiberMts': 'products.fiber_base', # RestAPI baseline for MTS4kv2, MTS2kv2 and SmartOTDR
    'fiberJobMgr': 'products.fiber_mts_jobmgr', # fiber_base with JobMgr support on top
    'ONA-800': 'products.ona_800',
    'OneExpert CATV': 'products.fit_base',
    'ONA-1000': 'products.ona_1000',
    'CA5000': 'products.ona_ca5g',
    'PSP Reference': 'products.pointsolution_base',
    'Test_Product':'products.test_product',
    'off-target': 'products.product_specific_base'
}

#Dynamically import solution installed product defintions. This provides a way for solutions to define a product file without
#checking in to python3-mts-restapi.
def import_products(product_definitions_dir):
    if product_definitions_dir == None:
        product_definitions_dir = '/usr/share/mts-restapi/products/'
    #Add the import path to our python
    sys.path.append(product_definitions_dir)
    log.debug("import_products product_definitions_dir: {}".format(product_definitions_dir))
    imported_product_definitions = {}
    #Recursively find python files in our directory
    for py in glob.glob("{}/**/*.py".format(product_definitions_dir), recursive = True):
        #String the base path and py extension
        #For subdirs replace "/" with "."        
        try:
            py = py.replace(product_definitions_dir,"").strip(".py").replace("/",".")
            product_module = importlib.import_module(py)
            log.debug("import_test_definitions imported: {}".format(py))
            #PRODUCT_NAME should be defined in any product file
            imported_product_definitions[product_module.PRODUCT_NAME] = product_module
        except:
            log.error("Failed to import test definition module: {}".format(py))
    return imported_product_definitions

def product_specific(product_str, product_definitions_dir=None):
    """Function to determine application configuration values based on the
    name of the product

    Args:
        product_module_str (str): the name of the python module containing
            configuration for the product.  This module must have a `config`
            callable that will return a dictionary of configs that need to
            be changed from defaults
    """  
    product = None  
    product_module_str = PRODUCT_STR_TO_MODULE_STR.get(product_str)    
    if product_module_str:
        #If our product_str matches a product that is defined directly in python3-mts-restapi
        product = importlib.import_module('rest_api.{}'.format(product_module_str))    
    else:
        #See if the product_str matches a solution installed product definition
        imported_product_definitions = import_products(product_definitions_dir)
        product = imported_product_definitions.get(product_str)

    if not product:
        log.error("Unknown product type {}".format(product_str))
        raise FileNotFoundError
    return product.ProductSpecific()
