"""Module containing a function to update the StrataSync info on a converged platform
product
"""
import logging
import datetime
import hashlib
import hmac
from rest_api.products.pointsolution_services import get_pydbus_bus
import json
import traceback

log = logging.getLogger(__name__)

scDBusObjectName = 'com.viavisolutions.platform.SettingsConfiguration'
scDBusObjectPath = '/com/viavisolutions/platform/SettingsConfiguration'

def update_stratasync_info(ssInfo):
    serverIp = ''
    if 'serverIp' in ssInfo:
        serverIp = ssInfo.pop('serverIp')

    try:
        bus = get_pydbus_bus()
        proxy = bus.get(scDBusObjectName, scDBusObjectPath)
        if not proxy.SetSettingsByGroup('UscAccount', json.dumps(ssInfo)):
            raise ValueError('Failed to set UscAccount settings in S&C Service')
        if serverIp and not proxy.SetSetting('UscServer', 'serverIp', serverIp):
            raise ValueError('Failed to set serverIp in S&C Service')
    except ValueError as e:
        log.debug(str(e))
        return False
    except:
        log.debug('Failed to set techinfo in S&C Service')
        return False
    return True

def get_instrument_login_config():
    try:
        bus = get_pydbus_bus()
        proxy = bus.get(scDBusObjectName, scDBusObjectPath)
        techId = json.loads(proxy.GetSettingByName('UscAccount', 'techId'))['techId']
        accountId = json.loads(proxy.GetSettingByName('UscAccount', 'tenantCode'))['tenantCode']
    except:
        log.info('Failed to get instrument login config from S&C Service')
        log.debug(traceback.format_exc())
        return {}

    # login config not found
    if not techId and not accountId:
        log.info('Instrument login config not found')
        return {}

    # omit if not configured (at least one will be configured)
    loginConfig = {}
    if techId:
        loginConfig['techId'] = techId
    if accountId:
        loginConfig['accountId'] = accountId

    return loginConfig

def set_instrument_login_config(techId, accountId):
    try:
        bus = get_pydbus_bus()
        proxy = bus.get(scDBusObjectName, scDBusObjectPath)
        if not proxy.SetSetting('UscAccount', json.dumps({'techId': techId})):
            raise ValueError('Failed to set techId in the S&C Service')
        if not proxy.SetSetting('UscAccount', json.dumps({'tenantCode': accountId})):
            raise ValueError('Failed to set tenantCode in the S&C Service')
    except ValueError as e:
        log.debug(str(e))
        return False
    except:
        log.debug('Failed to set instrument login config in the S&C Service')
        return False
    return True

def get_instrument_login_credentials(loginConfig):
    techId    = loginConfig['techId']
    accountId = loginConfig['accountId']
    if not set_instrument_login_config(techId, str(accountId)):
        return {}

    uniqueId  = loginConfig['uniqueId']
    assetType = loginConfig['assetType']
    clientId  = loginConfig['client_id']
    secretId  = loginConfig['client_secret']
    x_ss_date = datetime.datetime.utcnow().replace(microsecond=0).isoformat() + 'Z'

    password = create_token(str(accountId), x_ss_date, uniqueId, assetType, clientId, secretId)

    loginCredentials = {
        'x-ss-date': x_ss_date,
        'accountId': accountId,
        'techId': techId,
        'asset_client_id': clientId,
        'username': uniqueId,
        'password': password
    }

    return loginCredentials

def create_token(accountId, x_ss_date, uniqueId, assetType, clientId, secretId):
    signKey  = clientId + secretId
    message  = signKey + accountId + assetType + uniqueId + x_ss_date
    password = hmac.new(signKey.encode('utf-8'), message.encode('utf-8'), hashlib.sha512).hexdigest()
    return password
