"""
Module handles firmware upgrade through Mobile Tech REST API
"""

import logging
import threading, subprocess
import os, sys, shutil
from rest_api.products.pointsolution_services import get_dbus_bus
import json
import re, io
from enum import IntEnum

from bottle import request, response

log = logging.getLogger(__name__)
upgradeFifoPath   = '/tmp/upgrade-mobiletech.log'
upgradeLocalPath  = '/tmp/upgrade'
upgradeState = 'noUpgrade'
enableSwapCmd = 'psp-enable-swap'
upgradeProgress = 0
pipeReaderThread = None

def upgrade_cleanup():
    shutil.rmtree(upgradeLocalPath, ignore_errors=True)
    try:
        os.remove(upgradeFifoPath)
    except OSError:
        pass

def parse_upgrade_log():
    global upgradeState
    global upgradeProgress

    progressPattern = '^PROGRESS.*OverallProgress=(\d+)\S*$'
    bootstrapMsg = "Bootstrap completed with no errors"
    bootstrapErrorMsg = "Bootstrap failed due to one or more errors"
    completeMsg = "Upgrade completed with no errors"
    errorMsg = "Upgrade failed due to one or more errors"
    noPowerMsg = "Not enough power to start an upgrade"
    rebootMsg = "Upgrade successful, issuing reboot"

    # parse upgrade logs
    bootstrapComplete = False
    fifo = open(upgradeFifoPath, 'r')
    os.set_blocking(fifo.fileno(), False)
    line = fifo.readline()
    while True:
        match = re.search(progressPattern, line)
        if match:
            upgradeProgress = int(int(match.group(1)) / 2)
            if bootstrapComplete:
                # bootstrap is finished so upgrade progress should be 50% now
                upgradeProgress += 50
        elif bootstrapMsg in line:
            bootstrapComplete = True
        elif completeMsg in line:
            upgradeState = 'applyComplete'
            upgradeProgress = 100
            break
        elif noPowerMsg in line:
            upgradeState = 'noPower'
            break
        elif rebootMsg in line:
            upgradeState = 'rebooting'
            upgradeProgress = 100
            break
        elif bootstrapErrorMsg in line or errorMsg in line:
            upgradeState = 'applyFailed'
            break
        line = fifo.readline()

    fifo.close()

    if not bootstrapComplete or upgradeState == 'applying':
        upgradeState = 'applyFailed'

    upgrade_cleanup()


def reset_firmware_upgrade_status():
    global upgradeState
    global upgradeProgress

    upgradeState = 'noUpgrade'
    upgradeProgress = 0

def has_power():
    dbusObjectName = 'com.viavisolutions.platform.Battery'
    dbusIfaceName  = 'com.viavisolutions.platform.Battery'
    dbusObjectPath = '/com/viavisolutions/platform/Battery'
    minChargeLevelPct = None # TODO set those to something in the future
    chargingRequired = False
    battJson = None

    class BattChargeState(IntEnum):
        BATTCHARGE_CHARGING    = 0
        BATTCHARGE_DISCHARGING = 1
        BATTCHARGE_UNKNOWN     = 2

    try:
        bus = get_dbus_bus()
        battProxy = bus.get_object(dbusObjectName, dbusObjectPath)
        battInfo = battProxy.GetBatteryInfo(dbus_interface=dbusIfaceName)
        battJson = json.loads(battInfo)
    except:
        # if getting battery info fails then assume we have power
        # TODO add some logging
        return True

    if minChargeLevelPct and battJson[0]['charge_level_pct'] < minChargeLevelPct:
        return False
    if chargingRequired and battJson[0]['charging_state'] != BattChargeState.BATTCHARGE_CHARGING:
        return False

    return True

def process_firmware_upgrade_status():
    firmwareStatus = {
        'features': [
            "mobile-download"
        ],
        'hasPower': True, # has_power()
        'upgradeStatus': upgradeState,
        'progress': upgradeProgress,
    }

    return firmwareStatus

def process_firmware_upgrade_put(request):
    # enable swap here for now
    try:
        proc = subprocess.run(enableSwapCmd)
        if proc.returncode != 0:
            log.error("Firmware upgrade PUT request - error while setting swap")
    except:
        # if enable-swap fails then assume we don't have this entry point
        log.error("Firmware upgrade PUT request - could not enable swap")
    return True

def process_firmware_upgrade_post(request, filename, upgradeCmd):
    global pipeReaderThread
    global upgradeState
    global upgradeProgress

    class Myfileobj(io.FileIO):
        def __init__(self, length):
            self.headers = True
            self.length = length

        def read(self, size):
            seq = [b'\r', b'\n', b'\r', b'\n']
            seqn = 0

            # skip header, get to content
            while self.headers:
                a = request.environ['wsgi.input'].read(1)
                self.length -= 1
                if a == seq[seqn]:
                    seqn += 1
                else:
                    seqn = 0
                if seqn == len(seq):
                    self.headers = False

            s = min(size, self.length)
            d = request.environ['wsgi.input'].read(s)
            self.length -= s

            return d

    # get bundle from POST request
    bundlePath = os.path.join(upgradeLocalPath, filename)
    os.mkdir(upgradeLocalPath)
    # saving using bottle's FileUpload creates multiple copies in memory
    #upload = request.files.get('')
    #upload.save(bundlePath)
    with open(bundlePath, 'wb') as f:
        f.write(Myfileobj(request.content_length).read(request.content_length))

    # create named pipes used for upgrade logs
    os.mkfifo(upgradeFifoPath)

    # start the upgrade
    upgradeProgress = 0
    upgradeState = 'applying'
    proc = subprocess.Popen([upgradeCmd, bundlePath])

    # start log parsing thread
    if not pipeReaderThread or not pipeReaderThread.is_alive():
        pipeReaderThread = threading.Thread(target = parse_upgrade_log)
        pipeReaderThread.start()

    return True
