import json
import pydbus
from rest_api.products.pointsolution_services import get_pydbus_bus
import logging
from enum import IntEnum

log = logging.getLogger(__name__)

batteryDBusObjectName = 'com.viavisolutions.platform.Battery'
batteryDBusIfaceName  = 'com.viavisolutions.platform.Battery'
batteryDBusObjectPath = '/com/viavisolutions/platform/Battery'

class BattFaultType(IntEnum):
    BATTFAULT_NONE          = 0x00
    BATTFAULT_LOW_BATT      = 0x01
    BATTFAULT_UNDER_TEMP    = 0x02
    BATTFAULT_OVER_TEMP     = 0x04
    BATTFAULT_OVER_VOLTAGE  = 0x08
    BATTFAULT_OVER_AMPERAGE = 0x10
    BATTFAULT_BATT_FAULT    = 0x20
    BATTFAULT_UNKNOWN       = 0x40

class BattChargeState(IntEnum):
    BATTCHARGE_CHARGING    = 0
    BATTCHARGE_DISCHARGING = 1
    BATTCHARGE_UNKNOWN     = 2

def get_battery_error_msg(faults):
    if faults & BattFaultType.BATTFAULT_BATT_FAULT:
        return "Battery fault"
    return "Unknown error"

chargeStateToString = {
    BattChargeState.BATTCHARGE_CHARGING:    'charging',
    BattChargeState.BATTCHARGE_DISCHARGING: 'discharging',
    BattChargeState.BATTCHARGE_UNKNOWN:     'error'
}

def get_battery_info():
    try:
        bus       = get_pydbus_bus()
        battProxy = bus.get(batteryDBusObjectName, batteryDBusObjectPath)
        battInfo  = battProxy.GetBatteryInfo()
        battJson  = json.loads(battInfo)
    except:
        log.debug('## get_battery_info(): Failed to get battery info')
        return None
    return battJson

def get_battery_level():
    battInfo = get_battery_info()
    if not battInfo:
        return None
    return battInfo[0]['charge_level_pct']

def get_charging_state():
    battInfo = get_battery_info()
    if not battInfo:
        return None
    return battInfo[0]['charging_state']

def process_batteryinfo_request():
    battInfo = get_battery_info()
    if not battInfo:
        return (False, {})

    battery_level  = battInfo[0]['charge_level_pct']
    battery_temp   = '{} C'.format(battInfo[0]['temperature_C'])
    battery_status = chargeStateToString[battInfo[0]['charging_state']]

    batteryinfo_response = {
        'level':  battery_level,
        'temp':   battery_temp,
        'status': battery_status
    }

    # add error message if needed
    if battInfo[0]['charging_state'] == BattChargeState.BATTCHARGE_UNKNOWN:
        batteryinfo_response['error'] = get_battery_error_msg(battInfo[0]['fault_conditions'])

    return (True, batteryinfo_response)

def has_power():
    minChargeLevelPct = None # TODO set those to something in the future
    chargingRequired = False
    return True
