"""Module containing base configurations for pointsolution platform (point solution) products
"""
import logging
import os

from rest_api.products import product_specific_base

from rest_api.products.pointsolution_services import access_anywhere
from rest_api.products.pointsolution_services import vnc
from rest_api.products.pointsolution_services import instrument_time
from rest_api.products.pointsolution_services import instrument_location
from rest_api.products.pointsolution_services import update_stratasync_info
from rest_api.products.pointsolution_services import instrument_batteryinfo
from rest_api.products.pointsolution_services import instrument_upgrade
from rest_api.products.usc import instrument_sync_files

from rest_api.products.usc import asset_info
from rest_api.products.usc import file_sync
from rest_api.products.usc import cdm_sync

from rest_api.products import process_cdm_job_pointsolution

log = logging.getLogger(__name__)

class ProductSpecific(product_specific_base.ProductSpecific):
    upgrade_cmd = 'psp-upgrade'
    JOB_MANAGER_DIR = '/user/job-manager/'
    JOB_FILE_EXT = '.job.json'
    TEMPLATE_FILE_EXT = '.tpl.json'
    instrument_info = None
    instrument_login_client_id = '351f3365-b425-4e4a-bd0c-712c8be6ed87'
    instrument_login_client_secret = 'f6085c4a-f4a2-4bd2-b3e5-78cdf5c56bfc'

    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        file_sync.register_report_path(report_dir)
        file_sync.register_procedures_path(procedures_dir)
        cdm_sync.register_cdm_filepath(jobs_dir)
        ProductSpecific.JOB_MANAGER_DIR = jobs_dir

    @staticmethod
    def get_access_anywhere_code(servername):
        return access_anywhere.create_code(servername)

    @staticmethod
    def get_vnc_credentials(servername):
        return vnc.start_vnc(servername)

    @staticmethod
    def make_instrument_info():

        info_version = {'infoVersion':1}

        __version__ = '1.3.0'
        __files_api_version__ = 2
        __job_manager_api_version__ = 2
        __workflow_api_version__ = [1,2]
        __location_api_version__ = 1
        __datetime_api_version__ = 1
        __firmware_upgrade_api_version__ = 1
        __batteryinfo_api_version__ = 1
        __techinfo_api_version__ = 1
        __instrumentlogin_api_version__ = 1
        __vnc_api_version__ = 1
        __options_api_version__ = 1

        api_support = {
            'apiSupport': [
                {
                    'name': 'datetime',
                    'versions': [__location_api_version__]
                },
                {
                    'name': 'location',
                    'versions': [__datetime_api_version__]
                },
                {
                    'name': 'techinfo',
                    'versions': [__techinfo_api_version__]
                },
                {
                    'name': 'filemgr',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'syncfiles',
                    'versions': [1]
                },
                {
                    'name': 'testdata',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'jobmanager',
                    'versions': [__job_manager_api_version__]
                },
                {
                   'name': 'workflow', 
                   'versions': __workflow_api_version__
                },
                {
                    'name': 'vnc',
                    'versions': [__vnc_api_version__]
                },
                {
                    'name': 'firmware',
                    'versions': [__firmware_upgrade_api_version__]
                },
                {
                    'name': 'batteryinfo',
                    'versions': [__batteryinfo_api_version__]
                },
                {
                    'name': 'instrumentlogin',
                    'versions': [__instrumentlogin_api_version__]
                },
                {
                    'name': 'options',
                    'versions': [__options_api_version__]
                }
            ]
        }
        supported_file_extensions = {
          'supportedJobFileExtensions': [ 'png', 'jpg', 'json' ]
        }

        ProductSpecific.instrument_info = super(ProductSpecific, ProductSpecific).make_instrument_info()

        ProductSpecific.instrument_info['assetInfo'] = asset_info.make_asset_info()

        ProductSpecific.instrument_info.update(info_version)
        ProductSpecific.instrument_info.update(api_support)
        ProductSpecific.instrument_info.update(supported_file_extensions)

        return ProductSpecific.instrument_info

    @staticmethod
    def update_time(time_value, location):
        return instrument_time.update_local_time(time_value, location)

    @staticmethod
    def update_location(geolocation):
        return instrument_location.update_location(geolocation)

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        ss_info = {
            'techFirstName': tech_first_name,
            'techLastName': tech_last_name,
            'techId': tech_id
        }
        if account_id:
            ss_info['accountId'] = account_id
        if url:
            ss_info['url'] = url
        return update_stratasync_info.update_stratasync_info(ss_info)

    @staticmethod
    def get_sync_list(sync_all):
        return instrument_sync_files.get_sync_list(sync_all)

    @staticmethod
    def get_sync_file_type(full_path):
        return instrument_sync_files.get_sync_file_type(full_path)

    @staticmethod
    def set_sync_list_ack(full_path):
        instrument_sync_files.set_sync_list_ack(full_path)

    @staticmethod
    def call_process_cdm_job():
        process_cdm_job_pointsolution.main()

    @staticmethod
    def get_sync_file_info(filepath, base_dir):
        return instrument_sync_files.get_sync_file_info(filepath, base_dir)

    @staticmethod
    def update_test_data_side_effect(test_plan_index,job_status_ratio):
        log.info("update_test_data_side_effect")

    @staticmethod
    def update_loaded_job_side_effect(is_job_active,job_status_ratio):
        log.info("update_loaded_job_side_effect")

    @staticmethod
    def process_batteryinfo_request():
        return instrument_batteryinfo.process_batteryinfo_request()

    @staticmethod
    def update_options(options):
        index = 0
        for option in options:
            try:
                f = open ("/var/staged_options/option" + str(index) + ".json", "w")
                f.write(option["key"])
                f.close()
                index = index + 1
            except:
                return False
        return True

    @staticmethod
    def get_dirname(path, base_dir):
        if(base_dir == os.path.normpath(path)):
            return "/"
        else:
            return os.path.basename(os.path.normpath(path))

    @staticmethod
    def get_instrument_login_config():
        return update_stratasync_info.get_instrument_login_config()

    @staticmethod
    def get_instrument_login_credentials(loginConfig):
        asset_info_usc = asset_info.make_asset_info()

        fullLoginConfig = {
            'assetType': asset_info_usc.get('assetType', ''),
            'uniqueId':  asset_info_usc.get('uniqueId', ''),
            'techId':    loginConfig.get('techId', ''),
            'accountId': loginConfig.get('accountId', ''),
            'client_id': ProductSpecific.instrument_login_client_id,
            'client_secret': ProductSpecific.instrument_login_client_secret
        }

        return update_stratasync_info.get_instrument_login_credentials(fullLoginConfig)

    @staticmethod
    def reset_firmware_upgrade_status():
        instrument_upgrade.reset_firmware_upgrade_status()

    @staticmethod
    def process_firmware_upgrade_status():
        return instrument_upgrade.process_firmware_upgrade_status()

    @staticmethod
    def process_firmware_upgrade_put(request):
        return instrument_upgrade.process_firmware_upgrade_put(request)

    @classmethod
    def process_firmware_upgrade_post(cls, request, filename):
        return instrument_upgrade.process_firmware_upgrade_post(request, filename, cls.upgrade_cmd)

    @staticmethod
    def get_job_extension():
        return ProductSpecific.JOB_FILE_EXT

    @staticmethod
    def get_job_dir():
        return ProductSpecific.JOB_MANAGER_DIR

    @staticmethod
    def get_template_extension():
        return ProductSpecific.TEMPLATE_FILE_EXT
