"""Module containing a function to update the StrataSync info on a Fiber series (4K,2K,1K)
product
"""

import hashlib
import hmac
import datetime

from rest_api.products.mts_services.SCPI_Socket import SCPI_Socket

def update_stratasync_info(tech_id, account_id, url):
    """Function to update stratasync information on the Fiber series products (4K,2K,1K)

    Url arguments can optionally begin with `https://` only the hostname will be
    passed to the ctrlpannelsvc

    Args:
        tech_id (str): the stratasync techId for the technician
        account_id (str): the stratasync account_id for the technician's company
        url (str): the stratasync server url that the technician's company is using
    """
    ISU = SCPI_Socket('127.0.0.1',8000)
    ISU.debug_level(1)

    ISU.send("*rem")
    
    # TECHID
    #ISU.send('STSY:TECH "' + str(tech_id) + '"')
    
    #ACCOUNTID
    ISU.send('STSY:ACNT "' + str(account_id) + '"')
    
    #URL
    if "https://eu.stratasync.viavisolutions.com" in url:
        ISU.send('STSY:SRVT EUV')
    elif "https://stratasync.viavisolutions.com" in url:
        ISU.send('STSY:SRVT JDSU')
    else:
        ISU.send('STSY:SRVT NAME')
        urlEdit = url.replace("https://", "direct.")
        ISU.send('STSY:SRVN "' + str(urlEdit) + '"')
    
    ISU.close()
    
    # Some Sync problem with the TECHID unification
    FiberOptic = SCPI_Socket('127.0.0.1',8002)
    FiberOptic.debug_level(1)
    
    FiberOptic.send("*rem")
    
    # TECHID
    FiberOptic.send('FSETUP:OPERATOR "' + str(tech_id) + '"')
    
    FiberOptic.close()

    return True

def get_instrument_login_config():
    
    ISU = SCPI_Socket('127.0.0.1',8000)
    ISU.debug_level(1)

    ISU.send("*rem")
    
    # TECHID
    techId = ISU.query('STSY:TECH?')
    
    #ACCOUNTID
    accountId = ISU.query('STSY:ACNT?')
    
    instrument_login_config = {
            "accountId": accountId,
            "techId": techId
            }
    
    ISU.close()
    
    return instrument_login_config

def get_instrument_login_credentials(loginConfig):
    accountId = loginConfig['accountId']
    techId = loginConfig['techId']
    uniqueId = loginConfig['uniqueId']
    assetType = loginConfig['assetType']
    
    #Send last value enter by MTA
    send_login_config(accountId, techId)
    
    clientId = get_asset_client_id(assetType)
    secretId = get_asset_client_secret(assetType)
    
    current_date = datetime.datetime.utcnow().isoformat(timespec='seconds') + "Z"

    password = create_token(accountId, current_date, uniqueId, assetType, clientId, secretId)
    
    instrument_login_credentials = {
            "x-ss-date": current_date,
            "accountId": accountId,
            "techId": techId,
            "asset_client_id": clientId,
            "username": uniqueId,
            "password": password
            }
    
    return instrument_login_credentials

def create_token(account_id, current_date, unique_id, asset_type, client_id, secret_id):
    x_ss_date = current_date
    signKey = client_id + secret_id
    message = signKey + account_id + asset_type + unique_id + x_ss_date
    
    password = hmac.new( signKey.encode('utf-8'), message.encode('utf-8'), hashlib.sha512 ).hexdigest()
    
    return password

def get_asset_client_id(asset_type):
    map_client_id = {
        "SmartOTDR": "c54f4065-9beb-45b6-ac5e-d1c22abaab39",
        "Optimeter": "D0323214-5716-4688-9802-34A9525F066F",
        "T-BERD/MTS 4000 V2": "49a0e32a-42ab-4545-bb72-20f89ce7e8e9",
        "T-BERD/MTS 2000 V2": "2a6b1fdc-6e86-4cd0-a8ca-7bc3fb58adf3"
        }
    
    return map_client_id.get(asset_type, "")
    
def get_asset_client_secret(asset_type):
    map_client_secret = {
        "SmartOTDR": "9059334-5c78-4fef-a550-f226a76eb465A",
        "Optimeter": "73E221C-8830-4A38-BB10-0F20515E6806F",
        "T-BERD/MTS 4000 V2": "28fec260-38b7-4805-8924-e7be0858a63d",
        "T-BERD/MTS 2000 V2": "4e5f54ce-d117-4a18-85d0-1f1a8c4079fa"
        }
    
    return map_client_secret.get(asset_type, "")

def send_login_config(account_id, tech_id):
    ISU = SCPI_Socket('127.0.0.1',8000)
    ISU.debug_level(1)

    ISU.send("*rem")
    
    # TECHID
    #ISU.send('STSY:TECH "' + str(tech_id) + '"')
    
    #ACCOUNTID
    ISU.send('STSY:ACNT "' + str(account_id) + '"')
    
    ISU.close()
    
    FiberOptic = SCPI_Socket('127.0.0.1',8002)
    FiberOptic.debug_level(1)
    FiberOptic.send("*rem")
    
    # TECHID
    FiberOptic.send('FSETUP:OPERATOR "' + str(tech_id) + '"')
    
    FiberOptic.close()
    
