import os
import json
import datetime
import re
import logging

from rest_api.products.usc import dbus
from rest_api.products.usc import asset_info
from rest_api.products.usc import file_sync

log = logging.getLogger(__name__)

def get_sync_list(sync_all):

    if not dbus:
        return {}

    system_bus = dbus.SystemBus()

    proxy_object = file_sync.get_usc_proxy_object()

    file_list = []

    #modules_info.extend(get_modules_info(system_bus, service))
    file_list.extend(proxy_object.MtaGetFilesToSync(sync_all))


    #services = asset_info.get_services_from_usc(system_bus)

    return file_list

def set_sync_list_ack(full_path):
    
    if not dbus:
        return {}

    proxy_object = file_sync.get_usc_proxy_object()

    proxy_object.MtaSetSyncListAck(full_path)


def get_sync_file_type(full_path):

    if not dbus:
        return {}

    system_bus = dbus.SystemBus()

    proxy_object = file_sync.get_usc_proxy_object()

    file_type = proxy_object.MtaGetFileType(full_path)

    return file_type
    
def get_sync_file_metadata(full_path):
    
    if not dbus:
        return {}

    system_bus = dbus.SystemBus()

    proxy_object = file_sync.get_usc_proxy_object()
    
    file_metadata = []
    file_metadata.extend(proxy_object.MtaGetFileMetadata(full_path))
    
    return file_metadata

def format_metadata_result(result):
    if 'pass' in result:
        return 'Pass'
    elif 'fail' in result:
        return 'Fail'
    elif 'marginal' in result:
        return 'Marginal'
    elif 'none' in result:
        return 'None'
    else:
        return result
    

#The info is a comma separated string of key=value items. However values can have commas as well so we have to do some parsing to 
#get key value pairs. 
#Example string: 'customerName=Rajah Rhodes,loc=Wilson, NC,result=Fail,workflowId=920472,wo=RaSolutionUsingVMT_30Nov_V2'
def format_metadata_info_string(metadata_info):
    info_dict = {}
    try:
        #First split on '=' 
        sinfo = metadata_info.split('=')
        #The example string above parses to: ['customerName', 'Rajah Rhodes,loc', 'Wilson, NC,result', 'Fail,workflowId', '920472,wo', 'RaSolutionUsingVMT_30Nov_V2']
        #Go through each item in list and split on the last comma. The last comma is the list separator, previous commas are part of the value
        sinfo2 = []
        for i in sinfo:    
            sinfo2.extend(i.rsplit(',', 1))
        #sinfos is now a list of [key1, value1, key2, value2, ...]    
        # ['customerName', 'Rajah Rhodes', 'loc', 'Wilson, NC', 'result', 'Fail', 'workflowId', '920472', 'wo', 'RaSolutionUsingVMT_30Nov_V2']
        #Convert the list to a key:value dictionary
        info_dict = {sinfo2[i]: sinfo2[i + 1] for i in range(0, len(sinfo2), 2)}
        #info_dict = {'customerName': 'Rajah Rhodes', 'loc': 'Wilson, NC', 'result': 'Fail', 'workflowId': '920472', 'wo': 'RaSolutionUsingVMT_30Nov_V2'}
    except:
        pass
    return info_dict

def update_metadata(file_metadata, file_info):
    #Copy the current file_info
    file_info_updated = file_info
    
    sub_info = {}
    for metadata in file_metadata:
        #Split metadata String
        metadata_split = metadata.split(':', 1)        
        #Check all parameters send by USC
        if 'TechnicianId' in metadata_split[0]:
            file_info_updated['techId'] = metadata_split[1]
        elif 'Latitude' in metadata_split[0]:
            file_info_updated['latitude'] = metadata_split[1]
        elif 'Longitude' in metadata_split[0]:
            file_info_updated['longitude'] = metadata_split[1]
        elif 'Type' in metadata_split[0]:            
            if 'type' in file_info_updated:
                file_info_updated['type'] += metadata_split[1]
            else:
                file_info_updated['type'] = metadata_split[1]
        elif 'Info' in metadata_split[0]:
            all_info = format_metadata_info_string(metadata_split[1])
            for key, value in all_info.items():
                if key == 'result':
                    file_info_updated['result'] = format_metadata_result(value)
                elif key == 'workflowId':
                    try:
                        workflowId = int(value)
                        if workflowId > 0: # Exclude workflowId's that are zero
                            print("update_metadata: workflowId", workflowId)
                            file_info_updated['workflowId'] = workflowId 
                            log.debug('## update_metadata: workflowId = %s', file_info_updated['workflowId'] )
                    except:
                        pass
                elif key == 'testPlanIndex':
                    try:
                        testPlanIndex = int(value)
                        if testPlanIndex >= 0: 
                            print("update_metadata: testPlanIndex", testPlanIndex)
                            file_info_updated['testPlanIndex'] = testPlanIndex 
                            log.debug('## update_metadata: testPlanIndex = %s', file_info_updated['testPlanIndex'] )
                    except:
                        pass
                elif key == 'wo': # Note - must come *after* the workflowId test (which is also true when testing for 'wo' !)
                    file_info_updated['workOrderId'] = value
                    log.debug('## update_metadata: workOrderId = %s', file_info_updated['workOrderId'] )
                elif key == 'cdm':
                    file_info_updated['cdm'] = True  # if its here it's true! (expected as a boolean, but we provide all the metadata across Dbus in string format)
                else: 
                    sub_info[key] = value
                    
    file_info_updated['info'] = sub_info
    return file_info_updated

#StrataSync known file types for format field
known_extensions = \
    {'.json' : 'application/json', \
     '.pdf' : 'application/pdf',
     '.csv' : 'text/csv',
     '.html' : 'text/html',
     '.xml' : 'text/xml',
     '.txt' : 'text/plain',
     '.jpg' : 'image/jpg',
     '.jpeg' : 'image/jpg',
     '.png' : 'image/png',
     '.bmp' : 'image/bmp',
     '.xls' : 'application/vnd.ms-excel',
     '.doc' : 'application/ms-word',
     '.xpm' : 'image/xpm',
    }
def get_mime_type(filename):
    mime_type = ""
    try:
        extension = os.path.splitext(filename)[1]
        extension = extension.lower()
        mime_type = known_extensions[extension]
    except:
        pass
    return mime_type

def get_sync_file_info(filepath, base_dir):
    
    full_path = os.path.join(base_dir, filepath)
    fstats = os.stat(full_path)
    
    file_name = os.path.basename(full_path)
    
    file_type = get_sync_file_type(full_path)
    
    file_metadata = get_sync_file_metadata(full_path)

    if (file_type == ''):
        file_type = 'MISC.File'

    creation_time = fstats.st_ctime
    modified_time = fstats.st_mtime
    
    ct_iso = datetime.datetime.utcfromtimestamp(creation_time).isoformat(timespec='seconds') + "+00:00"
    md_iso = datetime.datetime.utcfromtimestamp(modified_time).isoformat(timespec='seconds') + "+00:00"

    file_info = {
            "name": file_name,
            "fileUri": "/" + filepath,
            "type": file_type,
            "format": '',
            "createdOn": ct_iso,
            "modifiedOn": md_iso,
            #"latitude": 0,
            #"longitude": 0,
            "fileSize": fstats.st_size,
            #"workflowId": 0,
            #"workOrderId": '',
            #"projectId": 0,
            #"result": '',
            #"location": '',
            #"moduleId": '',
            #"moduleType": '',
            #"techId": '',
            #"info": {},
            #"overwrite": true
            #"segments": 0,
        }
    
    file_info_updated = update_metadata(file_metadata, file_info)
    file_info_updated['format'] = get_mime_type(file_name)
    
    return file_info_updated


